<?php

namespace idoit\Module\Cabling\Export;

use DOMDocument;
use idoit\Module\Cabling\Model\Cabling;
use isys_application as App;

/**
 * GraphML export class.
 *
 * @package     Modules
 * @subpackage  Cabling
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class GraphML extends Main
{
    /**
     * @var  DOMDocument
     */
    private $dom;

    /**
     * @var \DOMElement
     */
    private $graph;

    /**
     * @var array
     */
    private $data = [];

    /**
     * @var array
     */
    private $exportedObjects = [];

    /**
     * @var array
     */
    private $edges = [];

    /**
     * @var array
     */
    private $cables = [];

    /**
     * @var array
     */
    private $connectorToObjectMatch = [];

    /**
     * @return $this
     */
    public function prepare()
    {
        set_time_limit(0);
        ob_end_clean();
        header('Cache-Control: no-store, no-cache, must-revalidate');
        header('Cache-Control: post-check=0, pre-check=0', false);
        header('Pragma: no-cache');
        header('Expires: 0');
        header('Last-Modified: ' . date('D, d M Y H:i:s') . ' GMT');
        header('Content-Type: application/xml; charset=utf-8');
        //header('Content-length: ' . (function_exists('mb_strlen') ? mb_strlen($l_export) : strlen($l_export)));
        header('Content-Disposition: attachement; filename="' . $this->filename . '.graphml"');
        header('Content-transfer-encoding: binary');


        $this->dom = new DOMDocument('1.0', 'UTF-8');

        return $this;
    }

    /**
     * @return $this
     */
    public function export()
    {
        // First we load the necessary data.
        $cablingDao = Cabling::instance(App::instance()->container->get('database'));
        $root       = $cablingDao->getSourceObject($this->objectId);
        $this->data = $cablingDao->resolve($this->objectId, true);

        // Now we build the GraphML file.
        $graphml = $this->dom->createElement('graphml');
        $graphml->setAttribute('xmlns', 'http://graphml.graphdrawing.org/xmlns');
        $graphml->setAttribute('xmlns:xsi', 'http://www.w3.org/2001/XMLSchema-instance');
        $graphml->setAttribute('xsi:schemaLocation', 'http://graphml.graphdrawing.org/xmlns http://www.yworks.com/xml/schema/graphml/1.1/ygraphml.xsd');
        $graphml->setAttribute('xmlns:y', 'http://www.yworks.com/xml/graphml');

        $key0 = $this->dom->createElement('key');
        $key0->setAttribute('id', 'd0');
        $key0->setAttribute('for', 'node');
        $key0->setAttribute('yfiles.type', 'nodegraphics');

        $key1 = $this->dom->createElement('key');
        $key1->setAttribute('id', 'd1');
        $key1->setAttribute('for', 'edge');
        $key1->setAttribute('yfiles.type', 'edgegraphics');

        $this->graph = $this->dom->createElement('graph');
        $this->graph->setAttribute('id', 'G');
        $this->graph->setAttribute('edgedefault', 'undirected');

        // Create the root node.
        $this->graph->appendChild(
            $this->renderObjectNode(
                [
                    'id'    => $root['objectId'],
                    'color' => $root['objectTypeColor'],
                    'label' => $root['objectTypeTitle'] . PHP_EOL . $root['objectTitle']
                ]
            )
        );

        $this->exportedObjects[$root['objectId']] = true;

        $this->iterateObjects('left');
        $this->iterateObjects('right');

        // Iterate over the collected object edges.
        foreach ($this->edges as $i => $edgeData)
        {
            // Only add the edge, if both connectors can be matched to an object.
            if (!isset($this->connectorToObjectMatch[$edgeData['source']]) || !isset($this->connectorToObjectMatch[$edgeData['target']]))
            {
                continue;
            }

            $edgeLabel = null;

            if ($this->displayCableNames)
            {
                $edgeLabel = $this->cables[$edgeData['source'] . '#' . $edgeData['target']] ?: $this->cables[$edgeData['target'] . '#' . $edgeData['source']] ?: null;
            }

            $this->graph->appendChild(
                $this->renderEdge(
                    [
                        'id'          => $i,
                        'source'      => $this->connectorToObjectMatch[$edgeData['source']]['objectId'],
                        'target'      => $this->connectorToObjectMatch[$edgeData['target']]['objectId'],
                        'label'       => $edgeLabel,
                        'labelSource' => $this->connectorToObjectMatch[$edgeData['source']]['title'],
                        'labelTarget' => $this->connectorToObjectMatch[$edgeData['target']]['title']
                    ]
                )
            );
        }

        $graphml->appendChild($key0);
        $graphml->appendChild($key1);
        $graphml->appendChild($this->graph);

        $this->dom->appendChild($graphml);

        return $this;
    }

    /**
     *
     */
    public function send()
    {
        echo $this->dom->saveXML();
    }

    /**
     * Method for iterating over all objects of one direction.
     *
     * @param string $direction
     */
    private function iterateObjects($direction)
    {
        // Iterate over every item we got.
        foreach ($this->data[$direction] as $connector)
        {
            if (isset($connector['cableTitle']) && !empty($connector['cableTitle']))
            {
                $this->cables[$connector['connectorId'] . '#' . $connector['id']] = $connector['cableTitle'];
            }

            // Check if we iterate over a object.
            if (isset($connector['objectId']))
            {
                $this->connectorToObjectMatch[$connector['id']] = [
                    'objectId' => $connector['objectId'],
                    'title'    => $connector['title']
                ];

                if ($connector['objectId'] != $this->objectId && isset($connector['parentId']) && !empty($connector['parentId']))
                {
                    $this->edges[] = [
                        'source' => $connector['parentId'],
                        'target' => $connector['id']
                    ];
                }

                // Skip the rest if the object has already been added to the XML.
                if (isset($this->exportedObjects[$connector['objectId']]))
                {
                    continue;
                }

                $this->graph->appendChild(
                    $this->renderObjectNode(
                        [
                            'id'    => $connector['objectId'],
                            'color' => $connector['objectTypeColor'],
                            'label' => ($connector['objectTypeTitle'] ?: '# missing #') . PHP_EOL . $connector['objectTitle']
                        ]
                    )
                );

                // Mark this object as exported.
                $this->exportedObjects[$connector['objectId']] = true;
            }
            else
            {
                if ($this->connectorToObjectMatch[$connector['parentId']])
                {
                    $this->connectorToObjectMatch[$connector['id']] = [
                        'objectId' => $this->connectorToObjectMatch[$connector['parentId']]['objectId'],
                        'title'    => $connector['title'],
                    ];
                }

                if ($this->connectorToObjectMatch[$connector['siblingId']])
                {
                    $this->connectorToObjectMatch[$connector['id']] = [
                        'objectId' => $this->connectorToObjectMatch[$connector['siblingId']]['objectId'],
                        'title'    => $connector['title'],
                    ];
                }
            }
        }
    }

    /**
     * Method for creating a object "Node".
     *
     * @param   array $data
     *
     * @return  \DOMElement
     */
    private function renderObjectNode(array $data)
    {
        // Prepare the node content.
        $node      = $this->dom->createElement('node');
        $nodeData  = $this->dom->createElement('data');
        $shapeNode = $this->dom->createElement('y:ShapeNode');
        $geometry  = $this->dom->createElement('y:Geometry');
        $geometry->setAttribute('width', '120');
        $geometry->setAttribute('height', '40');
        $fill = $this->dom->createElement('y:Fill');
        $fill->setAttribute('color', $data['color']);
        $fill->setAttribute('transparent', 'false');
        $borderStyle = $this->dom->createElement('y:BorderStyle');
        $borderStyle->setAttribute('type', 'line');
        $borderStyle->setAttribute('width', '1.0');
        $borderStyle->setAttribute('color', '#000000');
        $nodelabel = $this->dom->createElement('y:NodeLabel', $data['label'] ?: null);
        $shape     = $this->dom->createElement('y:Shape');
        $shape->setAttribute('type', 'rectangle');

        // Add the child elements to the node.
        $shapeNode->appendChild($geometry);
        $shapeNode->appendChild($fill);
        $shapeNode->appendChild($borderStyle);
        $shapeNode->appendChild($nodelabel);
        $shapeNode->appendChild($shape);

        $nodeData->setAttribute('key', 'd0');
        $nodeData->appendChild($shapeNode);

        $node->setAttribute('id', 'n' . $data['id']);
        $node->appendChild($nodeData);

        return $node;
    }

    /**
     * Method for creating a "Edge" (connection between to nodes).
     *
     * @param   array $data
     *
     * @return  \DOMElement
     */
    private function renderEdge(array $data)
    {
        // Prepare the edge content.
        $edge      = $this->dom->createElement('edge');
        $edgeData  = $this->dom->createElement('data');
        $polyline  = $this->dom->createElement('y:PolyLineEdge');
        $lineStyle = $this->dom->createElement('y:LineStyle');
        $lineStyle->setAttribute('type', 'line');
        $lineStyle->setAttribute('width', '1.0');
        $lineStyle->setAttribute('color', '#000000');
        $arrows = $this->dom->createElement('y:Arrows');
        $arrows->setAttribute('source', 'none');
        $arrows->setAttribute('target', 'standard');
        $edgeLabel       = $this->dom->createElement('y:EdgeLabel', $data['label'] ?: null);
        $edgeLabelSource = $this->dom->createElement('y:EdgeLabel', $data['labelSource'] ?: null);
        $edgeLabelSource->setAttribute('modelPosition', 'stail');
        $edgeLabelSource->setAttribute('modelName', 'six_pos');
        $edgeLabelTarget = $this->dom->createElement('y:EdgeLabel', $data['labelTarget'] ?: null);
        $edgeLabelTarget->setAttribute('modelPosition', 'ttail');
        $edgeLabelTarget->setAttribute('modelName', 'six_pos');
        $bendStyle = $this->dom->createElement('y:BendStyle');
        $bendStyle->setAttribute('smoothed', 'true');

        // Add the child elements to the edge.
        $polyline->appendChild($lineStyle);
        $polyline->appendChild($arrows);
        $polyline->appendChild($edgeLabel);
        $polyline->appendChild($edgeLabelSource);
        $polyline->appendChild($edgeLabelTarget);
        $polyline->appendChild($bendStyle);

        $edgeData->setAttribute('key', 'd1');
        $edgeData->appendChild($polyline);

        $edge->setAttribute('id', 'e' . $data['id']);
        $edge->setAttribute('source', 'n' . $data['source']);
        $edge->setAttribute('target', 'n' . $data['target']);
        $edge->appendChild($edgeData);

        return $edge;
    }
}
