<?php

namespace idoit\Module\Cabling\Model;

use isys_application;
use isys_helper_link as HelperLink;

/**
 * Cabling model
 *
 * @package     Modules
 * @subpackage  Cabling
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class Cabling extends \isys_component_dao
{
    /**
     * @var  \isys_cmdb_dao_category_g_connector
     */
    private $connectorDao;

    /**
     * @var \isys_component_template_language_manager
     */
    private $language;

    /**
     * @var  array
     */
    private $connectorIn = [];

    /**
     * @var  array
     */
    private $connectorOut = [];

    /**
     * @var array
     */
    private $recursionCheck = [];

    /**
     * @param  integer  $connectorID
     * @param  integer  $parent
     * @param  array   &$tree
     * @param  boolean  $onlyOneRow
     */
    public function getConnectorTree($connectorID, $parent, array &$tree = [], $onlyOneRow = false)
    {
        $outputResult = $this->connectorDao->get_data($connectorID, null, '', null, C__RECORD_STATUS__NORMAL);

        while ($row = $outputResult->get_row()) {
            $iteration = [
                'id'              => $row['isys_catg_connector_list__id'],
                'title'           => $row['isys_catg_connector_list__title'],
                'siblingId'       => $row['isys_catg_connector_list__isys_catg_connector_list__id'],
                'connectorType'   => $row['isys_catg_connector_list__isys_connection_type__id'],
                'parentId'        => $parent,
                'inner'           => true,
                'objectId'        => $row['isys_obj__id'],
                'objectTitle'     => $row['isys_obj__title'],
                'objectTypeId'    => $row['isys_obj__isys_obj_type__id'],
                'objectTypeTitle' => $this->language->get($row['object_type']),
                'objectTypeColor' => '#' . $row['isys_obj_type__color'],
                'doubling'        => ($connectorID != $row['isys_catg_connector_list__id'] && isset($this->recursionCheck[$row['isys_catg_connector_list__id']]))
            ];

            $tree[] = $iteration;

            $this->recursionCheck[$row['isys_catg_connector_list__id']] = true;

            // Now get all siblings.
            $siblingResult = $this->connectorDao->get_sibling_by_connector($row['isys_catg_connector_list__id']);

            while ($siblingRow = $siblingResult->get_row()) {
                $iteration = [
                    'id'            => $siblingRow['isys_catg_connector_list__id'],
                    'title'         => $siblingRow['isys_catg_connector_list__title'],
                    'siblingId'     => $siblingRow['isys_catg_connector_list__isys_catg_connector_list__id'],
                    'connectorType' => $siblingRow['isys_catg_connector_list__isys_connection_type__id'],
                    'parentId'      => $row['isys_catg_connector_list__id'],
                    'outer'         => true,
                    'doubling'      => isset($this->recursionCheck[$siblingRow['isys_catg_connector_list__id']])
                ];

                // @see CABLING-12 Prevent recursions
                if ($iteration['doubling']) {
                    $iteration['id'] = 'd' . $iteration['id'];
                    $tree[] = $iteration;
                    continue;
                }

                // Get some additional data for the connected item.
                if ($siblingRow['con_connector'] > 0) {
                    $sql = 'SELECT isys_obj__id, isys_obj__title, isys_obj_type__title, isys_catg_connector_list__type
                        FROM isys_obj
                        INNER JOIN isys_obj_type ON isys_obj_type__id = isys_obj__isys_obj_type__id
                        INNER JOIN isys_catg_connector_list ON isys_catg_connector_list__isys_obj__id = isys_obj__id
                        WHERE isys_catg_connector_list__id = ' . $this->connectorDao->convert_sql_id($siblingRow['con_connector']) . ';';

                    $connectorRow = $this->connectorDao->retrieve($sql)->get_row();

                    $iteration['connected'] = true;
                    $iteration['cableId'] = $siblingRow['cable_id'];
                    $iteration['cableTitle'] = $siblingRow['cable_title'];
                    $iteration['connectorId'] = $siblingRow['con_connector'];
                    $iteration['connectorTitle'] = $siblingRow['connector_name'];
                    $iteration['connectorInOut'] = $connectorRow['isys_catg_connector_list__type'];
                    $iteration['connectorObjId'] = $connectorRow['isys_obj__id'];
                    $iteration['connectorObjTitle'] = $connectorRow['isys_obj__title'];
                    $iteration['connectorObjTypeTitle'] = $this->language->get($connectorRow['isys_obj_type__title']);
                }

                $tree[] = $iteration;

                // This step needs to be the last one so that the tree can be iterated without missing information.
                if (!isset($this->recursionCheck[$iteration['connectorId']]) && $siblingRow['con_connector'] > 0) {
                    $this->recursionCheck[$siblingRow['isys_catg_connector_list__id']] = true;

                    $this->getConnectorTree(
                        $iteration['connectorId'],
                        $siblingRow['isys_catg_connector_list__id'],
                        $tree,
                        $onlyOneRow
                    );
                }

                if ($onlyOneRow) {
                    return;
                }
            }
        }
    }

    /**
     * @param  integer $objectID
     * @param  boolean $displayOnlyConnected
     * @param  array   $connectorWhitelist
     *
     * @return array
     */
    public function resolve($objectID, $displayOnlyConnected = false, array $connectorWhitelist = [])
    {
        $rootObject = $this->connectorDao->get_object($objectID, true)->get_row();

        $inputCondition = ' AND isys_catg_connector_list.isys_catg_connector_list__type = ' . $this->convert_sql_int(C__CONNECTOR__INPUT) . ' ';
        $inputResult = $this->connectorDao->get_data(null, $objectID, $inputCondition, null, C__RECORD_STATUS__NORMAL);

        $outputCondition = ' AND isys_catg_connector_list.isys_catg_connector_list__type = ' . $this->convert_sql_int(C__CONNECTOR__OUTPUT) . ' ';
        $outputResult = $this->connectorDao->get_data(null, $objectID, $outputCondition, null, C__RECORD_STATUS__NORMAL);

        $this->connectorIn = [
            [
                'id'              => 'root',
                'title'           => '',
                'objectId'        => $rootObject['isys_obj__id'],
                'objectTitle'     => $rootObject['isys_obj__title'],
                'objectTypeId'    => $rootObject['isys_obj_type__id'],
                'objectTypeTitle' => $this->language->get($rootObject['isys_obj_type__title']),
            ]
        ];

        $this->connectorOut = [
            [
                'id'              => 'root',
                'title'           => '',
                'objectId'        => $rootObject['isys_obj__id'],
                'objectTitle'     => $rootObject['isys_obj__title'],
                'objectTypeId'    => $rootObject['isys_obj_type__id'],
                'objectTypeTitle' => $this->language->get($rootObject['isys_obj_type__title']),
            ]
        ];

        $indexMatch = [];
        $this->recursionCheck = [];

        while ($inputRow = $inputResult->get_row()) {
            if (count($connectorWhitelist) && !in_array($inputRow['isys_catg_connector_list__id'], $connectorWhitelist, false)) {
                continue;
            }

            $iteration = [
                'id'              => $inputRow['isys_catg_connector_list__id'],
                'title'           => $inputRow['isys_catg_connector_list__title'],
                'objectId'        => $inputRow['isys_obj__id'],
                'objectTypeColor' => '#' . $inputRow['isys_obj_type__color'],
                'siblingId'       => $inputRow['isys_catg_connector_list__isys_catg_connector_list__id'],
                'connectorType'   => $inputRow['isys_catg_connector_list__isys_connection_type__id'],
                'parentId'        => 'root',
                'outer'           => true,
                'wired'           => false,
                'doubling'        => isset($this->recursionCheck[$inputRow['isys_catg_connector_list__id']])
            ];

            if ($iteration['doubling']) {
                continue;
            }

            $this->recursionCheck[$iteration['id']] = true;

            // Get some additional data for the connected item.
            if ($inputRow['con_connector'] > 0) {
                $sql = 'SELECT isys_obj__id, isys_obj__title, isys_obj_type__title, isys_catg_connector_list__type
                    FROM isys_obj
                    INNER JOIN isys_obj_type ON isys_obj_type__id = isys_obj__isys_obj_type__id
                    INNER JOIN isys_catg_connector_list ON isys_catg_connector_list__isys_obj__id = isys_obj__id
                    WHERE isys_catg_connector_list__id = ' . $this->connectorDao->convert_sql_id($inputRow['con_connector']) . ';';

                $connectorRow = $this->connectorDao->retrieve($sql)->get_row();

                $iteration['connected'] = true;
                $iteration['cableId'] = $inputRow['cable_id'];
                $iteration['cableTitle'] = $inputRow['cable_title'];
                $iteration['connectorId'] = $inputRow['con_connector'];
                $iteration['connectorTitle'] = $inputRow['connector_name'];
                $iteration['connectorInOut'] = $connectorRow['isys_catg_connector_list__type'];
                $iteration['connectorObjId'] = $connectorRow['isys_obj__id'];
                $iteration['connectorObjTitle'] = $connectorRow['isys_obj__title'];
                $iteration['connectorObjTypeTitle'] = $this->language->get($connectorRow['isys_obj_type__title']);
            } elseif ($displayOnlyConnected) {
                // We don't want to display "not connected" connectors - skip this :)
                continue;
            }

            // Set the node index.
            $indexMatch[$iteration['id']] = $iteration['index'] = count($this->connectorIn);

            $this->connectorIn[] = $iteration;

            // This step needs to be the last one so that the tree can be iterated without missing information.
            if (!$iteration['doubling'] && $inputRow['con_connector'] > 0) {
                $this->recursionCheck[$iteration['connectorId']] = true;

                $this->getConnectorTree(
                    $iteration['connectorId'],
                    $inputRow['isys_catg_connector_list__id'],
                    $this->connectorIn,
                    true
                );
            }
        }

        $this->recursionCheck = [];

        while ($outputRow = $outputResult->get_row()) {
            if (count($connectorWhitelist) && !in_array($outputRow['isys_catg_connector_list__id'], $connectorWhitelist)) {
                continue;
            }

            // @see  CABLING-35  Mark connectors that are wired to itself.
            $wiredToItself = $objectID == $outputRow['isys_obj__id'] && $outputRow['con_connector'] > 0;

            $iteration = [
                'id'              => $outputRow['isys_catg_connector_list__id'],
                'title'           => $outputRow['isys_catg_connector_list__title'],
                'objectId'        => $outputRow['isys_obj__id'],
                'objectTypeColor' => '#' . $outputRow['isys_obj_type__color'],
                'siblingId'       => $outputRow['isys_catg_connector_list__isys_catg_connector_list__id'],
                'connectorType'   => $outputRow['isys_catg_connector_list__isys_connection_type__id'],
                'parentId'        => 'root',
                'outer'           => true,
                'wired'           => false,
                'toIndex'         => null,
                'doubling'        => isset($this->recursionCheck[$outputRow['isys_catg_connector_list__id']]),
                'wiredToItself'   => $wiredToItself
            ];

            if ($iteration['doubling']) {
                continue;
            }

            $this->recursionCheck[$iteration['id']] = true;

            // Get some additional data for the connected item.
            if ($outputRow['con_connector'] > 0) {
                $sql = 'SELECT isys_obj__id, isys_obj__title, isys_obj_type__title, isys_catg_connector_list__type
                    FROM isys_obj
                    INNER JOIN isys_obj_type ON isys_obj_type__id = isys_obj__isys_obj_type__id
                    INNER JOIN isys_catg_connector_list ON isys_catg_connector_list__isys_obj__id = isys_obj__id
                    WHERE isys_catg_connector_list__id = ' . $this->connectorDao->convert_sql_id($outputRow['con_connector']) . ';';

                $connectorRow = $this->connectorDao->retrieve($sql)->get_row();

                $iteration['connected'] = true;
                $iteration['cableId'] = $outputRow['cable_id'];
                $iteration['cableTitle'] = $outputRow['cable_title'];
                $iteration['connectorId'] = $outputRow['con_connector'];
                $iteration['connectorTitle'] = $outputRow['connector_name'];
                $iteration['connectorInOut'] = $connectorRow['isys_catg_connector_list__type'];
                $iteration['connectorObjId'] = $connectorRow['isys_obj__id'];
                $iteration['connectorObjTitle'] = $connectorRow['isys_obj__title'];
                $iteration['connectorObjTypeTitle'] = $this->language->get($connectorRow['isys_obj_type__title']);
            } elseif ($displayOnlyConnected) {
                // We don't want to display "not connected" connectors - skip this :)
                continue;
            }

            // This connector is internally wired to another one (or multiple).
            if (isset($indexMatch[$outputRow['isys_catg_connector_list__isys_catg_connector_list__id']])) {
                $iteration['toIndex'] = $indexMatch[$outputRow['isys_catg_connector_list__isys_catg_connector_list__id']];
                $iteration['wired'] = true;

                $this->connectorIn[$indexMatch[$outputRow['isys_catg_connector_list__isys_catg_connector_list__id']]]['wired'] = true;
            }

            $this->connectorOut[] = $iteration;

            // This step needs to be the last one so that the tree can be iterated without missing information.
            if (!$wiredToItself && !$iteration['doubling'] && $outputRow['con_connector'] > 0 && $iteration['connectorId'] > 0) {
                $this->recursionCheck[$iteration['connectorId']] = true;

                $this->getConnectorTree(
                    $iteration['connectorId'],
                    $outputRow['isys_catg_connector_list__id'],
                    $this->connectorOut
                );
            }
        }

        return [
            'left'  => array_values($this->connectorIn),
            'right' => array_values($this->connectorOut)
        ];
    }

    /**
     * Method for retrieving information about the source object.
     *
     * @param integer $objectID
     *
     * @return array
     */
    public function getSourceObject($objectID)
    {
        $sql = 'SELECT 
            isys_obj__id AS objectId, 
            isys_obj__title AS objectTitle, 
            isys_obj_type__id AS objectTypeId, 
            isys_obj_type__title AS objectTypeTitle, 
            isys_obj_type__color AS objectTypeColor, 
            isys_cmdb_status__color AS cmdbStatusColor,
            isys_cmdb_status__title AS cmdbStatusTitle,
            isys_catg_image_list__image_link,
            isys_obj_type__obj_img_name
            FROM isys_obj
            INNER JOIN isys_obj_type ON isys_obj_type__id = isys_obj__isys_obj_type__id
            INNER JOIN isys_cmdb_status ON isys_cmdb_status__id = isys_obj__isys_cmdb_status__id
            LEFT JOIN isys_catg_image_list ON isys_catg_image_list__isys_obj__id = isys_obj__id
            WHERE isys_obj__id = ' . $this->convert_sql_id($objectID) . '
            LIMIT 1;';

        $result = $this->retrieve($sql)->get_row();

        global $g_dirs;
        $imageFolder = $g_dirs["images"];

        // Adding some additional data.
        $result['objectImage'] = $imageFolder . 'objecttypes/' . $result['isys_obj_type__obj_img_name'];
        $result['objectTypeColor'] = '#' . $result['objectTypeColor'];
        $result['objectTypeTitle'] = $this->language->get($result['objectTypeTitle']);
        $result['cmdbStatusColor'] = '#' . $result['cmdbStatusColor'];
        $result['cmdbStatusTitle'] = $this->language->get($result['cmdbStatusTitle']);
        $result['locationPath'] = \isys_tenantsettings::get('gui.empty_value', '-');
        $result['locationInRack'] = \isys_tenantsettings::get('gui.empty_value', '-');

        if (empty($result['isys_obj_type__obj_img_name'])) {
            $result['objectImage'] = $imageFolder . 'empty.gif';
        }

        if (!empty($result['isys_catg_image_list__image_link'])) {
            $result['objectImage'] = HelperLink::create_url([
                C__GET__MODULE_ID    => C__MODULE__CMDB,
                C__GET__FILE_MANAGER => 'image',
                'file'               => $result['isys_catg_image_list__image_link']
            ], true);
        }

        $locationRow = \isys_cmdb_dao_category_g_location::instance($this->m_db)->get_data(null, $objectID)->get_row();

        if (is_array($locationRow)) {
            // Only load the location path if a parent is set.
            if ($locationRow['isys_catg_location_list__parentid'] > 0) {
                $result['locationPath'] = (new \isys_popup_browser_location())->format_selection($objectID);
            }

            // Only load the position if it's really set.
            if ($locationRow['isys_catg_location_list__pos'] > 0) {
                $positions = \isys_cmdb_dao_category_g_location::instance($this->m_db)
                    ->get_free_rackslots(
                        $locationRow['isys_catg_location_list__parentid'],
                        $locationRow['isys_catg_location_list__insertion'],
                        $locationRow['isys_obj__id'],
                        $locationRow['isys_catg_location_list__option']
                    );

                foreach ($positions as $key => $value) {
                    if (explode(';', $key)[0] == $locationRow['isys_catg_location_list__pos']) {
                        $result['locationInRack'] = $value;
                    }
                }
            }
        }

        unset($result['isys_obj_type__obj_img_name'], $result['isys_catg_image_list__image_link']);

        return $result;
    }

    /**
     * Method for retrieving information about the source cable.
     *
     * @param integer $objectID
     *
     * @return array
     */
    public function getSourceCable($objectID)
    {
        $sql = 'SELECT 
            isys_obj__id AS objectId, 
            isys_obj__title AS objectTitle, 
            isys_obj_type__id AS objectTypeId, 
            isys_obj_type__title AS objectTypeTitle, 
            isys_obj_type__color AS objectTypeColor, 
            isys_cmdb_status__color AS cmdbStatusColor,
            isys_cmdb_status__title AS cmdbStatusTitle,
            isys_catg_image_list__image_link,
            isys_obj_type__obj_img_name
            FROM isys_obj
            INNER JOIN isys_obj_type ON isys_obj_type__id = isys_obj__isys_obj_type__id
            INNER JOIN isys_cmdb_status ON isys_cmdb_status__id = isys_obj__isys_cmdb_status__id
            LEFT JOIN isys_catg_image_list ON isys_catg_image_list__isys_obj__id = isys_obj__id
            WHERE isys_obj__id = ' . $this->convert_sql_id($objectID) . '
            LIMIT 1;';

        $result = $this->retrieve($sql)->get_row();

        global $g_dirs;
        $imageFolder = $g_dirs["images"];

        // Adding some additional data.
        $result['objectImage'] = $imageFolder . 'objecttypes/' . $result['isys_obj_type__obj_img_name'];
        $result['objectTypeColor'] = '#' . $result['objectTypeColor'];
        $result['objectTypeTitle'] = $this->language->get($result['objectTypeTitle']);
        $result['cmdbStatusColor'] = '#' . $result['cmdbStatusColor'];
        $result['cmdbStatusTitle'] = $this->language->get($result['cmdbStatusTitle']);
        $result['cableLength'] = \isys_tenantsettings::get('gui.empty_value', '-');
        $result['cableOccupancy'] = \isys_tenantsettings::get('gui.empty_value', '-');

        if (!empty($result['isys_catg_image_list__image_link'])) {
            $result['objectImage'] = HelperLink::create_url([
                C__GET__MODULE_ID    => C__MODULE__CMDB,
                C__GET__FILE_MANAGER => 'image',
                'file'               => $result['isys_catg_image_list__image_link']
            ], true);
        }

        $cableRow = \isys_cmdb_dao_category_g_cable::factory($this->m_db)->get_data(null, $objectID)->get_row();

        if (is_array($cableRow)) {
            if ($cableRow['isys_catg_cable_list__length'] > 0 && $cableRow['isys_catg_cable_list__isys_depth_unit__id'] > 0) {
                $result['cableLength'] = \isys_convert::measure(
                    $cableRow['isys_catg_cable_list__length'],
                    $cableRow['isys_depth_unit__const'],
                    C__CONVERT_DIRECTION__BACKWARD
                ) . $this->language->get($cableRow['isys_depth_unit__title']);
            }

            if ($cableRow['isys_catg_cable_list__isys_cable_occupancy__id'] > 0) {
                $cableOccupancy = \isys_cmdb_dao_dialog::factory($this->m_db)->set_table('isys_cable_occupancy')->get_data($cableRow['isys_catg_cable_list__isys_cable_occupancy__id']);

                if (is_array($cableOccupancy) && !empty($cableOccupancy['isys_cable_occupancy__title'])) {
                    $result['cableOccupancy'] = $this->language->get($cableOccupancy['isys_cable_occupancy__title']);
                }
            }

            if ($cableRow['isys_catg_cable_list__isys_cable_type__id'] > 0) {
                $cableOccupancy = \isys_cmdb_dao_dialog::factory($this->m_db)->set_table('isys_cable_type')->get_data($cableRow['isys_catg_cable_list__isys_cable_type__id']);

                if (is_array($cableOccupancy) && !empty($cableOccupancy['isys_cable_type__title'])) {
                    $result['cableType'] = $this->language->get($cableOccupancy['isys_cable_type__title']);
                }
            }
        }

        unset($result['isys_obj_type__obj_img_name'], $result['isys_catg_image_list__image_link']);

        return $result;
    }

    /**
     * @param \isys_component_database $p_db
     *
     * @throws \Exception
     */
    public function __construct(\isys_component_database $p_db)
    {
        parent::__construct($p_db);

        // Initialize database objects.
        $this->connectorDao = \isys_cmdb_dao_category_g_connector::instance($this->get_database_component());
        $this->language = isys_application::instance()->container->get('language');
    }
}
