<?php

namespace idoit\Module\Cabling\PDF;

use idoit\Module\Cabling\Model\Cabling;

/**
 * i-doit
 *
 * PDF Worker class.
 *
 * @package     Idoit\Module\Cabling\PDF
 * @subpackage  Cabling
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.9
 */
class Worker extends \TCPDF
{
    /**
     * PDF default font.
     *
     * @var  string
     */
    protected $defaultFont = 'helvetica';

    /**
     * Default "pt" font size.
     *
     * @var integer
     */
    protected $defaultFontSize = 10;

    /**
     * @var \isys_component_template
     */
    protected $template;

    /**
     * @var \isys_component_database
     */
    protected $database;

    /**
     * @var \isys_component_template_language_manager
     */
    protected $language;

    /**
     * @var array
     */
    protected $locationPathCache = [];

    /**
     * @var array
     */
    protected $locationInRackCache = [];

    /**
     * @var Cabling
     */
    protected $daoCabling;

    /**
     * Worker constructor.
     *
     * @param string  $orientation
     * @param string  $unit
     * @param string  $format
     * @param boolean $unicode
     * @param string  $encoding
     * @param boolean $diskcache
     * @param boolean $pdfa
     */
    public function __construct($orientation = 'P', $unit = 'mm', $format = 'A4', $unicode = true, $encoding = 'UTF-8', $diskcache = false, $pdfa = false)
    {
        parent::__construct($orientation, $unit, $format, $unicode, $encoding, $diskcache, $pdfa);

        $this->template = \isys_application::instance()->container->get('template');
        $this->database = \isys_application::instance()->container->get('database');
        $this->language = \isys_application::instance()->container->get('language');

        // Page orientation, default page unit and margin.
        $this->setPageOrientation($orientation);
        $this->setPageUnit($unit);
        $this->SetMargins(20, 30, 20, true);
        $this->SetFont($this->defaultFont, '', $this->defaultFontSize);

        // Set PDF title and subject.
        $this->SetTitle('PDF title');
        $this->SetSubject('PDF subject');

        // Set up the cabling Dao.
        $this->daoCabling = Cabling::instance($this->database);
    }

    /**
     * Render header.
     */
    public function Header()
    {
        $this->SetFont($this->defaultFont, 'B', 15);

        $this->SetY(15);

        $this->Cell(0, 10, 'i-doit Verkabelungsstrecke', 1, 2, 'C');
    }

    /**
     * Draw footer
     */
    public function Footer()
    {
        $this->SetY(-15);

        $this->SetFont($this->defaultFont, '', 8);

        $this->writeHTML('Seite {:pnp:} / {:ptp:}', true, false, true, false, 'C');
    }

    /**
     * @param array $rootData
     */
    public function printRootInformation(array $rootData)
    {
        $this->AddPage();

        // Remove the JS from the location path.
        $rootData['locationPath'] = \isys_helper_textformat::remove_scripts($rootData['locationPath']);

        $htmlTemplate = $this->template->assign('object', $rootData)
            ->fetch(\isys_module_cabling::getPath() . 'templates/pdf/rootInformation.tpl');

        $this->writeHTML(trim($htmlTemplate));
    }

    /**
     * @param array  $rootObjectData
     * @param array  $connectorPaths
     * @param string $direction
     */
    public function printConnectorPaths($rootObjectData, array $connectorPaths = [], $direction = 'right')
    {
        if (!count($connectorPaths)) {
            return;
        }

        $popupBrowserLocation = new \isys_popup_browser_location();
        $popupBrowserLocation->set_format_exclude_self(true);
        $connectorTypeDialog = new \isys_cmdb_dao_dialog($this->database, 'isys_connection_type');

        $locationDao = new \isys_cmdb_dao_category_g_location($this->database);

        foreach ($connectorPaths as &$connectorPath) {
            $connectorCount = count($connectorPath);
            $connectors = [];

            if ($connectorCount < 2) {
                unset($connectorPath);
                continue;
            }

            for ($i = 0; $i < $connectorCount; $i++) {
                if ($i === 0) {
                    // This is the "root" object.
                    $connectorPath[$i] = $rootObjectData;
                    $connectorPath[$i]['locationPath'] = \isys_helper_textformat::remove_scripts($connectorPath[$i]['locationPath']);
                }

                $connectorPath[$i]['connectorTypeTitle'] = '-';

                if (isset($connectorPath[$i]['connectorType']) && $connectorPath[$i]['connectorType'] > 0) {
                    $connectorPath[$i]['connectorTypeTitle'] = $connectorTypeDialog->get_data($connectorPath[$i]['connectorType'])['title'];
                }

                if ($connectorPath[$i]['inner']) {
                    $location = $locationDao->get_data(null, $connectorPath[$i]['objectId'])
                        ->get_row();

                    $connectorPath[$i]['locationPath'] = null;
                    $connectorPath[$i]['locationInRack'] = null;
                    $locationInRack = null;

                    if ($location['isys_catg_location_list__parentid'] > 0) {
                        if (!isset($this->locationPathCache[$connectorPath[$i]['objectId']])) {
                            $this->locationPathCache[$connectorPath[$i]['objectId']] = \isys_helper_textformat::remove_scripts($popupBrowserLocation->format_selection($connectorPath[$i]['objectId']));
                        }

                        if (!isset($this->locationInRackCache[$connectorPath[$i]['objectId']])) {
                            $positions = $locationDao->get_free_rackslots($location['isys_catg_location_list__parentid'], $location['isys_catg_location_list__insertion'],
                                $location['isys_obj__id'], $location['isys_catg_location_list__option']);

                            foreach ($positions as $key => $value) {
                                if (explode(';', $key)[0] == $location['isys_catg_location_list__pos']) {
                                    $locationInRack = $value;
                                }
                            }

                            $this->locationInRackCache[$connectorPath[$i]['objectId']] = $locationInRack;
                        }

                        $connectorPath[$i]['locationPath'] = $this->locationPathCache[$connectorPath[$i]['objectId']];
                        $connectorPath[$i]['locationInRack'] = $this->locationInRackCache[$connectorPath[$i]['objectId']];
                    }
                }

                if ($connectorPath[$i]['outer']) {
                    // Fetch additional cable data.
                    $connectors[] = [
                        'in'    => $connectorPath[$i - 1],
                        'cable' => $this->daoCabling->getSourceCable($connectorPath[$i]['cableId']),
                        'out'   => $connectorPath[$i],
                        'next'  => (($i + 1) < $connectorCount)
                    ];
                }

                // Explicitly add the last connector, if it has no "outer" connector.
                if (($i + 1) === $connectorCount && $connectorPath[$i]['inner']) {
                    $connectors[] = [
                        'in'   => $connectorPath[$i],
                        'out'  => null,
                        'next' => false
                    ];
                }
            }

            $connectorPath = $connectors;
        }

        $htmlTemplate = $this->template->assign('baseUrl', \isys_helper_link::get_base())
            ->assign('connectorPaths', array_filter($connectorPaths))
            ->assign('connectorDirection', ($direction === 'left' ?
                $this->language->get('LC__MODULE__CABLING__PDF__DIRECTION_INPUT') :
                $this->language->get('LC__MODULE__CABLING__PDF__DIRECTION_OUTPUT')))
            ->fetch(\isys_module_cabling::getPath() . 'templates/pdf/cablingPaths.tpl');

        $this->writeHTML(trim($htmlTemplate));
    }
}
