<?php

namespace idoit\Module\Cabling\Controller;

use idoit\Module\Cabling\Model\Cabling;
use isys_application;
use isys_cmdb_dao_category_g_connector as DaoConnector;
use isys_cmdb_dao_dialog as DaoDialog;
use isys_component_dao_logbook as DaoLogbook;
use isys_format_json as JSON;
use isys_register;

/**
 * Ajax controller
 *
 * @package     Modules
 * @subpackage  Cabling
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class Ajax extends Main
{
    protected $response;

    /**
     * Overwriting the "handle" method.
     *
     * @param   isys_register    $p_request
     * @param   isys_application $p_application
     *
     * @return  null
     */
    public function handle(isys_register $p_request, isys_application $p_application)
    {
        return null;
    }

    /**
     * Pre method gets called by the framework.
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function pre()
    {
        header('Content-Type: application/json');

        $this->response = [
            'success' => true,
            'data'    => null,
            'message' => null
        ];
    }

    /**
     * Method for retrieving the necessary cabling (and object-) data.
     *
     * @param  isys_register $request
     */
    public function getCablingData(isys_register $request)
    {
        try {
            $post = $request->get('POST');
            $objectId = $post->get('objectId');
            $hideDisconnected = $post->get('hideDisconnected');

            if ($objectId > 0) {
                $cablingDao = Cabling::instance($this->container->get('database'));

                $this->response['data'] = [
                    'rootObject'     => $cablingDao->getSourceObject($objectId),
                    'connections'    => $cablingDao->resolve($objectId, (bool)$hideDisconnected),
                    'cablingOptions' => []
                ];
            } else {
                $this->response['data'] = null;
            }
        } catch (\Exception $e) {
            $this->response['success'] = false;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * Method for retrieving object information for the object info window on the top left.
     *
     * @param isys_register $request
     */
    public function getObjectInformation(isys_register $request)
    {
        try {
            $objectId = $request->get('id', 0);

            if ($objectId > 0) {
                $cablingDao = Cabling::instance($this->container->get('database'));

                if ($request->get('POST')->get('type', 'object') === 'object') {
                    $this->response['data'] = $cablingDao->getSourceObject($objectId);
                } else {
                    $this->response['data'] = $cablingDao->getSourceCable($objectId);
                }
            } else {
                $this->response['data'] = null;
            }
        } catch (\Exception $e) {
            $this->response['success'] = false;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * Method for updating connector types
     *
     * @param isys_register $request
     */
    public function updateConnectorTypes(isys_register $request)
    {
        try {
            $post = $request->get('POST');

            $connectorTypeId = $post->get('connectorTypeId');
            $connectors = JSON::decode($post->get('connectors'));

            if (is_array($connectors) && count($connectors)) {
                $dao = DaoConnector::instance($this->container->get('database'));
                $logbook = DaoLogbook::instance($this->container->get('database'));
                $connectorTypes = new DaoDialog($this->container->get('database'), 'isys_connection_type');

                // CABLING-5 write logbook entries when changing a connector type.
                $sql = 'SELECT * 
                    FROM isys_catg_connector_list 
                    INNER JOIN isys_obj ON isys_obj__id = isys_catg_connector_list__isys_obj__id 
                    INNER JOIN isys_obj_type ON isys_obj_type__id = isys_obj__isys_obj_type__id
                    WHERE isys_catg_connector_list__id ' . $dao->prepare_in_condition($connectors) . ' 
                    AND isys_catg_connector_list__isys_connection_type__id != ' . $dao->convert_sql_id($connectorTypeId) . ';';

                $result = $dao->retrieve($sql);

                while ($row = $result->get_row()) {
                    $p_changes = [
                        'isys_cmdb_dao_category_g_connector::connection_type' => [
                            'from' => $connectorTypes->get_data($row['isys_catg_connector_list__isys_connection_type__id'])['title'],
                            'to'   => $connectorTypes->get_data($connectorTypeId)['title'],
                        ]
                    ];

                    // Create logbook entries.
                    $logbook->set_entry(
                        $this->container->get('language')->get('LC__MODULE__CABLING__LOGBOOK__CONNECTOR_TYPES_CHANGED', $row['isys_catg_connector_list__title']),
                        '',
                        null,
                        C__LOGBOOK__ALERT_LEVEL__0,
                        $row['isys_obj__id'],
                        $row['isys_obj__title'],
                        $row['isys_obj_type__title'],
                        'LC__CMDB__CATG__CONNECTORS',
                        C__LOGBOOK_SOURCE__INTERNAL,
                        serialize($p_changes)
                    );
                }

                $dao->update_connector_type($connectors, $connectorTypeId);
            } else {
                $this->response['data'] = null;
            }
        } catch (\Exception $e) {
            $this->response['success'] = false;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * Post method gets called by the framework.
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function post()
    {
        echo JSON::encode($this->response);

        die;
    }
}
