<?php

namespace idoit\Module\Cabling\Model;

use isys_application as Application;
use isys_cmdb_dao_category_g_connector;
use isys_component_template_language_manager;
use isys_helper_color;

/**
 * Cabling model
 *
 * @package     Modules
 * @subpackage  Cabling
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class Cabling extends \isys_component_dao
{
    /**
     * @var isys_cmdb_dao_category_g_connector|\isys_component_dao
     */
    private isys_cmdb_dao_category_g_connector $connectorDao;

    /**
     * @var isys_component_template_language_manager|object|null
     */
    private isys_component_template_language_manager $language;

    /**
     * @var array
     */
    private array $recursionCheck = [];

    /**
     * @param       $connectorID
     * @param       $parent
     * @param array $tree
     * @param       $onlyOneRow
     *
     * @return void
     * @throws \isys_exception_database
     */
    public function getConnectorTree($connectorID, $parent, array &$tree = [], $onlyOneRow = false)
    {
        $outputResult = $this->connectorDao->get_data($connectorID, null, '', null, C__RECORD_STATUS__NORMAL);

        while ($row = $outputResult->get_row()) {
            $iteration = [
                'id'              => $row['isys_catg_connector_list__id'],
                'title'           => $row['isys_catg_connector_list__title'],
                'siblingId'       => $row['isys_catg_connector_list__isys_catg_connector_list__id'],
                'connectorType'   => $row['isys_catg_connector_list__isys_connection_type__id'],
                'parentId'        => $parent,
                'inner'           => true,
                'objectId'        => $row['isys_obj__id'],
                'objectTitle'     => $row['isys_obj__title'],
                'objectTypeId'    => $row['isys_obj__isys_obj_type__id'],
                'objectTypeTitle' => $this->language->get($row['object_type']),
                'objectTypeColor' => isys_helper_color::unifyHexColor((string)$row['isys_obj_type__color']),
                'doubling'        => ($connectorID != $row['isys_catg_connector_list__id'] && isset($this->recursionCheck[$row['isys_catg_connector_list__id']]))
            ];

            $tree[] = $iteration;

            $this->recursionCheck[$row['isys_catg_connector_list__id']] = true;

            // Now get all siblings.
            $siblingResult = $this->connectorDao->get_sibling_by_connector($row['isys_catg_connector_list__id']);

            while ($siblingRow = $siblingResult->get_row()) {
                $iteration = [
                    'id'            => $siblingRow['isys_catg_connector_list__id'],
                    'title'         => $siblingRow['isys_catg_connector_list__title'],
                    'siblingId'     => $siblingRow['isys_catg_connector_list__isys_catg_connector_list__id'],
                    'connectorType' => $siblingRow['isys_catg_connector_list__isys_connection_type__id'],
                    'parentId'      => $row['isys_catg_connector_list__id'],
                    'outer'         => true,
                    'doubling'      => isset($this->recursionCheck[$siblingRow['isys_catg_connector_list__id']])
                ];

                // @see CABLING-12 Prevent recursions
                if ($iteration['doubling']) {
                    $iteration['id'] = 'd' . $iteration['id'];
                    $tree[] = $iteration;
                    continue;
                }

                // Get some additional data for the connected item.
                if ($siblingRow['con_connector'] > 0) {
                    $sql = 'SELECT isys_obj__id, isys_obj__title, isys_obj_type__title, isys_catg_connector_list__type
                        FROM isys_obj
                        INNER JOIN isys_obj_type ON isys_obj_type__id = isys_obj__isys_obj_type__id
                        INNER JOIN isys_catg_connector_list ON isys_catg_connector_list__isys_obj__id = isys_obj__id
                        WHERE isys_catg_connector_list__id = ' . $this->connectorDao->convert_sql_id($siblingRow['con_connector']) . ';';

                    $connectorRow = $this->connectorDao->retrieve($sql)->get_row();

                    $iteration['connected'] = true;
                    $iteration['cableId'] = $siblingRow['cable_id'];
                    $iteration['cableTitle'] = $siblingRow['cable_title'];
                    $iteration['connectorId'] = $siblingRow['con_connector'];
                    $iteration['connectorTitle'] = $siblingRow['connector_name'];
                    $iteration['connectorInOut'] = $connectorRow['isys_catg_connector_list__type'];
                    $iteration['connectorObjId'] = $connectorRow['isys_obj__id'];
                    $iteration['connectorObjTitle'] = $connectorRow['isys_obj__title'];
                    $iteration['connectorObjTypeTitle'] = $this->language->get($connectorRow['isys_obj_type__title']);
                }

                $tree[] = $iteration;

                // This step needs to be the last one so that the tree can be iterated without missing information.
                if (!isset($this->recursionCheck[$iteration['connectorId']]) && $siblingRow['con_connector'] > 0) {
                    $this->recursionCheck[$siblingRow['isys_catg_connector_list__id']] = true;

                    $this->getConnectorTree(
                        $iteration['connectorId'],
                        $siblingRow['isys_catg_connector_list__id'],
                        $tree,
                        $onlyOneRow
                    );
                }

                if ($onlyOneRow) {
                    return;
                }
            }
        }
    }

    /**
     * @param int   $objectID
     * @param bool  $displayOnlyConnected
     * @param array $connectorWhitelist
     *
     * @return array
     * @throws \isys_exception_database
     */
    public function resolve($objectID, $displayOnlyConnected = false, array $connectorWhitelist = []): array
    {
        $rootObject = $this->connectorDao->get_object($objectID, true)->get_row();

        $inputCondition = ' AND isys_catg_connector_list.isys_catg_connector_list__type = ' . $this->convert_sql_int(C__CONNECTOR__INPUT) . ' ';
        $inputResult = $this->connectorDao->get_data(null, $objectID, $inputCondition, null, C__RECORD_STATUS__NORMAL);

        $outputCondition = ' AND isys_catg_connector_list.isys_catg_connector_list__type = ' . $this->convert_sql_int(C__CONNECTOR__OUTPUT) . ' ';
        $outputResult = $this->connectorDao->get_data(null, $objectID, $outputCondition, null, C__RECORD_STATUS__NORMAL);

        $connectorIn = [
            [
                'id'              => 'root',
                'title'           => '',
                'objectId'        => $rootObject['isys_obj__id'],
                'objectTitle'     => $rootObject['isys_obj__title'],
                'objectTypeId'    => $rootObject['isys_obj_type__id'],
                'objectTypeTitle' => $this->language->get($rootObject['isys_obj_type__title']),
            ]
        ];

        $connectorOut = [
            [
                'id'              => 'root',
                'title'           => '',
                'objectId'        => $rootObject['isys_obj__id'],
                'objectTitle'     => $rootObject['isys_obj__title'],
                'objectTypeId'    => $rootObject['isys_obj_type__id'],
                'objectTypeTitle' => $this->language->get($rootObject['isys_obj_type__title']),
            ]
        ];

        $indexMatch = [];
        $this->recursionCheck = [];

        while ($inputRow = $inputResult->get_row()) {
            if (count($connectorWhitelist) && !in_array($inputRow['isys_catg_connector_list__id'], $connectorWhitelist, false)) {
                continue;
            }

            // @see  CABLING-35  Mark connectors that are wired to itself.
            $wiredToItself = false;

            $iteration = [
                'id'              => $inputRow['isys_catg_connector_list__id'],
                'title'           => $inputRow['isys_catg_connector_list__title'],
                'objectId'        => $inputRow['isys_obj__id'],
                'objectTypeColor' => isys_helper_color::unifyHexColor((string)$inputRow['isys_obj_type__color']),
                'siblingId'       => $inputRow['isys_catg_connector_list__isys_catg_connector_list__id'],
                'connectorType'   => $inputRow['isys_catg_connector_list__isys_connection_type__id'],
                'parentId'        => 'root',
                'outer'           => true,
                'wired'           => false,
                'doubling'        => isset($this->recursionCheck[$inputRow['isys_catg_connector_list__id']])
            ];

            if ($iteration['doubling']) {
                continue;
            }

            $this->recursionCheck[$iteration['id']] = true;

            // Get some additional data for the connected item.
            if ($inputRow['con_connector'] > 0) {
                $sql = 'SELECT isys_obj__id, isys_obj__title, isys_obj_type__title, isys_catg_connector_list__type
                    FROM isys_obj
                    INNER JOIN isys_obj_type ON isys_obj_type__id = isys_obj__isys_obj_type__id
                    INNER JOIN isys_catg_connector_list ON isys_catg_connector_list__isys_obj__id = isys_obj__id
                    WHERE isys_catg_connector_list__id = ' . $this->connectorDao->convert_sql_id($inputRow['con_connector']) . ';';

                $connectorRow = $this->connectorDao->retrieve($sql)->get_row();

                // @see  CABLING-39  Fix the 'wired to self' indicator.
                $wiredToItself = ($connectorRow['isys_obj__id'] == $objectID);

                $iteration['connected'] = true;
                $iteration['cableId'] = $inputRow['cable_id'];
                $iteration['cableTitle'] = $inputRow['cable_title'];
                $iteration['connectorId'] = $inputRow['con_connector'];
                $iteration['connectorTitle'] = $inputRow['connector_name'];
                $iteration['connectorInOut'] = $connectorRow['isys_catg_connector_list__type'];
                $iteration['connectorObjId'] = $connectorRow['isys_obj__id'];
                $iteration['connectorObjTitle'] = $connectorRow['isys_obj__title'];
                $iteration['connectorObjTypeTitle'] = $this->language->get($connectorRow['isys_obj_type__title']);
            } elseif ($displayOnlyConnected) {
                // We don't want to display "not connected" connectors - skip this :)
                continue;
            }

            $iteration['wiredToItself'] = $wiredToItself;

            // Set the node index.
            $indexMatch[$iteration['id']] = $iteration['index'] = count($connectorIn);

            $connectorIn[] = $iteration;

            // This step needs to be the last one so that the tree can be iterated without missing information.
            if (!$iteration['wiredToItself'] && !$iteration['doubling'] && $inputRow['con_connector'] > 0 && $iteration['connectorId'] > 0) {
                $this->recursionCheck[$iteration['connectorId']] = true;

                $this->getConnectorTree(
                    $iteration['connectorId'],
                    $inputRow['isys_catg_connector_list__id'],
                    $connectorIn,
                    true
                );
            }
        }

        $this->recursionCheck = [];

        while ($outputRow = $outputResult->get_row()) {
            if (count($connectorWhitelist) && !in_array($outputRow['isys_catg_connector_list__id'], $connectorWhitelist)) {
                continue;
            }

            // @see  CABLING-35  Mark connectors that are wired to itself.
            $wiredToItself = false;

            $iteration = [
                'id'              => $outputRow['isys_catg_connector_list__id'],
                'title'           => $outputRow['isys_catg_connector_list__title'],
                'objectId'        => $outputRow['isys_obj__id'],
                'objectTypeColor' => isys_helper_color::unifyHexColor((string)$outputRow['isys_obj_type__color']),
                'siblingId'       => $outputRow['isys_catg_connector_list__isys_catg_connector_list__id'],
                'connectorType'   => $outputRow['isys_catg_connector_list__isys_connection_type__id'],
                'parentId'        => 'root',
                'outer'           => true,
                'wired'           => false,
                'doubling'        => isset($this->recursionCheck[$outputRow['isys_catg_connector_list__id']]),
            ];

            if ($iteration['doubling']) {
                continue;
            }

            $this->recursionCheck[$iteration['id']] = true;

            // Get some additional data for the connected item.
            if ($outputRow['con_connector'] > 0) {
                $sql = 'SELECT isys_obj__id, isys_obj__title, isys_obj_type__title, isys_catg_connector_list__type
                    FROM isys_obj
                    INNER JOIN isys_obj_type ON isys_obj_type__id = isys_obj__isys_obj_type__id
                    INNER JOIN isys_catg_connector_list ON isys_catg_connector_list__isys_obj__id = isys_obj__id
                    WHERE isys_catg_connector_list__id = ' . $this->connectorDao->convert_sql_id($outputRow['con_connector']) . ';';

                $connectorRow = $this->connectorDao->retrieve($sql)->get_row();

                // @see  CABLING-39  Fix the 'wired to self' indicator.
                $wiredToItself = ($connectorRow['isys_obj__id'] == $objectID);

                $iteration['connected'] = true;
                $iteration['cableId'] = $outputRow['cable_id'];
                $iteration['cableTitle'] = $outputRow['cable_title'];
                $iteration['connectorId'] = $outputRow['con_connector'];
                $iteration['connectorTitle'] = $outputRow['connector_name'];
                $iteration['connectorInOut'] = $connectorRow['isys_catg_connector_list__type'];
                $iteration['connectorObjId'] = $connectorRow['isys_obj__id'];
                $iteration['connectorObjTitle'] = $connectorRow['isys_obj__title'];
                $iteration['connectorObjTypeTitle'] = $this->language->get($connectorRow['isys_obj_type__title']);
            } elseif ($displayOnlyConnected) {
                // We don't want to display "not connected" connectors - skip this :)
                continue;
            }

            $iteration['wiredToItself'] = $wiredToItself;

            // This connector is internally wired to another one (or multiple).
            if (isset($indexMatch[$outputRow['isys_catg_connector_list__isys_catg_connector_list__id']])) {
                $iteration['wired'] = true;

                $connectorIn[$indexMatch[$outputRow['isys_catg_connector_list__isys_catg_connector_list__id']]]['wired'] = true;
            }

            $connectorOut[] = $iteration;

            // This step needs to be the last one so that the tree can be iterated without missing information.
            if (!$wiredToItself && !$iteration['doubling'] && $outputRow['con_connector'] > 0 && $iteration['connectorId'] > 0) {
                $this->recursionCheck[$iteration['connectorId']] = true;

                $this->getConnectorTree(
                    $iteration['connectorId'],
                    $outputRow['isys_catg_connector_list__id'],
                    $connectorOut
                );
            }
        }

        return [
            'left'  => array_values($connectorIn),
            'right' => array_values($connectorOut)
        ];
    }

    /**
     * @param $objectID
     *
     * @return array
     * @throws \isys_exception_database
     */
    public function getSourceObject($objectID): array
    {
        $sql = 'SELECT
            isys_obj__id AS objectId,
            isys_obj__title AS objectTitle,
            isys_obj_type__id AS objectTypeId,
            isys_obj_type__title AS objectTypeTitle,
            isys_obj_type__color AS objectTypeColor,
            isys_cmdb_status__color AS cmdbStatusColor,
            isys_cmdb_status__title AS cmdbStatusTitle
            FROM isys_obj
            INNER JOIN isys_obj_type ON isys_obj_type__id = isys_obj__isys_obj_type__id
            INNER JOIN isys_cmdb_status ON isys_cmdb_status__id = isys_obj__isys_cmdb_status__id
            WHERE isys_obj__id = ' . $this->convert_sql_id($objectID) . '
            LIMIT 1;';

        $result = $this->retrieve($sql)->get_row();

        // Adding some additional data.
        $result['objectImage'] = Application::instance()->container->get('route_generator')->generate('cmdb.object.image', ['objectId' => $objectID]);
        $result['objectTypeColor'] = isys_helper_color::unifyHexColor((string)$result['objectTypeColor']);
        $result['objectTypeTitle'] = $this->language->get($result['objectTypeTitle']);
        $result['cmdbStatusColor'] = isys_helper_color::unifyHexColor((string)$result['cmdbStatusColor']);
        $result['cmdbStatusTitle'] = $this->language->get($result['cmdbStatusTitle']);
        $result['locationPath'] = \isys_tenantsettings::get('gui.empty_value', '-');
        $result['locationInRack'] = \isys_tenantsettings::get('gui.empty_value', '-');

        $locationRow = \isys_cmdb_dao_category_g_location::instance($this->m_db)->get_data(null, $objectID)->get_row();

        if (is_array($locationRow)) {
            // Only load the location path if a parent is set.
            if ($locationRow['isys_catg_location_list__parentid'] > 0) {
                $result['locationPath'] = (new \isys_popup_browser_location())->format_selection($objectID);
            }

            // Only load the position if it's really set.
            if ($locationRow['isys_catg_location_list__pos'] > 0) {
                $positions = \isys_cmdb_dao_category_g_location::instance($this->m_db)
                    ->get_free_rackslots(
                        $locationRow['isys_catg_location_list__parentid'],
                        $locationRow['isys_catg_location_list__insertion'],
                        $locationRow['isys_obj__id'],
                        $locationRow['isys_catg_location_list__option']
                    );

                foreach ($positions as $key => $value) {
                    if (explode(';', $key)[0] == $locationRow['isys_catg_location_list__pos']) {
                        $result['locationInRack'] = $value;
                    }
                }
            }
        }

        return $result;
    }

    /**
     * @param $objectID
     *
     * @return array
     * @throws \isys_exception_database
     */
    public function getSourceCable($objectID): array
    {
        $sql = 'SELECT
            isys_obj__id AS objectId,
            isys_obj__title AS objectTitle,
            isys_obj_type__id AS objectTypeId,
            isys_obj_type__title AS objectTypeTitle,
            isys_obj_type__color AS objectTypeColor,
            isys_cmdb_status__color AS cmdbStatusColor,
            isys_cmdb_status__title AS cmdbStatusTitle
            FROM isys_obj
            INNER JOIN isys_obj_type ON isys_obj_type__id = isys_obj__isys_obj_type__id
            INNER JOIN isys_cmdb_status ON isys_cmdb_status__id = isys_obj__isys_cmdb_status__id
            WHERE isys_obj__id = ' . $this->convert_sql_id($objectID) . '
            LIMIT 1;';

        $result = $this->retrieve($sql)->get_row();

        // Adding some additional data.
        $result['objectImage'] = Application::instance()->container->get('route_generator')->generate('cmdb.object.image', ['objectId' => $objectID]);
        $result['objectTypeColor'] = isys_helper_color::unifyHexColor((string)$result['objectTypeColor']);
        $result['objectTypeTitle'] = $this->language->get($result['objectTypeTitle']);
        $result['cmdbStatusColor'] = isys_helper_color::unifyHexColor((string)$result['cmdbStatusColor']);
        $result['cmdbStatusTitle'] = $this->language->get($result['cmdbStatusTitle']);
        $result['cableLength'] = \isys_tenantsettings::get('gui.empty_value', '-');
        $result['cableOccupancy'] = \isys_tenantsettings::get('gui.empty_value', '-');

        $cableRow = \isys_cmdb_dao_category_g_cable::factory($this->m_db)->get_data(null, $objectID)->get_row();

        if (is_array($cableRow)) {
            if ($cableRow['isys_catg_cable_list__length'] > 0 && $cableRow['isys_catg_cable_list__isys_depth_unit__id'] > 0) {
                $result['cableLength'] = \isys_convert::measure(
                    $cableRow['isys_catg_cable_list__length'],
                    $cableRow['isys_depth_unit__const'],
                    C__CONVERT_DIRECTION__BACKWARD
                ) . $this->language->get($cableRow['isys_depth_unit__title']);
            }

            if ($cableRow['isys_catg_cable_list__isys_cable_occupancy__id'] > 0) {
                $cableOccupancy = \isys_cmdb_dao_dialog::factory($this->m_db)->set_table('isys_cable_occupancy')->get_data($cableRow['isys_catg_cable_list__isys_cable_occupancy__id']);

                if (is_array($cableOccupancy) && !empty($cableOccupancy['isys_cable_occupancy__title'])) {
                    $result['cableOccupancy'] = $this->language->get($cableOccupancy['isys_cable_occupancy__title']);
                }
            }

            if ($cableRow['isys_catg_cable_list__isys_cable_type__id'] > 0) {
                $cableOccupancy = \isys_cmdb_dao_dialog::factory($this->m_db)->set_table('isys_cable_type')->get_data($cableRow['isys_catg_cable_list__isys_cable_type__id']);

                if (is_array($cableOccupancy) && !empty($cableOccupancy['isys_cable_type__title'])) {
                    $result['cableType'] = $this->language->get($cableOccupancy['isys_cable_type__title']);
                }
            }
        }

        return $result;
    }

    /**
     * @param \isys_component_database $p_db
     *
     * @throws \Exception
     */
    public function __construct(\isys_component_database $p_db)
    {
        parent::__construct($p_db);

        // Initialize database objects.
        $this->connectorDao = \isys_cmdb_dao_category_g_connector::instance($this->get_database_component());
        $this->language = Application::instance()->container->get('language');
    }
}
