<?php

namespace idoit\Module\Cabling\View;

use idoit\View\Base as ViewBase;
use isys_application as Application;
use isys_format_json;
use isys_helper_color;
use isys_module_cabling as CablingModule;
use isys_popup_browser_object_ng as ObjectBrowser;

/**
 * Visualization view
 *
 * @package     Modules
 * @subpackage  Cabling
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class Visualization extends ViewBase
{
    /**
     * @param   \isys_module             $module
     * @param   \isys_component_template $template
     * @param   \idoit\Model\Dao\Base    $model
     *
     * @return  $this
     */
    public function process(\isys_module $module, \isys_component_template $template, \idoit\Model\Dao\Base $model)
    {
        $routeGenerator = Application::instance()->container->get('route_generator');

        // First order: retrieve the current cabling ID.
        $cablingObjectID = $this->request->get('id', 0);

        // Set paths to templates.
        $this->paths['leftcontent'] = CablingModule::getPath() . 'templates/visualizationLeft.tpl';
        $this->paths['contenttop'] = CablingModule::getPath() . 'templates/visualizationTop.tpl';
        $this->paths['contentbottomcontent'] = CablingModule::getPath() . 'templates/visualization.tpl';

        $objectTypes = [];

        $objectTypeRes = $model->getObjectTypes();

        if (count($objectTypeRes)) {
            $language = Application::instance()->container->get('language');

            while ($row = $objectTypeRes->get_row()) {
                $objectTypes[$row['isys_obj_type__id']] = [
                    'icon'  => $routeGenerator->generate('cmdb.object-type.image', ['objectTypeId' => $row['isys_obj_type__id']]),
                    'title' => $language->get($row['isys_obj_type__title']),
                    'color' => isys_helper_color::unifyHexColor((string)$row['isys_obj_type__color'])
                ];
            }
        }

        $connectorTypesForDialog = [];
        $connectorTypes = $model->getConnectionTypes();

        foreach ($connectorTypes as $id => $connector) {
            $connectorTypesForDialog[$id] = $connector['title'];
        }

        // Prepare rules.
        $contentSmartyRules = [];
        $leftSmartyRules = [];
        $topSmartyRules = [
            'cabling_object_browser'  => [
                'p_strValue'                         => $cablingObjectID,
                'p_bInfoIconSpacer'                  => 0,
                ObjectBrowser::C__CALLBACK__ACCEPT => 'window.initializeCabling($F(\'cabling_object_browser__HIDDEN\'));',
                ObjectBrowser::C__DISABLE_DETACH   => true,
                ObjectBrowser::C__CAT_FILTER       => 'C__CATG__CABLING;C__CATG__CONNECTOR;C__CATG__NETWORK_PORT;C__CATG__NETWORK_LOG_PORT'
            ],
            'cabling_connector_types' => [
                'p_arData'              => $connectorTypesForDialog,
                'p_strClass'            => 'input-block',
                'p_bDbFieldNN'          => true,
                'inputGroupMarginClass' => 'p5'
            ]
        ];

        $preferences = Application::instance()->container->get('settingsUser')->get('cabling.visualization-preferences');

        if (!empty($preferences) && isys_format_json::is_json_array($preferences)) {
            $preferences = isys_format_json::decode($preferences);
        } else {
            $preferences = [
                'displayWiring'        => true,
                'displayOnlyConnected' => false,
                'displayCableNames'    => false
            ];
        }

        $template->activate_editmode()
            ->assign('cablingWwwPath', CablingModule::getWwwPath())
            ->assign('connectorTypes', $connectorTypes)
            ->assign('objectId', $cablingObjectID)
            ->assign('urlBase', \isys_helper_link::get_base())
            ->assign('objectTypeData', $objectTypes)
            ->assign('authEdit', (int)$module->getAuth()->is_allowed_to(\isys_auth::EDIT, 'visualization'))
            ->assign('assets_dir', CablingModule::getPath() . 'assets/')
            ->assign('assets_www_dir', CablingModule::getWwwPath() . 'assets/')
            ->assign('saveVisualizationPreferencesUrl', $routeGenerator->generate('cabling.save-visualization-preferences'))
            ->assign('visualizationPreferences', $preferences)
            ->smarty_tom_add_rules('tom.left', $leftSmartyRules)
            ->smarty_tom_add_rules('tom.content.top', $topSmartyRules)
            ->smarty_tom_add_rules('tom.content.bottom.content', $contentSmartyRules);

        return $this;
    }
}
