<?php

namespace idoit\Module\Document\Controller;

use Exception;
use FilesystemIterator;
use isys_application;
use isys_component_template;
use isys_component_template_language_manager;
use isys_exception_filesystem;
use isys_exception_general;
use isys_helper_upload;
use isys_module_document;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use SplFileInfo;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Generator\UrlGenerator;
use Throwable;

/**
 * i-doit 'documents wysiwyg' controller
 *
 * @package   Modules
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class WysiwygController
{
    private isys_component_template_language_manager $language;
    private UrlGenerator $routeGenerator;
    private string $targetPath;
    private string $targetWwwPath;
    private isys_component_template $template;

    public function __construct()
    {
        $this->language = isys_application::instance()->container->get('language');
        $this->routeGenerator = isys_application::instance()->container->get('route_generator');
        $this->template = isys_application::instance()->container->get('template');

        $tenantId = (int)isys_application::instance()->container->get('session')->get_mandator_id();

        $this->targetPath = str_replace('\\', '/', isys_module_document::getPath() . "resources/upload/{$tenantId}/");
        $this->targetWwwPath = str_replace('\\', '/', isys_module_document::getWwwPath() . "resources/upload/{$tenantId}/");
    }

    /**
     * @param Request $request
     *
     * @return Response
     */
    public function uploadImage(Request $request): Response
    {
        try {
            $filesystem = new Filesystem();

            /** @var UploadedFile $upload */
            $upload = $request->files->get('upload');

            if ($upload->getError() !== UPLOAD_ERR_OK) {
                throw new Exception($upload->getErrorMessage());
            }

            $fileExtension = $upload->getClientOriginalExtension();
            $allowedExtensions = ['jpeg', 'jpg', 'png', 'gif', 'bmp', 'svg'];

            if (strpos($upload->getClientMimeType(), 'image/') !== 0 || !in_array($fileExtension, $allowedExtensions, true)) {
                $errorMessage = $this->language->get('LC__UNIVERSAL__FILE_UPLOAD__EXTENSION_ERROR');
                $allowed = implode(', ', $allowedExtensions);

                throw new isys_exception_general("{$errorMessage} ({$allowed})");
            }

            $fileName = $upload->getClientOriginalName();

            $hashPrefix = substr(md5($fileName . microtime(true)), 0, 8);
            $directoryHash = substr($hashPrefix, 0, 2);
            $cleanFileName = isys_helper_upload::prepare_filename(substr($fileName, 0, -(strlen($fileExtension) + 1)));

            $targetFileName = "{$hashPrefix}_{$cleanFileName}.{$fileExtension}";

            $targetFilePath = "{$this->targetPath}/{$directoryHash}/";
            $targetWwwPath = "{$this->targetWwwPath}/{$directoryHash}/";

            // Create the directory.
            $filesystem->mkdir($targetFilePath, 0775);
            // Move the uploaded file.
            $filesystem->rename($upload->getPathname(), $targetFilePath . $targetFileName);

            $imageUrl = htmlentities($targetWwwPath . $targetFileName);
            $message = '';

            $responseData = [
                "uploaded" => true,
                "filename" => $fileName,
                "url" => $imageUrl
            ];

        } catch (Throwable $e) {
            $imageUrl = '';
            $message = htmlentities($e->getMessage());

            $responseData = [
                "uploaded" => false,
                "error" => $message
            ];
        }

        return new JsonResponse($responseData);
    }

    /**
     * @param Request $request
     *
     * @return Response
     * @throws \SmartyException
     */
    public function browseImages(Request $request): Response
    {
        $images = [];
        $message = '';

        // Only iterate the directory if it exists.
        if (file_exists($this->targetPath)) {
            $directoryIterator = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($this->targetPath, FilesystemIterator::SKIP_DOTS));

            foreach ($directoryIterator as $file) {
                /** @var SplFileInfo $file */
                $images[] = str_replace($this->targetPath, $this->targetWwwPath, str_replace('\\', '/', $file->getPathname()));
            }
        }

        // For legacy reasons, we also need to check the main directory:
        $legacyFiles = glob(isys_module_document::getPath() . 'resources/upload/*');

        foreach ($legacyFiles as $file) {
            if (is_dir($file)) {
                continue;
            }

            $images[] = str_replace(
                str_replace('\\', '/', isys_module_document::getPath()),
                str_replace('\\', '/', isys_module_document::getWwwPath()),
                str_replace('\\', '/', $file)
            );
        }

        if (count($images) === 0) {
            $message = $this->language->get('LC__UNIVERSAL__FILE__NOT_FOUND');
        }

        $responseContent = $this->template
            ->assign('title', isys_application::instance()->container->get('language')->get('LC_UNIVERSAL__FILE_BROWSER'))
            ->assign('files', $images)
            ->assign('file_body', 'popup/ckeditor_filebrowser.tpl')
            ->assign('ckeditor_func_num', (int)$request->query->get('CKEditorFuncNum'))
            ->assign('message', $message)
            ->assign('delete_url', $this->routeGenerator->generate('document.wysiwyg.delete-image'))
            ->fetch('popup/main.tpl');

        return new Response($responseContent);
    }

    /**
     * @param Request $request
     *
     * @return Response
     */
    public function deleteImage(Request $request): Response
    {
        try {
            $baseUploadPath = str_replace('\\', '/', isys_module_document::getPath() . 'resources/upload/');

            $filesystem = new Filesystem();
            $file = $request->request->get('file');

            $fileName = basename($file);
            $dirName = basename(dirname($file));

            $fileExistsInBase = file_exists($baseUploadPath . $fileName);
            $fileExistsInSubDirectory = file_exists($this->targetPath . $dirName . '/' . $fileName);

            // Check if the file is located in the correct directory.
            if (!$fileExistsInBase && !$fileExistsInSubDirectory) {
                throw new isys_exception_filesystem($this->language->get('LC_FILEBROWSER__NO_FILE_FOUND'));
            }

            if ($fileExistsInBase) {
                $filesystem->remove($baseUploadPath . $fileName);
            }

            if ($fileExistsInSubDirectory) {
                $filesystem->remove($this->targetPath . $dirName . '/' . $fileName);
            }

            $result = [
                'success' => true,
                'data' => null,
                'message' => ''
            ];
        } catch (Throwable $e) {
            $result = [
                'success' => false,
                'data' => null,
                'message' => $e->getMessage()
            ];
        }

        return new JsonResponse($result);
    }

    /**
     * @param Request $request
     *
     * @return Response
     * @see DOKU-457 We implement this in order to show the "custom css" while editing templates.
     */
    public function getCustomCss(Request $request): Response
    {
        $emptyResult = new Response('/* no custom CSS */', 200, ['Content-Type' => 'text/css']);

        try {
            $documentId = (int)$request->query->get(C__DOCUMENT__GET__DOCUMENT_ID);

            if (!$documentId) {
                return $emptyResult;
            }

            $options = \isys_document_dao_templates::instance(isys_application::instance()->container->get('database'))
                ->get_options($documentId);

            if (!isset($options['pdf.css']) || empty(trim($options['pdf.css']))) {
                return $emptyResult;
            }

            return new Response($options['pdf.css'], 200, ['Content-Type' => 'text/css']);
        } catch (Throwable $e) {
        }

        return $emptyResult;
    }
}
