<?php

/**
 * i-doit
 *
 * SuperClass for our compiler.
 *
 * @package     i-doit
 * @subpackage  Modules
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
abstract class isys_document_compiler
{
    /**
     * Master-Object used by the placeholder.
     *
     * @var  integer
     */
    protected $m_master_object;

    /**
     * Global master object for the whole class parts. This is used if the class member var m_master_object is not set.
     *
     * @var  integer
     */
    protected static $m_global_master_object;

    /**
     * Compiled text.
     *
     * @var  string
     */
    protected $m_compiled_text;

    /**
     * Allready compiled?
     *
     * @var  boolean
     */
    protected $m_compiled = false;

    /**
     * The compiler model.
     *
     * @var  isys_document_compiler_model
     */
    protected $m_model;

    /**
     * Holder for parent unit. In case of chapter it should be the document object.
     *
     * @var  isys_document_compiler_document
     */
    protected $m_document;

    /**
     * Construct.
     *
     * @param  isys_document_compiler_model $p_model
     */
    final public function __construct(isys_document_compiler_model $p_model)
    {
        $this->set_model($p_model);
    }

    /**
     * Returns the masterobjectID after handling.
     *
     * @return  integer
     */
    public function get_master_object()
    {
        if (isset($this->m_master_object)) {
            return $this->m_master_object;
        }

        return static::$m_global_master_object;
    }

    /**
     * Set masterobjectID.
     *
     * @param   integer $p_value
     *
     * @return  isys_document_compiler
     */
    public function set_master_object($p_value)
    {
        $this->m_master_object = $p_value;

        return $this;
    }

    /**
     * Get document.
     *
     * @return  isys_document_compiler
     */
    public function get_document()
    {
        return $this->m_document;
    }

    /**
     * Set document model
     *
     * @param isys_document_compiler $p_value
     *
     * @return $this
     */
    public function set_document(isys_document_compiler $p_value)
    {
        $this->m_document = $p_value;

        return $this;
    }

    /**
     * Get model
     *
     * @return isys_document_compiler_model_chapter
     */
    public function get_model()
    {
        return $this->m_model;
    }

    /**
     * Set model
     *
     * @param isys_document_compiler_model $p_value
     *
     * @return $this
     */
    public function set_model(isys_document_compiler_model $p_value)
    {
        $this->m_model = $p_value;

        return $this;
    }

    /**
     * Is allready compiled?
     *
     * @return bool
     */
    public function get_compiled()
    {
        return $this->m_compiled;
    }

    /**
     * Marks a compiler as compiled
     *
     * @param bool $p_value
     *
     * @return \isys_document_compiler
     */
    public function set_compiled($p_value)
    {
        $this->m_compiled = $p_value;

        return $this;
    }

    /**
     * Get compiled text
     *
     * @return string
     */
    public function get_compiled_text()
    {
        if (!$this->get_compiled()) {
            $this->compile();
        }

        return $this->m_compiled_text;
    }

    /**
     * Set compiled text manually
     *
     * @param string $p_value
     *
     * @return \isys_document_compiler
     */
    public function set_compiled_text($p_value = null)
    {
        $this->m_compiled_text = $p_value;

        return $this;
    }

    /**
     * Set an global master object id for all instances of called class
     *
     * @param int $p_master_object_id
     */
    public static function set_global_master_object($p_master_object_id)
    {
        static::$m_global_master_object = $p_master_object_id;
    }

    /**
     * Output compiled text on 'echo'
     *
     * @return string
     */
    public function __toString()
    {
        return $this->get_compiled_text();
    }

    /**
     * Compiles the raw text
     *
     * @return \isys_document_compiler
     */
    public final function compile()
    {
        $this->pre();
        $this->get();
        $this->set_compiled(true);
        $this->post();

        return $this;
    }

    /**
     * Compile some html
     *
     * @param string                         $p_text
     * @param isys_document_compiler_chapter $p_compiler_chapter
     *
     * @return string
     */
    protected function compile_html($p_text, isys_document_compiler_chapter $p_compiler_chapter)
    {
        if (!empty($p_text)) {
            $l_domDocument = new DOMDocument();
            // @see  DOKU-184  We need to make sure, DOMDocument understands we are using UTF8 here.
            $l_domDocument->loadHTML('<meta http-equiv="Content-Type" content="text/html; charset=utf-8">' . $p_text);
            $l_finder = new DOMXPath($l_domDocument);

            $nodes = $l_finder->query("//*[@class='document-wysiwyg-placeholder']");
            $i = $nodes->length - 1;

            while ($i > -1) {
                $node = $nodes->item($i);
                $l_data = json_decode(str_replace("'", '"', $node->getAttribute('data-json')), true);

                $l_placeholder_class = isys_document_compiler_placeholder::factory($l_data['type'], $l_data, $p_compiler_chapter);

                // Decode to prevent XML errors
                $l_return = $this->format($l_placeholder_class->compile());

                $l_replacer = $l_domDocument->createDocumentFragment();

                $l_replacer->appendXML($l_return);

                $l_root = $node->parentNode;

                $l_root->replaceChild($l_replacer, $node);

                $i--;
            }

            return ($this->replace($l_domDocument->saveHTML()));
        }

        return '';
    }

    /**
     * Format string to valid XML.
     *
     * @param   string $p_string
     *
     * @return  string
     */
    protected function format($p_string)
    {
        // Because of various XML methods, we need to "mask" everything that looks like a tag: "<demo>".
        $p_string = str_replace(['&lt;', '&gt;'], ['&amp;lt;', '&amp;gt;'], $p_string);

        // We now decode HTML entities for umlaute like "&ouml;".
        $p_string = html_entity_decode($p_string, ENT_COMPAT);

        // @see  DOKU-295  At some point we might get "invalid" XML (tags that don't close etc.).
        $p_string = str_replace('<br>', '<br />', $p_string);

        // Finally we replace all "left over" ampersands, because we later on work with XML.
        return preg_replace('/&(?!(?:apos|quot|[gl]t|amp);|#)/', '&amp;', $p_string);
    }

    /**
     * Replace unwanted <html> and <body> tags
     *
     * @param string $p_html
     *
     * @return string
     */
    protected function replace($p_html)
    {
        // LF: This has been done with regex before, but it did not work on PHP 5.5, so we changed it.
        return str_replace(['</body>', '</html>'], '', substr($p_html, (strpos($p_html, '<body>') + 6)));
    }

    /**
     * Pre-Compile routine
     */
    protected abstract function pre();

    /**
     * Compile routine
     */
    protected abstract function get();

    /**
     * Post-Compile routine
     */
    protected abstract function post();

    /**
     * Initializer
     */
    protected abstract function init();
}
