<?php

use Masterminds\HTML5;

/**
 * i-doit
 *
 * Chapter compiler.
 *
 * @package     i-doit
 * @subpackage  Modules
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class isys_document_compiler_chapter extends isys_document_compiler
{
    private const HTML5_SNIPPET = '<!DOCTYPE html><meta http-equiv="Content-Type" content="text/html; charset=utf-8">';

    private DOMDocument|null $m_dom_document = null;

    /**
     * Transform rawtext to DOMDocument.
     */
    public function load_dom()
    {
        $html = $this->get_model()->get_raw_text();

        try {
            $this->m_dom_document = (new HTML5)->loadHTML(self::HTML5_SNIPPET . $html);
        } catch (Exception $e) {
            // @see  DOKU-72  Catching possible parse errors.
            isys_application::instance()->container->get('logger')
                ->error('There is a syntax error in your HTML Chapter named "' . $this->get_model()
                        ->get_title() . '" (' . $e->getMessage() . ')', ['html' => $html]);
        }

        return true;
    }

    /**
     * Compile chapters and link them together
     *
     * @return $this
     * @throws isys_exception_general
     */
    protected function get()
    {
        if ($this->init()) {
            // At this point we want to refresh the existing "ref key" cache!
            isys_document_compiler_placeholder_property::reset_ref_keys();

            $l_finder = new DOMXPath($this->m_dom_document);

            $nodes = $l_finder->query("//*[contains(@class,'document-wysiwyg-placeholder')]");
            $i = $nodes->length - 1;

            while ($i > -1) {
                $node = $nodes->item($i);

                $l_data = json_decode(str_replace("'", '"', $node->getAttribute('data-json')), true);

                $l_placeholder_class = isys_document_compiler_placeholder::factory($l_data['type'], $l_data, $this);

                // Decode to prevent XML errors
                $compiledText = (string)$this->format($l_placeholder_class->compile());

                $domNode = (new HTML5)->loadHTMLFragment(self::HTML5_SNIPPET . $compiledText);

                $parentNode = $node->parentNode;

                // It might be necessary to import the new node.
                if ($domNode->ownerDocument !== $parentNode->ownerDocument) {
                    $importedNode = $this->m_dom_document->importNode($domNode, true);
                } else {
                    $importedNode = $domNode;
                }

                // @see DOKU-383 Block elements can not be placed inside paragraphs.
                if ($parentNode->nodeName === 'p' && $this->isBlockElement($compiledText)) {
                    try {
                        // We replace the paragraph node itself with the content, not the CONTENT of the paragraph.
                        $parentNode->parentNode->replaceChild($importedNode, $parentNode);
                    } catch (Throwable $e) {
                        // In case something goes wrong we use the old logic.
                        $parentNode->replaceChild($importedNode, $node);
                    }
                } else {
                    $parentNode->replaceChild($importedNode, $node);
                }

                $i--;
            }

            // Insert chapter title.
            $this->get_model()->set_compiled_text($this->m_dom_document->saveHTML());
            $this->set_compiled(true);
        }

        return $this;
    }

    /**
     * The WYSIWYG content will be replaced into a paragraph, which is wrong if the content is a block element.
     * So we check if the content IS a block element to not inject it into the P but replace the P.
     *
     * @param string $text
     *
     * @return bool
     * @see DOKU-383
     */
    private function isBlockElement(string $text): bool
    {
        $trimmedText = trim($text);

        return str_starts_with($trimmedText, '<table') || str_starts_with($trimmedText, '<p');
    }

    /**
     * Post-Compile.
     */
    protected function post()
    {
    }

    /**
     * Pre-Compile.
     */
    protected function pre()
    {
    }

    /**
     * Init
     */
    protected function init()
    {
        return $this->load_dom();
    }
}
