<?php

use idoit\Module\Document\Compiler\Table\Config;
use idoit\Module\Document\Compiler\Table\Table;

/**
 * i-doit
 *
 * Exports a selected document to HTML.
 *
 * @package      i-doit
 * @subpackage   Modules
 * @author       Selcuk Kekec <skekec@i-doit.com>
 * @version      1.0.0
 * @copyright    synetics GmbH
 * @license      http://www.i-doit.com/license
 * @since        i-doit 1.4.0
 */
class isys_document_export_html extends isys_document_export
{
    /**
     * Exportable text.
     *
     * @var  string
     */
    protected $documentContent;

    /**
     * Export the HTML.
     *
     * @throws  Exception
     * @return  isys_document_export_html
     */
    public function export()
    {
        // Initialize the HTML formatter.
        $this->m_formatter->initialize($this->m_options);

        $this->prepareStylesheet();

        // Build table of contents.
        $toc = '';
        $tableBody = [];
        $renderTableOfContents = isset($this->m_options['general.toc']) && $this->m_options['general.toc'];

        /* @var  $l_model  isys_document_export_model */
        foreach ($this->m_models as $l_model) {
            $tableBody[] = [$l_model->get_pos() . ' ' . $l_model->get_title()];

            // @see DOKU-377 Only show headlines if the option is active.
            if ($this->m_options['general.headlines']) {
                $this->m_formatter->addHeadline($l_model->get_title(), $l_model->get_pos(), $l_model->get_level());
            }

            $this->m_formatter->addContent($l_model->get_text(), $l_model->get_pos(), $l_model->get_level());
        }

        // Should we build table of content
        if ($renderTableOfContents) {
            $language = isys_application::instance()->container->get('language');

            $tableHeader = [$language->get('LC__MODULE__DOCUMENT__TABLE_OF_CONTENTS')];

            $config = (new Config())
                ->setShowHeader(true)
                ->setGrid(Config::GRID_NONE);

            $toc = (new Table($tableHeader, $tableBody, $config))->compile();
        }

        $template = isys_application::instance()->container->get('template')
            ->assign('documentTitle', $this->get_title())
            ->assign('style', $this->get_style())
            ->assign('tableOfContents', $toc)
            ->assign('header', $this->m_formatter->getHeader())
            ->assign('body', $this->m_formatter->getContent())
            ->assign('footer', $this->m_formatter->getFooter());

        $this->documentContent = $template->fetch(isys_module_document::getPath() . 'templates/export-base.tpl');

        return $this;
    }

    /**
     * @param             $p_filename
     *
     * @param null|string $filePath
     *
     * @return $this|isys_document_export
     */
    public function write($p_filename, $filePath = null)
    {
        if ($filePath !== null) {
            $file = rtrim($filePath, '/') . '/' . $p_filename . '.html';
        } else {
            $file = $this->get_filepath($p_filename) . '.html';
        }

        file_put_contents($file, $this->documentContent);

        return $this;
    }

    /**
     * @param string $filename
     *
     * @return self
     * @throws Exception
     */
    public function send(string $filename): self
    {
        if (!headers_sent()) {
            header('Content-disposition: attachment; filename="' . $filename . '.html"');
            header('Content-Type: application/download');
            echo $this->documentContent;
            die;
        }

        throw new Exception('Error sending HTML. There was an unknown export problem.');
    }

    /**
     * Mimetype
     *
     * @return string
     */
    public function get_mimetype(): string
    {
        return 'application/html';
    }

    /**
     * File extension.
     *
     * @return string
     */
    public function get_file_extension(): string
    {
        return 'html';
    }

    /**
     * Initialize
     */
    protected function init()
    {
        // Setting some default options.
        $this->m_options = [
            'general.toc'       => 1,
            'general.headlines' => 1,
        ];

        // Setting default styles.
        $this->prepareStylesheet();
    }

    /**
     * @return void
     */
    private function prepareStylesheet()
    {
        // @see  DOKU-304 Preparing some default styles.
        $style = 'p { margin: 0 0 10px 0; }
            table { width: 100%; border-spacing: 0; border-collapse: collapse; }
            code { font-family: courier; }
            ins { text-decoration: underline; }
            blockquote { padding: 5px 20px; margin-left: 20px; border-left: 3px solid #ccc; }
            .marker { background-color: #ff0; }
            .text-monospace, .font-monospace { font-family: Courier New, Lucida Console, Monospace, Monaco, sans-serif, serif; } ';

        // @see  DOKU-308  Pass the defined headline-sizes.
        foreach ($this->m_options as $key => $value) {
            if (preg_match('~general\.h([1-6])\.size~', $key, $matches)) {
                $style .= 'h' . $matches[1] . ' { font-size: ' . $value . 'pt; } ';
            }
        }

        $this->set_style($style . $this->m_options['pdf.css']);
    }
}
