<?php

namespace idoit\Module\Document\Compiler\Helper;

use Exception;
use isys_application;
use isys_cmdb_dao_category_g_images;
use Symfony\Component\DomCrawler\Crawler;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Throwable;

/**
 * i-doit "Image" compiler helper.
 *
 * @package   document
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Image
{
    /**
     * @param string $content
     *
     * @return string
     * @throws Exception
     */
    public function replaceLocalImageSources(string $content): string
    {
        try {
            $crawler = new Crawler($content);

            // Find images with 'src' attribute.
            $crawler->filter('img[src]')->each(function (Crawler $node) {
                // Try to fetch the image data via URL.
                [$mimeType, $imageData] = $this->convertUrlContentToBase64($node->attr('src'));

                if ($imageData === null) {
                    return;
                }

                // Replace 'src' attribute with base64 data.
                $node->getNode(0)->setAttribute('src', "data:{$mimeType};base64,{$imageData}");
            });

            // Fetch the replaced HTML content.
            return $crawler->filter('body')->html();
        } catch (Throwable $e) {
            // Do nothing, just keep the original value.
            isys_application::instance()->container->get('notify')->warning($e->getMessage(), ['life' => 10]);
        }

        return $content;
    }

    /**
     * Try to convert linked images into base64 data.
     *
     * @param string $url
     *
     * @return array
     * @throws Exception
     * @see DOKU-475
     */
    private function convertUrlContentToBase64(string $url): array
    {
        try {
            // First we'll check if we have a symfony route for the given URL.
            $response = isys_application::instance()->container->get('kernel')->handle(Request::create($url));

            return $this->processSymfonyResponse($response);
        } catch (NotFoundHttpException $e) {
            // Then we'll parse and check the URL by hand.
            $parsedUrl = parse_url($url);
            parse_str($parsedUrl['query'], $queryParams);

            // First we verify that the host is the same, or not set at all (in case of '?xyz').
            if ($_SERVER['HTTP_HOST'] === $parsedUrl['host'] || !isset($parsedUrl['host'])) {
                $isImagesCategoryUrl = isset($queryParams[C__CMDB__GET__OBJECT]) && $queryParams[C__CMDB__GET__OBJECT] > 0
                    && isset($queryParams[C__CMDB__GET__CATG]) && $queryParams[C__CMDB__GET__CATG] == C__CATG__IMAGES
                    && isset($queryParams[C__GET__FILE__ID]) && $queryParams[C__GET__FILE__ID] > 0;

                if ($isImagesCategoryUrl) {
                    return $this->processImageCategory((int)$queryParams[C__CMDB__GET__OBJECT], (int)$queryParams[C__GET__FILE__ID]);
                }
            }
        }

        return [null, null];
    }

    /**
     * @param Response $response
     *
     * @return array
     */
    private function processSymfonyResponse(Response $response): array
    {
        try {
            if ($response instanceof BinaryFileResponse) {
                return [
                    $response->headers->get('content-type', $response->getFile()->getMimeType()),
                    base64_encode($response->getFile()->getContent())
                ];
            }

            return [
                $response->headers->get('content-type', 'image/*'),
                base64_encode($response->getContent())
            ];
        } catch (Throwable $e) {
            isys_application::instance()->container->get('notify')->warning($e->getMessage(), ['life' => 10]);
        }

        return [null, null];
    }

    /**
     * @param int $objectId
     * @param int $fileId
     *
     * @return array
     * @throws Exception
     */
    private function processImageCategory(int $objectId, int $fileId): array
    {
        // Fetch the image data from 'Images' category.
        $imageData = isys_cmdb_dao_category_g_images::instance(isys_application::instance()->container->get('database'))
            ->get_data($fileId, $objectId)
            ->get_row();

        if (empty($imageData)) {
            return [null, null];
        }

        return [
            $imageData['isys_catg_images_list__filemime'],
            $imageData['isys_catg_images_list__filecontent']
        ];
    }
}
