<?php

namespace idoit\Module\Document\Console\Command;

use Exception;
use idoit\Console\Command\AbstractCommand;
use isys_document_compiler_document;
use isys_document_dao;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputDefinition;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class CompileDocumentsCommand extends AbstractCommand
{
    const NAME = 'compile-documents';

    /**
     * Get name for command
     *
     * @return string
     */
    public function getCommandName()
    {
        return self::NAME;
    }

    /**
     * Get description for command
     *
     * @return string
     */
    public function getCommandDescription()
    {
        return 'Allows to compile one or multiple documents';
    }

    /**
     * @return void
     */
    protected function configure()
    {
        parent::configure();

        $this->setAliases(['CompileDocuments']);
    }

    /**
     * Retrieve Command InputDefinition
     *
     * @return InputDefinition
     */
    public function getCommandDefinition()
    {
        return new InputDefinition([
            new InputOption(
                'documentId',
                't',
                InputOption::VALUE_REQUIRED | InputOption::VALUE_IS_ARRAY,
                'Several document ids'
            )
        ]);
    }

    /**
     * Checks if a command can have a config file via --config
     *
     * @return bool
     */
    public function isConfigurable()
    {
        return false;
    }

    /**
     * Returns an array of command usages
     *
     * @return string[]
     */
    public function getCommandUsages()
    {
        return [];
    }

    /**
     * @param InputInterface  $input
     * @param OutputInterface $output
     *
     * @return int
     * @throws \isys_exception_database
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $errorCount = 0;
        // @see DOKU-414 Cast the option correctly!
        $documentIds = (array)$input->getOption('documentId');
        $documentIdsCount = count($documentIds);

        if ($documentIdsCount === 0) {
            $output->writeln('You need to pass at least one document via <info>-t</info> or <info>--documentId</info> option.');
            return Command::SUCCESS;
        }

        if (!is_numeric($documentIds[0])) {
            $output->writeln([
                'You can only pass singular document IDs to the <info>-t</info> or <info>--documentId</info> option.',
                'If you want to compile multiple documents, please do it like so <comment>-t 1 -t 2 -t 3</comment>.',
            ]);
            return Command::FAILURE;
        }

        $l_document_dao = new isys_document_dao($this->container->get('database'));
        $output->writeln("Prepare update process for <infor>{$documentIdsCount} documents</infor>");

        foreach ($documentIds as $documentId) {
            $output->writeln('');
            $documentResult = $l_document_dao->get_data($documentId);

            if ($documentResult->count()) {
                $documentTitle = $documentResult->get_row_value('isys_document_template__title');

                $output->writeln("Document with <info>ID {$documentId}</info> found: <comment>'{$documentTitle}'</comment>");
                $output->writeln(" > Refresh documents content...");

                try {
                    isys_document_compiler_document::refresh_document($documentId);

                    $output->writeln(" > Document refreshed <info>successfully</info>.");
                } catch (Exception $e) {
                    $errorCount++;
                    $output->writeln(" > Error: " . $e->getMessage());
                }
            } else {
                $errorCount++;
                $output->writeln("<error>Given document ID {$documentId} does not exist. Skipped!</error>");
            }
        }

        $output->writeln([
            '',
            "Job finished with <info>{$errorCount} error(s)</info>."
        ]);

        return Command::SUCCESS;
    }
}
