<?php

/**
 * i-doit
 *
 * Document chapter DAO.
 *
 * @package     i-doit
 * @subpackage  Modules
 * @author      Selcuk Kekec <skekec@i-doit.com>
 * @version     1.0.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.3.0
 */
class isys_document_dao_chapters extends isys_component_dao
{

    /**
     * Used by our tree methods
     * @var array
     */
    private $m_level = array();

    /**
     * Current level holder
     * @var int
     */
    private $m_current_level = -1;

    /**
     * Single level representation of the retrieved chapter tree
     * @var array
     */
    private $m_single_level = array();

	/**
	 * Last chapter id
	 * @var null|int
	 */
	private static $m_last_chapter_id = null;


    /**
	 * Setter for last chapter id
	 * @param $p_id
	 */
	public static function set_last_chapter_id ($p_id)
	{
		self::$m_last_chapter_id = $p_id;
	} // function


	/**
	 * Getter for last chapter id
	 * @return int|null
	 */
	public static function get_last_chapter_id ()
	{
		return self::$m_last_chapter_id;
	} // function


    /**
     * Method for retrieving a chapter.
     *
     * @param   integer  $p_id
     * @param   integer  $p_documentID
     * @param   integer  $p_condition
     * @return  isys_component_dao_result
     * @author  Selcuk Kekec <skekec@i-doit.com>
     */
	public function get_data ($p_id = null, $p_documentID = null, $p_condition = null)
	{
		$l_sql = 'SELECT MAIN.*, COMPONENT.*,
                IF (COMPONENT.isys_document_component__isys_document_component__id IS NULL, COMPONENT.isys_document_component__text, COMPONENTPARENT.isys_document_component__text) AS text
                FROM isys_document_chapter AS MAIN
                INNER JOIN isys_document_template ON MAIN.isys_document_chapter__isys_document_template__id = isys_document_template__id
                LEFT JOIN isys_document_component COMPONENT ON MAIN.isys_document_chapter__isys_document_component__id = COMPONENT.isys_document_component__id
                LEFT JOIN isys_document_chapter PARENT ON PARENT.isys_document_chapter__id = MAIN.isys_document_chapter__isys_document_chapter__id
                LEFT JOIN isys_document_component COMPONENTPARENT ON COMPONENTPARENT.isys_document_component__id = COMPONENT.isys_document_component__isys_document_component__id
                WHERE TRUE ';

		if ($p_id !== null)
		{
			if (! is_array($p_id))
			{
				$p_id = array($this->convert_sql_id($p_id));
			} // if

			if (count($p_id))
			{
				$l_sql .= ' AND MAIN.isys_document_chapter__id IN (' . implode(', ', array_map('intval', $p_id)) . ')';
			} // if
		} // if

		if ($p_documentID !== null)
		{
			if (! is_array($p_documentID))
			{
				$p_documentID = array($this->convert_sql_id($p_documentID));
			} // if

			if (count($p_documentID))
			{
				$l_sql .= ' AND MAIN.isys_document_chapter__isys_document_template__id IN (' . implode(', ', array_map('intval', $p_documentID)) . ')';
			} // if
		} // if

		if ($p_condition)
		{
			$l_sql .= ' ' . $p_condition . ' ';
		}

		$l_sql .= ' ORDER BY MAIN.isys_document_chapter__isys_document_chapter__id ASC, isys_document_chapter__sort ASC';

		return $this->retrieve($l_sql . ';');
	} // function


    /**
     * @param $p_chapter_id
     *
     * @return isys_component_dao_result
     * @throws Exception
     * @throws isys_exception_database
     */
    public function get_parent_chapter($p_chapter_id) {
        $l_sql = "SELECT * FROM isys_document_component " .
                "LEFT JOIN isys_document_chapter ON isys_document_component__id = isys_document_chapter__isys_document_component__id " .
                "WHERE isys_document_component__id = " .
                "(SELECT isys_document_chapter__isys_document_component__id FROM isys_document_chapter WHERE isys_document_chapter__id = " . $this->convert_sql_id($p_chapter_id) . ");";

        return $this->retrieve($l_sql);
    }

    /**
     * Method for creating/saving a chapter.
     *
     * @param  integer $p_id
     * @param  array   $p_values
     *
     * @return integer
     * @throws isys_exception_dao
     * @author Selcuk Kekec <skekec@i-doit.com>
     */
    public function saveChapter($p_id = null, array $p_values = [])
    {
        $l_data = [];

        if ($p_id === null || empty($p_id)) {
            $l_sql = 'INSERT INTO isys_document_chapter SET %s;';
            $l_document_component = new isys_document_dao_components($this->get_database_component());

            $l_document_component->saveComponent(null, [
                'text' => '',
            ]);

            $p_values['isys_document_component__id'] = $l_document_component->get_last_insert_id();
            isys_document_dao_components::set_last_component_id($p_values['isys_document_component__id']);

            if (!$p_values['sort']) {
                $p_values['sort'] = $this->get_free_position($p_values['isys_document_chapter__id'], $p_values['isys_document_template__id']);
            }
        } else {
            $l_sql = 'UPDATE isys_document_chapter SET %s WHERE isys_document_chapter__id = ' . $this->convert_sql_id($p_id) . ';';
        }

        if (count($p_values) > 0) {
            // Handling for changed chapter id.
            if ($p_id !== null || !empty($p_id)) {
                // We try to update a existing chapter.
                $l_chapter_data = $this->get_data($p_id)->get_row();

                // Do we have a changed chapter id?.
                $l_parent_chapter_id = ($p_values['isys_document_chapter__id'] == 'NULL') ? null : $p_values['isys_document_chapter__id'];

                if (isset($l_parent_chapter_id) && ($l_parent_chapter_id != $l_chapter_data['isys_document_chapter__isys_document_chapter__id'])) {
                    // Let us get a free position in new chapter context.
                    $p_values['sort'] = $this->get_free_position($p_values['isys_document_chapter__id'], $p_values['isys_document_template__id']);
                }
            }

            foreach ($p_values as $l_key => $l_value) {
                if ($l_value || is_numeric($l_value)) {
                    $l_data[] = 'isys_document_chapter__' . $l_key . ' = ' .
                        (strpos($l_key, '__id') !== false ? $this->convert_sql_id($l_value) : $this->convert_sql_text($l_value));
                }
            }

            // @see  DOKU-245  Set "updated" when saving a chapter.
            if (isset($p_values['isys_document_template__id']) && $p_values['isys_document_template__id']) {
                isys_document_dao_templates::instance($this->m_db)
                    ->saveTemplate($p_values['isys_document_template__id']);
            }

            $this->update(str_replace('%s', implode(', ', $l_data), $l_sql)) && $this->apply_update();
        }

        return (self::$m_last_chapter_id = ($p_id ?: $this->get_last_insert_id()));
    }

    /**
     * Get next free positon for adding a chapter as child.
     *
     * @param   integer  $p_parentChapterID
     * @param   integer  $p_documentID
     * @return  integer
     */
	public function get_free_position ($p_parentChapterID = null, $p_documentID = null)
	{
		if (isset($p_parentChapterID) && is_numeric($p_parentChapterID))
		{
			$l_sql = "SELECT isys_document_chapter__sort
            	FROM isys_document_chapter
            	WHERE isys_document_chapter__isys_document_chapter__id = " . $this->convert_sql_id($p_parentChapterID) . "
            	ORDER BY isys_document_chapter__sort DESC;";
		}
		else if (isset($p_documentID) && is_numeric($p_documentID))
		{
			$l_sql = "SELECT isys_document_chapter__sort
				FROM isys_document_chapter
				WHERE isys_document_chapter__isys_document_template__id = " . $this->convert_sql_id($p_documentID) . "
				ORDER BY isys_document_chapter__sort DESC";
		} // if

		if (isset($l_sql))
		{
			$l_res = $this->retrieve($l_sql);

			if ($l_res->num_rows())
			{
				$l_row = $l_res->get_row();

				return $l_row['isys_document_chapter__sort'] + 1;
			} // if
		} // if

		return 1;
	} // function


	/**
	 * Update chapters title.
	 *
	 * @param   integer  $p_id
	 * @param   string   $p_title
	 * @return  boolean
	 */
	public function update_title ($p_id, $p_title)
	{
		$l_sql = "UPDATE isys_document_chapter SET isys_document_chapter__title = " . $this->convert_sql_text($p_title) . " WHERE isys_document_chapter__id = " . $p_id . ";";

		return $this->update($l_sql) && $this->apply_update();
	} // function


	/**
	 * Update position of chapter.
	 *
	 * @param   integer  $p_id
	 * @param   integer  $p_position
	 * @return  bool
	 */
	public function update_position ($p_id, $p_position = null)
	{
		return $this->saveChapter($p_id, array('sort' => $p_position));
	}


    /**
     * Update parent chapter.
     *
     * @param   integer  $p_id
     * @param   integer  $p_parent_chapter_id
     * @return  boolean
     */
    public function update_parent_chapter($p_id, $p_parent_chapter_id = NULL) {
	    $l_sql = 'UPDATE isys_document_chapter
			SET isys_document_chapter__isys_document_chapter__id = ' . $this->convert_sql_id($p_parent_chapter_id) . '
			WHERE isys_document_chapter__id = ' . $this->convert_sql_id($p_id) . ';';

        return ($this->update($l_sql) && $this->apply_update());
    } // function


    /**
     * Update component of chapter.
     *
     * @param   integer  $p_chapterID
     * @param   string   $p_text
     * @param   integer  $p_parentComponentID Link to component
     * @param   boolean  $p_copy Hardlink or copy?
     * @return  boolean
     */
    public function update_component($p_chapterID, $p_text = null, $p_parentComponentID = null, $p_copy = FALSE) {
        $l_values = array();
        $l_sql = "UPDATE isys_document_component SET ";

        if (isset($p_text)) {
            $l_values[] = 'isys_document_component__text = ' . $this->convert_sql_text($p_text);
            $l_values[] = 'isys_document_component__isys_document_component__id = NULL';
        }

        $p_parentComponentID = intval($p_parentComponentID);
        if ($p_parentComponentID > 0) {
            $l_values = array();
            if ($p_copy) {
                $l_text = $this->retrieve('SELECT isys_document_component__text FROM isys_document_component WHERE isys_document_component__id = ' . $this->convert_sql_id($p_parentComponentID))->get_row_value('isys_document_component__text');
                $l_values[] = 'isys_document_component__isys_document_component__id = NULL';
                $l_values[] = 'isys_document_component__text = ' . $this->convert_sql_text($l_text);
            } else {
                $l_values[] = 'isys_document_component__isys_document_component__id = ' . $this->convert_sql_id($p_parentComponentID);
                $l_values[] = 'isys_document_component__text = NULL';
            }
        }

        if (is_array($l_values) && count($l_values)) {
            $l_sql = $l_sql . implode(',', $l_values) .
                    ' WHERE isys_document_component__id = (SELECT isys_document_chapter__isys_document_component__id FROM isys_document_chapter WHERE isys_document_chapter__id = ' . $this->convert_sql_id($p_chapterID) . ')';

            return ($this->update($l_sql) && $this->apply_update());
        }

        /* Nothing to update */
        return true;
    }

    /**
     * Remove one or multiple document chapters.
     *
     * @param  int|array $ids
     *
     * @return boolean
     * @throws isys_exception_dao
     * @throws isys_exception_database
     * @author Selcuk Kekec <skekec@i-doit.com>
     */
    public function deleteChapter($ids)
    {
        if (!is_array($ids)) {
            $ids = [$ids];
        }

        foreach ($ids as $id) {
            if (!is_numeric($id)) {
                continue;
            }

            $sql = 'SELECT isys_document_chapter__isys_document_template__id 
                FROM isys_document_chapter 
                WHERE isys_document_chapter__id = ' . $this->convert_sql_id($id) . ';';

            $documentTemplateId = $this->retrieve($sql)->get_row_value('isys_document_chapter__isys_document_template__id');

            // @see  DOKU-245  Update the document template, if we delete a chapter.
            if ($documentTemplateId !== null && $documentTemplateId > 0) {
                isys_document_dao_templates::instance($this->m_db)->saveTemplate($documentTemplateId);
            }

            $this->update('DELETE FROM isys_document_chapter WHERE isys_document_chapter__id = ' . $this->convert_sql_id($id) . ';');
            $this->apply_update();
        }

        return true;
    }

    /**
     * Get chapter tree recursive
     *
     * @param   isys_component_dao_result  $p_sqlRsc
     * @param   boolean                    $p_with_text
     * @return  array
     */
	private function get_chapter_tree_rec ($p_sqlRsc = null, $p_with_text = false)
	{
		$this->m_current_level++;

		if ($this->m_current_level == 0)
		{
			$this->m_single_level = $this->m_level = array();
		} // if

		if (!isset($this->m_level[$this->m_current_level]))
		{
			$this->m_level[$this->m_current_level] = 0;
		} // if

		$l_data = array();

		if (count($p_sqlRsc))
		{
			while ($l_row = $p_sqlRsc->get_row())
			{
				$this->m_level[$this->m_current_level]++;
				$l_data[] = array(
					'id' => $l_row['isys_document_chapter__id'],
					'position' => implode(".", array_values($this->m_level)) . ".",
					'level' => count($this->m_level),
					'title' => $l_row['isys_document_chapter__title'],
					'sort' => $l_row['isys_document_chapter__sort'],
					'new_page' => $l_row['isys_document_chapter__new_page']
				);

				if ($p_with_text)
				{
					$l_data[count($l_data) - 1]['text'] = $l_row['text'];
				} // if

				$this->m_single_level[] = $l_data[count($l_data) - 1];

				$l_data[count($l_data) - 1]['children'] = $this->get_chapter_tree_rec($this->get_data(null, null, "AND MAIN.isys_document_chapter__isys_document_chapter__id = " . $l_row['isys_document_chapter__id']), $p_with_text);
			} // while
		} // if

		$this->m_current_level--;

		unset($this->m_level[count($this->m_level) - 1]);

		return $l_data;
	} // function


    /**
     * Get tree recursive as single level array.
     *
     * @param   integer  $p_document_id
     * @param   boolean  $p_with_text
     * @return  array
     */
	public function get_chapter_tree_in_single_level ($p_document_id, $p_with_text = false)
	{
		$this->get_chapter_tree_rec($this->get_data(null, $p_document_id, " AND MAIN.isys_document_chapter__isys_document_chapter__id IS NULL "), $p_with_text);

		return $this->m_single_level;
	} // function


	/**
	 *
	 * @param   isys_component_dao_result  $p_sqlRsc
	 * @global  isys_component_database    $g_comp_database
	 * @return  array
	 */
    public function get_chapter_tree($p_sqlRsc)
    {
        global $g_comp_database;

        $l_dao  = isys_factory::get_instance('isys_document_dao_components', $g_comp_database);
        $l_data = [];

        if ($p_sqlRsc->num_rows())
        {
            while ($l_row = $p_sqlRsc->get_row())
            {
                $l_data[] = [
                    'id'     => $l_row['isys_document_chapter__id'],
                    'title'  => $l_row['isys_document_chapter__title'],
                    'parent' => $l_row['isys_document_chapter__isys_document_chapter__id']
                ];

                if ($l_row['isys_document_component__isys_document_component__id'])
                {
                    $l_parent_chapter = $l_dao->get_data($l_row['isys_document_component__isys_document_component__id'])
                        ->get_row();

                    $l_data[count($l_data) - 1]['component'] = [
                        'chapterID'     => $l_parent_chapter['isys_document_chapter__id'],
                        'chapterTitle'  => $l_parent_chapter['isys_document_chapter__title'],
                        'templateID'    => $l_parent_chapter['isys_document_template__id'],
                        'templateTitle' => $l_parent_chapter['isys_document_template__title']
                    ];
                }
            }
        }

        return $l_data;
    }
}
