<?php

use idoit\Module\Document\CustomFields\Entity\CustomField;
use idoit\Module\Document\CustomFields\Exception\UnknownDataType;
use idoit\Module\Document\CustomFields\Model\Configuration;
use idoit\Module\Document\CustomFields\Model\Document;
use idoit\Module\Document\CustomFields\Repository\CustomFieldsRepository;

/**
 * i-doit
 *
 * Document module class.
 *
 * @package     Module
 * @subpackage  Document
 * @author      Selcuk Kekec <skekec@i-doit.com>
 * @version     1.0.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.3.0
 */
class isys_module_document extends isys_module implements isys_module_interface, isys_module_authable
{
    // Define, if this module shall be displayed in the named menus.
    const DISPLAY_IN_MAIN_MENU   = true;
    const DISPLAY_IN_SYSTEM_MENU = false;

    /**
     * Action indicator constants
     */
    const CL__ACTION__DOCUMENT_TEMPLATES       = 'document_templates';
    const CL__ACTION__CHAPTERS                 = 'chapters';
    const CL__ACTION__DOCUMENTS                = 'documents';
    const CL__ACTION__TEMPLATE_EXPORT_DOWNLOAD = 'template_export_download';
    const CL__ACTION__ONLINE_REPOSITORY        = 'online_repository';
    const CL__ACTION__DOCUMENT_EXPORT          = 'document_export';
    const CL__ACTION__TREE_RELOAD              = 'treeReload';
    const CL__ACTION__TEMPLATE_PREVIEW         = 'template_preview';

    /**
     * @var bool
     */
    protected static $m_licenced = false;

    /**
     * Variable which holds the template component.
     *
     * @var  isys_component_template
     */
    private $template;

    /**
     * Variable which holds the monitoring DAO class.
     *
     * @var  isys_document_dao_templates
     */
    protected $m_dao_document_templates = null;

    /**
     * Variable which holds the database component.
     *
     * @var  isys_component_database
     */
    private $database;

    /**
     * Request
     *
     * @var  isys_module_request
     */
    protected $m_request = null;

    /**
     * Document component DAO.
     *
     * @var  isys_document_dao_components
     */
    protected $m_dao_document_components = null;

    /**
     * Document chapter DAO.
     *
     * @var  isys_document_dao_chapters
     */
    protected $m_dao_document_chapters = null;

    /**
     * Document revision DAO.
     *
     * @var  isys_document_dao_revision
     */
    protected $m_dao_document_revisions = null;

    /**
     * Document DAO
     *
     * @var  isys_document_dao
     */
    protected $m_dao_documents = null;

    /**
     * Tree Counter.
     *
     * @var  integer
     */
    private $m_tree_counter = 0;

    /**
     * Document options.
     *
     * @var  array
     */
    protected $m_formatOptions = [
        'general.toc'        => [
            'title'      => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__EXPORT_GENERATE_TABLE_OF_CONTENT',
            'type'       => 'select',
            'default'    => '1',
            'options'    => [
                '0' => 'LC__UNIVERSAL__NO',
                '1' => 'LC__UNIVERSAL__YES'
            ],
            'p_strStyle' => 'width:70px;'
        ],
        'general.headlines'  => [
            'title'      => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__EXPORT_GENERATE_HEADLINES',
            'type'       => 'select',
            'default'    => '1',
            'options'    => [
                '0' => 'LC__UNIVERSAL__NO',
                '1' => 'LC__UNIVERSAL__YES'
            ],
            'p_strStyle' => 'width:70px;'
        ],
        'general.index-page' => [
            'title'      => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__DISPLAY_INDEX',
            'type'       => 'select',
            'default'    => '1',
            'options'    => [
                '0' => 'LC__UNIVERSAL__NO',
                '1' => 'LC__UNIVERSAL__YES'
            ],
            'p_strStyle' => 'width:70px;'
        ],
        'general.hx.show_numbers'   => [
            'title'       => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__SHOW_NUMBERS_HEADING',
            'type'       => 'select',
            'default'    => '1',
            'options'    => [
                '0' => 'LC__UNIVERSAL__NO',
                '1' => 'LC__UNIVERSAL__YES'
            ],
            'p_strStyle' => 'width:70px;'
        ],
        'general.hx.color'   => [
            'title'       => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__EXPORT_HX_COLOR',
            'type'        => 'color',
            'default'     => '#000000',
            'placeholder' => '#000000',
            'p_strClass'  => 'input-size-mini'
        ],
        'general.hx.size'   => [
            'title'       => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__EXPORT_HX_SIZE',
            'type'        => 'text',
            'p_strClass'  => 'input-size-mini'
        ]
    ];

    /**
     * Retrieves the general format option definition.
     *
     * @return  array
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function getFormatOptionDefinition()
    {
        $defaultHeadlineSizes = [
            1 => 16,
            2 => 14,
            3 => 12,
            4 => 10,
            5 => 8,
            6 => 6
        ];

        for ($i = 1;$i <= 6;$i++) {
            $this->m_formatOptions['general.h' . $i . '.color'] = $this->m_formatOptions['general.hx.color'];
            $this->m_formatOptions['general.h' . $i . '.color']['title'] = $this->language->get($this->m_formatOptions['general.hx.color']['title'], $i);

            $this->m_formatOptions['general.h' . $i . '.size'] = $this->m_formatOptions['general.hx.size'];
            $this->m_formatOptions['general.h' . $i . '.size']['title'] = $this->language->get($this->m_formatOptions['general.hx.size']['title'], $i);
            $this->m_formatOptions['general.h' . $i . '.size']['default'] = $defaultHeadlineSizes[$i];
            $this->m_formatOptions['general.h' . $i . '.size']['placeholder'] = $defaultHeadlineSizes[$i];
        }

        // Unset the "template".
        unset($this->m_formatOptions['general.hx.color'], $this->m_formatOptions['general.hx.size']);

        return [
            'LC__DOCUMENT__COMMON_SETTINGS' => $this->m_formatOptions
        ];
    }

    /**
     * Retrieves the saved options to the given
     *
     * @param   integer $p_document_id
     *
     * @return  array
     * @throws  Exception
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function getFormatOptions($p_document_id = null)
    {
        $l_options = [];

        if ($p_document_id === null || !is_numeric($p_document_id)) {
            foreach ($this->getFormatOptionDefinition() as $l_key => $l_definition) {
                $l_options[$l_key] = $l_definition['default'];
            }
        } else {
            return $this->m_dao_document_templates->get_options($p_document_id);
        }

        return $l_options;
    }

    /**
     * Initializes the module.
     *
     * @param   isys_module_request &$p_req
     *
     * @return  isys_module_monitoring
     * @author  Selcuk Kekec <skekec@i-doit.com>
     */
    public function init(isys_module_request &$p_req)
    {
        $this->m_request = $p_req;
        $this->database = isys_application::instance()->container->get('database');
        $this->template = isys_application::instance()->container->get('template');

        // DAOS
        $this->m_dao_documents = new isys_document_dao($this->database);
        $this->m_dao_document_templates = new isys_document_dao_templates($this->database);
        $this->m_dao_document_chapters = new isys_document_dao_chapters($this->database);
        $this->m_dao_document_components = new isys_document_dao_components($this->database);
        $this->m_dao_document_revisions = new isys_document_dao_revision($this->database);

        return $this;
    }

    /**
     * Retrieve the WYSIWYG configuration.
     *
     * @return string
     */
    private function getWysiwygConfiguration(): string
    {
        if (!method_exists('isys_smarty_plugin_f_wysiwyg', 'get_toolbar_configuration')) {
            return "[
                { name: 'document', groups: [ 'mode', 'document', 'doctools' ], items: [ 'Source' ] },
                { name: 'clipboard', groups: [ 'clipboard', 'undo' ], items: [ 'Cut', 'Copy', 'Paste', 'PasteText', 'PasteFromWord', '-', 'Undo', 'Redo' ] },
                { name: 'editing', groups: [ 'find', 'selection', 'spellchecker' ], items: [ 'Find', 'Replace', '-', 'SelectAll'] },
                { name: 'forms', items: [ 'Form', 'Checkbox', 'Radio', 'TextField', 'Textarea', 'Select', 'Button', 'ImageButton', 'HiddenField' ] },
                { name: 'basicstyles', groups: [ 'basicstyles', 'cleanup' ], items: [ 'Bold', 'Italic', 'Underline', 'Strike', '-', 'RemoveFormat' ] },
                { name: 'paragraph', groups: [ 'list', 'blocks', 'align', 'bidi' ], items: [ 'NumberedList', 'BulletedList', '-', 'Blockquote', 'CreateDiv', '-', 'JustifyLeft', 'JustifyCenter', 'JustifyRight', 'JustifyBlock' ] },
                { name: 'insert', items: [ 'Image', 'Flash', 'Table', 'HorizontalRule', 'Smiley', 'SpecialChar', 'PageBreak', 'Iframe' ] },
                { name: 'idoit', items: [ 'masterObject', 'externalObject', 'report', 'template', 'images', 'floorplan' ] },
                { name: 'idoit-secondary', items: [ 'pageBreak'] },
                { name: 'styles', items: [ 'Styles', 'Format', 'Font', 'FontSize' ] },
                { name: 'colors', items: [ 'TextColor', 'BGColor' ] },
                { name: 'tools', items: [ 'Maximize', 'ShowBlocks' ] },
                { name: 'others', items: [ '-' ] },
                { name: 'about', items: [ 'About' ] }
            ]";
        }

        // We take the "full" WYSIWYG configuration and append the i-doit buttons.
        $toolbarConfig = isys_smarty_plugin_f_wysiwyg::get_toolbar_configuration('full');

        // @see  DOKU-304  We'll remove "indent" and "outdent" for now, because TCPDF can not handle it :(
        foreach ($toolbarConfig as &$config) {
            if ($config['name'] === 'paragraph') {
                $itemList = implode(',', $config['items']);

                // We use this logic to be able to remove all at once (including the '-').
                $itemList = str_replace('Outdent,Indent,-,', '', $itemList);

                $config['items'] = explode(',', $itemList);
            }
        }

        $toolbarConfig[] = ['name' => 'idoit', 'items' => ['masterObject', 'externalObject', 'report', 'template', 'images', 'floorplan']];
        $toolbarConfig[] = ['name' => 'idoit-secondary', 'items' => ['pageBreak']];

        return (string)isys_format_json::encode($toolbarConfig);

    }

    /**
     * Static method for retrieving the path, to the modules templates.
     *
     * @deprecated Use isys_module_document::getPath()
     *
     * @static
     * @return  string
     * @author  Selcuk Kekec <skekec@i-doit.com>
     */
    public static function get_tpl_dir()
    {
        return __DIR__ . '/templates/';
    } // function

    /**
     * Method for retrieving the export directory.
     *
     * @static
     * @return  string
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public static function get_export_dir()
    {
        return __DIR__ . '/resources/export';
    }

    /**
     * Method for retrieving the import directory.
     *
     * @static
     * @return  string
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public static function get_import_dir()
    {
        return __DIR__ . '/resources/import';
    }

    /**
     * Method for retrieving the upload directory.
     *
     * @static
     * @return  string
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public static function get_upload_dir()
    {
        return __DIR__ . '/resources/upload';
    }

    /**
     * Method for retrieving the temp directory.
     *
     * @static
     * @return  string
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public static function get_temp_dir()
    {
        return __DIR__ . '/resources/temp';
    }

    /**
     * Method for retrieving all files inside the upload directory.
     *
     * @static
     * @return  array
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public static function get_upload_dir_files()
    {
        return glob(self::get_upload_dir() . '/*');
    }

    /**
     * Create a treenode.
     *
     * @param   string  $nodeTitle
     * @param   integer $parentNodeId
     * @param   array   $urlParts
     * @param   integer $id
     * @param   string  $nodeIcon
     * @param   boolean $hasRight
     *
     * @return  integer
     */
    private function add_document_node($nodeTitle, $parentNodeId = null, $urlParts = [], $id = null, $nodeIcon = null, $hasRight = false)
    {
        global $g_dirs;

        $icon = $nodeIcon ? $g_dirs['images'] . $nodeIcon : null;

        $link = null;
        $nodeId = ($id === null) ? C__MODULE__DOCUMENT . $this->m_tree_counter++ : $id;

        if (count($urlParts)) {
            $urlParts = array_merge([
                C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
                C__GET__SETTINGS_PAGE => 'documents',
                C__GET__TREE_NODE     => $nodeId
            ], $urlParts);

            $link = isys_helper_link::create_url($urlParts);
        }

        return $this->m_request->get_menutree()
            ->add_node(
                $nodeId,
                $parentNodeId,
                $this->language->get($nodeTitle),
                $link,
                '',
                $icon,
                $_GET[C__GET__TREE_NODE] == $nodeId,
                '',
                '',
                $hasRight
            );
    }

    /**
     * This method builds the tree for the menu.
     *
     * @param   isys_component_tree $treeComponent
     * @param   boolean             $isSystemMenu
     * @param   integer             $parentNodeId
     *
     * @author  Selcuk Kekec <skekec@i-doit.com>
     * @see     isys_module_cmdb->build_tree();
     */
    public function build_tree(isys_component_tree $treeComponent, $isSystemMenu = true, $parentNodeId = null)
    {
        $auth = isys_auth_document::instance();

        // Add-on root node.
        $rootNode = $this->add_document_node('LC__MANUAL', -1, null);

        // "Documents" node.
        $nodeDocuments = $this->add_document_node(
            'LC__MODULE__DOCUMENT__DOCUMENTS',
            $rootNode,
            [C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENTS],
            null,
            'icons/silk/book.png',
            true
        );

        // "Templates" node.
        $nodeTemplates = $this->add_document_node(
            'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATES',
            $rootNode,
            [C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENT_TEMPLATES],
            null,
            null,
            true
        );

        // "Online repository" node.
        $this->add_document_node(
            'LC__MODULE__DOCUMENT__ONLINE_REPOSITORY',
            $rootNode,
            [C__GET__SETTINGS_PAGE => self::CL__ACTION__ONLINE_REPOSITORY],
            null,
            'icons/silk/world.png',
            $auth->is_allowed_to(isys_auth::VIEW, 'online-repository')
        );

        $categorySql = 'SELECT isys_document_type__id AS id, isys_document_type__title AS title
            FROM isys_document_type
            LEFT JOIN isys_document ON isys_document_type__id = isys_document__isys_document_type__id
            WHERE TRUE 
            GROUP BY isys_document_type__id;';

        // Add Types to documents and templates.
        $cateoryResult = $this->m_dao_documents->retrieve($categorySql);

        while ($categoryRow = $cateoryResult->get_row()) {
            // Count the inherited documents.
            $counter = null;
            $documentCountQuery = 'SELECT COUNT(*) AS cnt 
                FROM isys_document 
                WHERE isys_document__isys_document_type__id = ' . $this->m_dao_documents->convert_sql_id($categoryRow['id']) . ';';
            $documentCounter = (int) $this->m_dao_documents->retrieve($documentCountQuery)->get_row_value('cnt');

            if ($documentCounter) {
                $counter = '<span class="ml5 text-grey">(' . $documentCounter . ')</span>';
            }

            // Create document node.
            $this->add_document_node(
                '<span>' . $this->language->get($categoryRow['title']) . '</span>' . $counter,
                $nodeDocuments,
                [
                    C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENTS,
                    'documentTypeID'      => $categoryRow['id'],
                ],
                C__MODULE__DOCUMENT . '11' . $categoryRow['id'],
                'icons/silk/package_green.png',
                true
            );

            // Count the inherited documents.
            $counter = null;
            $templateCountQuery = 'SELECT COUNT(*) AS cnt 
                FROM isys_document_template 
                WHERE isys_document_template__isys_document_type__id = ' . $this->m_dao_documents->convert_sql_id($categoryRow['id']) . ';';
            $templateCounter = (int) $this->m_dao_documents->retrieve($templateCountQuery)->get_row_value('cnt');

            if ($templateCounter) {
                $counter = '<span class="ml5 text-grey">(' . $templateCounter . ')</span>';
            }

            // Create template node.
            $this->add_document_node(
                '<span>' . $this->language->get($categoryRow['title']) . '</span>' . $counter,
                $nodeTemplates,
                [
                    C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENT_TEMPLATES,
                    'documentTypeID'      => $categoryRow['id'],
                ],
                C__MODULE__DOCUMENT . '12' . $categoryRow['id'],
                'icons/silk/package_green.png',
                true
            );
        }

        // Build the template tree.
        $this->buildTemplateTree();

        // Tree-Node selection handling.
        $selectedNode = null;

        if ($_GET['treeNode']) {
            $selectedNode = $_GET['treeNode'];
        } elseif (isset($_GET[C__GET__SETTINGS_PAGE]) && $_GET[C__GET__SETTINGS_PAGE] === self::CL__ACTION__DOCUMENTS) {
            $selectedNode = $nodeDocuments;
        } elseif (isset($_GET[C__GET__SETTINGS_PAGE]) && $_GET[C__GET__SETTINGS_PAGE] === self::CL__ACTION__DOCUMENT_TEMPLATES && isset($_GET['id'])) {
            $selectedNode = C__MODULE__DOCUMENT . '0000' . $_GET['id'];
        }

        if ($selectedNode) {
            $this->m_request->get_menutree()->select_node_by_id($selectedNode, true);
        }

        $treeComponent->set_tree_sort(false);
    }


    /**
     * Recursive method for rendering template chapters inside the tree.
     *
     * @param   integer $p_parent_node
     * @param   integer $p_document_id
     * @param   integer $p_chapter_node
     * @param   string  $p_prepend
     *
     * @throws  Exception
     * @throws  isys_exception_database
     * @author  Leonard Fischer <lfischer@i-doit.com>
     * @see     DOKU-40
     */
    private function buildDocumentChapterTree($p_parent_node, $p_document_id, $p_chapter_node = null, $p_prepend = null)
    {
        $sql = 'SELECT isys_document_chapter__id AS id, isys_document_chapter__title AS title
            FROM isys_document_chapter
            WHERE isys_document_chapter__isys_document_template__id = ' . $this->m_dao_documents->convert_sql_id($p_document_id) . '
            AND isys_document_chapter__isys_document_chapter__id ' . ($p_chapter_node === null ? 'IS NULL' : '= ' . $this->m_dao_documents->convert_sql_id($p_chapter_node)) . '
            ORDER BY isys_document_chapter__sort ASC;';

        $result = $this->m_dao_documents->retrieve($sql);
        $l_cnt = 1;

        while ($l_row = $result->get_row()) {
            $l_treenode = C__MODULE__DOCUMENT . '990' . $l_row['id'];

            $l_chapter_node = $this->add_document_node(
                $p_prepend . $l_cnt . ' ' . $l_row['title'],
                $p_parent_node,
                [
                    C__GET__SETTINGS_PAGE         => self::CL__ACTION__CHAPTERS,
                    'action'                      => 'edit',
                    C__DOCUMENT__GET__DOCUMENT_ID => $p_document_id,
                    C__GET__ID                    => $l_row['id'],
                    C__GET__TREE_NODE             => $l_treenode
                ],
                $l_treenode,
                'icons/silk/book_open.png',
                true
            );

            $this->buildDocumentChapterTree($l_chapter_node, $p_document_id, $l_row['id'], $p_prepend . $l_cnt . '.');

            $l_cnt++;
        }
    }

    /**
     * Will build the document tree and return all categories of authorized documents.
     *
     * @return void
     * @throws isys_exception_database
     */
    private function buildTemplateTree()
    {
        $auth = isys_auth_document::instance();
        $documentResult = $this->m_dao_document_templates->get_data();

        while ($documentRow = $documentResult->get_row()) {
            $authAllowedToViewTemplate = $auth->is_allowed_to(isys_auth::VIEW, 'TEMPLATES/' . $documentRow['isys_document_template__id']);

            $documentNodeId = $this->add_document_node(
                $documentRow['isys_document_template__title'],
                C__MODULE__DOCUMENT . '12' . $documentRow['isys_document_type__id'],
                [
                    C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENT_TEMPLATES,
                    'action'              => 'edit',
                    'id'                  => $documentRow['isys_document_template__id']
                ],
                C__MODULE__DOCUMENT . '0000' . $documentRow['isys_document_template__id'],
                'icons/silk/book_edit.png',
                $authAllowedToViewTemplate
            );

            if ($authAllowedToViewTemplate) {
                $this->buildDocumentChapterTree($documentNodeId, $documentRow['isys_document_template__id']);
            }
        }
    }

    /**
     * Start method.
     */
    public function start()
    {
        try {
            // Check for the licence.
            if (!isys_module_document::is_licenced()) {
                throw new isys_exception_licence($this->language->get('LC__LICENCE__NO_MODULE_LICENCE', [$this->language->get('LC__MODULE__DOCUMENT')]), 1);
            } // if

            $auth = isys_auth_document::instance();

            // Always look for templates and documents without type and choose global as default
            $templateSql = 'UPDATE isys_document_template 
                SET isys_document_template__isys_document_type__id = (
                    SELECT isys_document_type__id 
                    FROM isys_document_type 
                    WHERE isys_document_type__const = "C__DOCUMENT_TYPE__BHB" 
                    LIMIT 1) 
                WHERE isys_document_template__isys_document_type__id IS NULL;';
            $documentSql = 'UPDATE isys_document 
                SET isys_document__isys_document_type__id = (
                    SELECT isys_document_type__id 
                    FROM isys_document_type 
                    WHERE isys_document_type__const = "C__DOCUMENT_TYPE__BHB" 
                    LIMIT 1) 
                WHERE isys_document__isys_document_type__id IS NULL;';

            $this->database->query($templateSql) && $this->database->commit();
            $this->database->query($documentSql) && $this->database->commit();

            if (!$auth->has_any_rights_in_module()) {
                new isys_exception_auth($this->language->get('LC__AUTH__MANUAL_EXCEPTION__MISSING_RIGHT_FOR_DOCUMENTS'));
            }

            if ($_GET[C__GET__AJAX] && $_GET['func'] == 'build_tree') {
                $this->build_tree($this->m_request->get_menutree(), false, null);

                echo $this->m_request->get_menutree()->process();
            }

            if ($_GET[C__GET__MODULE_ID] != C__MODULE__SYSTEM) {
                $this->build_tree($this->m_request->get_menutree(), false, null);

                $this->m_request->get_template()
                    ->assign("menu_tree", $this->m_request->get_menutree()->process());
            }

            $this->template->appendJavascript(self::getWwwPath() . 'templates/document-base.js');

            // Routing
            switch ($_GET[C__GET__SETTINGS_PAGE]) {
                case self::CL__ACTION__DOCUMENTS:
                    $this->process_documents();
                    break;

                case self::CL__ACTION__DOCUMENT_EXPORT:
                    $this->process_document_export($_GET['type'], $_GET[C__DOCUMENT__GET__DOCUMENT_ID], $_GET[C__DOCUMENT__GET__REVISION_ID], $_GET['inline']);
                    break;

                case self::CL__ACTION__TEMPLATE_EXPORT_DOWNLOAD:
                    $this->process_template_export_download($_GET['file']);
                    break;

                case self::CL__ACTION__ONLINE_REPOSITORY:
                    return (new isys_document_view_online_repository($this->template, $this->m_dao_documents, $this->m_dao_document_templates))->process();

                case self::CL__ACTION__DOCUMENT_TEMPLATES:
                    $this->process_document_templates();
                    break;

                case self::CL__ACTION__CHAPTERS:
                    $this->process_chapters();
                    break;

                case self::CL__ACTION__TREE_RELOAD:
                    echo $this->m_request->get_menutree()
                        ->process();
                    die;

                case self::CL__ACTION__TEMPLATE_PREVIEW:
                    $this->process_document_templates__preview($_GET['id']);
                    break;

                default:
                    if ($auth->has('documents') || $auth->has('documents_in_categories')) {
                        $this->process_documents();
                    } else {
                        $this->process_document_templates();
                    }
            }
        } catch (Exception $e) {
            global $g_error;

            $g_error = $e->getMessage();

            isys_notify::error($e->getMessage(), ['sticky' => true]);
        }
    }

    /**
     * Document export handler.
     *
     * @param string  $type
     * @param integer $documentId
     * @param integer $revisionId
     * @param boolean $inline
     * @param boolean $deleteDocument
     */
    protected function process_document_export($type, $documentId, $revisionId = null, $inline = false, $deleteDocument = false)
    {
        $exporter = null;

        try {
            if (!$type) {
                return;
            }

            if (!$documentId) {
                throw new isys_exception_general('Provide a document, please.');
            }

            switch ($type) {
                case 'pdf':
                    $l_defaultPDFFormatter = isys_tenantsettings::get('document.default-pdf-formatter', 'isys_document_format_pdf');
                    if (!class_exists($l_defaultPDFFormatter)) {
                        $l_defaultPDFFormatter = 'isys_document_format_pdf';
                    }

                    // Create the exporter and inject the lib.
                    $exporter = new isys_document_export_pdf([], new $l_defaultPDFFormatter());
                    break;

                case 'html':
                    $exporter = new isys_document_export_html([], new isys_document_format_html());
                    break;

                default:
                    throw new isys_exception_general("Export type '" . $type . "' is not supported.");
            }

            try {
                $session = isys_application::instance()->container->get('session');
            } catch (Exception $e) {
                $session = false;
            }

            // DOKU-41 Check if we have cached the requested document.
            $cacheDocuments = (bool)isys_tenantsettings::get('idoit.addon.document.caching', 1);
            $cacheDirectory = isys_glob_get_temp_dir();

            // If the session-object exists, we try to receive the mandator-cache directory.
            if (is_a($session, 'isys_component_session')) {
                $l_mandator_data = $session->get_mandator_data();

                $cacheDirectory = rtrim($cacheDirectory, '/') . '/' . $l_mandator_data['isys_mandator__dir_cache'];
            }

            if ($revisionId === null) {
                $revisionId = $this->m_dao_document_revisions->get_last_revision($documentId);
            }

            $filename = $this->prepareExportFilename($documentId, $revisionId);
            $cacheDirectory = rtrim($cacheDirectory, '/') . '/';
            $cacheFilePath = $cacheDirectory . $filename . '.' . $exporter->get_file_extension();

            if ($cacheDocuments && file_exists($cacheFilePath)) {
                if ($inline) {
                    header('Content-Disposition: inline; filename="' . $filename . '.' . $exporter->get_file_extension() . '"');
                    header('Content-Type: ' . $exporter->get_mimetype());
                } else {
                    header('Content-disposition: attachment; filename="' . $filename . '.' . $exporter->get_file_extension() . '"');
                    header('Content-Type: application/download');
                }

                echo file_get_contents($cacheFilePath);
                die;
            }

            $l_document_result = $this->m_dao_document_revisions->get_data($revisionId);

            if (!count($l_document_result)) {
                throw new isys_exception_general('Document does not exist.');
            }

            $l_document_data = $l_document_result->get_row();

            $l_models = $this->m_dao_document_revisions->load_models_for_revision($documentId, $revisionId);

            if (!is_array($l_models) || !count($l_models)) {
                throw new isys_exception_general('Document does not have any chapters.');
            }

            // Get export configuration from template.
            $l_options = $this->m_dao_document_templates->get_options($l_document_data['isys_document_template__id']);

            $exporter->set_title($l_document_data['isys_document__title'])
                ->set_models($l_models)
                ->set_options($l_options + [
                        'index'           => $l_document_data['isys_document_revision__index'],
                        'header'          => $l_document_data['isys_document_revision__header'],
                        'footer'          => $l_document_data['isys_document_revision__footer'],
                        'revisionId'      => $revisionId,
                        'revisionComment' => $l_document_data['isys_document_revision__comment']
                    ])
                ->export();

            if ($cacheDocuments) {
                $exporter->write($filename, $cacheDirectory);

                if (file_exists($cacheFilePath)) {
                    if ($inline) {
                        header('Content-Disposition: inline; filename="' . $filename . '.' . $exporter->get_file_extension() . '"');
                        header('Content-Type: ' . $exporter->get_mimetype());
                    } else {
                        header('Content-disposition: attachment; filename="' . $filename . '.' . $exporter->get_file_extension() . '"');
                        header('Content-Type: application/download');
                    }

                    echo file_get_contents($cacheFilePath);
                }
            } else {
                $exporter->send($filename, $inline);
            }

            if ($deleteDocument) {
                $this->process_documents__delete($documentId);
            }

            die;

        } catch (ErrorException $e) {
            global $g_error;

            $l_exception = new isys_exception_general($e->getMessage());
            $l_exception->write_log();

            isys_notify::error($this->language->get('LC__MODULE__DOCUMENT__ERROR__EXPORT', $e->getMessage()));
            $g_error = $this->language->get('LC__MODULE__DOCUMENT__ERROR__EXPORT', $e->getMessage());
        } catch (Exception $e) {
            global $g_error;

            isys_notify::error($this->language->get('LC__MODULE__DOCUMENT__ERROR__EXPORT', $e->getMessage()));
            $g_error = $this->language->get('LC__MODULE__DOCUMENT__ERROR__EXPORT', $e->getMessage());
        }
    }

    /**
     * Method for offering a "template export ZIP" as download.
     *
     * @param  string $p_filename
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function process_template_export_download($p_filename)
    {
        if ($p_filename === 'data.zip' || strpos($p_filename, 'document-template-') === 0) {
            $l_filemanager = new isys_component_filemanager();
            $l_filemanager->send(self::get_export_dir() . DS . $p_filename);
        }

        // This is necessary, because some browsers will actually forward the user to the "download page".
        header('Location: ' . isys_helper_link::create_url([C__GET__MODULE_ID => C__MODULE__DOCUMENT, C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENT_TEMPLATES]));
        die;
    }

    /**
     * This method returns a filename, like "<document-template-name>_<object-name>_<timestamp>".
     *
     * @param  integer $documentId
     * @param  integer $revisionId
     *
     * @return string
     * @author Leonard Fischer <lfischer@i-doit.com>
     */
    protected function prepareExportFilename($documentId, $revisionId)
    {
        $l_doc = isys_document_dao::instance($this->database)
            ->get_data($documentId)
            ->get_row();

        $filename = $this->language->get($l_doc['isys_document_type__title']) .
            ' - ' . $l_doc['isys_document_template__title'] .
            ' - ' . $l_doc['isys_obj__title'] .
            ' - Revision ' . $revisionId;

        return isys_helper_upload::prepare_filename($filename);
    }

    /**
     * Document processor
     *
     * @author  Selcuk Kekec <skekec@i-doit.com>
     */
    protected function process_documents()
    {
        // Hiding the record status dialog.
        $this->template->smarty_tom_add_rule('tom.content.navbar.cRecStatus.p_bInvisible=1');

        $l_id = $_POST['id'][0];

        switch (isys_glob_get_param(C__GET__NAVMODE)) {
            case C__NAVMODE__DELETE:
                $this->process_documents__delete(self::id_handler(false, $_GET['id']));
                $this->process_documents__list();
                break;

            case C__NAVMODE__SAVE:
                $this->process_documents__save($_POST['template_id'], $_POST);
                $this->process_documents__list();
                break;

            case C__NAVMODE__EDIT:
                if ($l_id) {
                    $this->process_documents__edit($l_id);
                } else {
                    $this->process_documents__list();
                }
                break;

            case C__NAVMODE__NEW:
                $this->process_documents__edit();
                break;

            default:
                if ($_GET['action'] === 'edit' && isset($_GET['id'])) {
                    $this->process_documents__edit($_GET['id']);
                } else {
                    $this->process_documents__list();
                }
        }
    }

    /**
     *
     * @param   boolean $p_first
     * @param   integer $p_default
     *
     * @return  integer
     */
    protected static function id_handler($p_first = false, $p_default)
    {
        $l_id = null;

        if (isset($_POST['id']) && !empty($_POST['id'])) {
            $l_id = $_POST['id'];
        } else {
            $l_id = $p_default;
        } // if

        if ($p_first && is_array($l_id)) {
            $l_id = $l_id[0];
        } // if

        return $l_id;
    } // function

    /**
     * Document templates processor.
     *
     * @author  Selcuk Kekec <skekec@i-doit.com>
     * @throws \idoit\Exception\JsonException
     * @throws isys_exception_auth
     * @throws isys_exception_general
     */
    protected function process_document_templates()
    {
        $auth = isys_auth_document::instance();

        if (!$auth->has('templates') && !$auth->has('templates_in_categories')) {
            throw new isys_exception_auth($this->language->get('LC__AUTH__MANUAL_EXCEPTION__MISSING_RIGHT_FOR_TEMPLATES'));
        }

        $id         = $_GET['id'];
        $navMode    = isys_glob_get_param(C__GET__NAVMODE);
        $templateId = $_POST['template_id'];

        $this->template->smarty_tom_add_rule("tom.content.navbar.cRecStatus.p_bInvisible=1");

        if (is_array($id)) {
            $id = current($id);
        }

        if (is_array($navMode)) {
            $navMode = current($navMode);
        }

        if (is_array($templateId)) {
            $templateId = current($templateId);
        }

        switch ($navMode) {
            case C__NAVMODE__DELETE:
                $this->process_document_templates__delete(self::id_handler(false, $id));
                $this->process_document_templates__list();
                break;

            case C__NAVMODE__DUPLICATE:
                $this->process_document_templates__duplicate(
                    isys_format_json::decode($_POST['documentTemplateID']),
                    $_POST['C__POPUP__DUPLICATE__DOCUMENT_TEMPLATE__CATEGORY'],
                    $_POST['C__POPUP__DUPLICATE__DOCUMENT_TEMPLATE__NEW_NAME']
                );
                $this->process_document_templates__list();
                break;

            case C__NAVMODE__SAVE:
                $id = $this->process_document_templates__save($templateId, $_POST);

                $this->process_document_templates__edit($id);
                //$this->process_document_templates__list();
                break;

            case C__NAVMODE__EDIT:
                $id = self::id_handler(true, $id);

                $daoLock = isys_component_dao_lock::instance($this->database);
                // Check if there is any lock existing
                if ($daoLock->is_locked(null, null, null, 'isys_document_template', 'isys_document_template__id', $id)) {
                    isys_notify::warning($this->language->get('LC__MODULE__DOCUMENT_TEMPLATE__LOCKED'), ['life' => 10]);
                    $this->process_document_templates__list();
                    break;
                }

                // don't open new on edition with empty selection
                if ($id) {
                    $daoLock->add_lock(null, 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE', 'isys_document_template', 'isys_document_template__id', $id);
                    $this->process_document_templates__edit($id, $_GET['documentTypeID']);
                } else {
                    $this->process_document_templates__list();
                }
                break;

            case C__NAVMODE__NEW:
                $this->process_document_templates__edit(null, $_GET['documentTypeID']);
                break;

            default:
                if ($_GET['action'] === "edit" && isset($id)) {
                    $daoLock = isys_component_dao_lock::instance($this->database);
                    // Check if there is any lock existing
                    if ($daoLock->is_locked(null, null, null, 'isys_document_template', 'isys_document_template__id', $id)) {
                        isys_notify::warning($this->language->get('LC__MODULE__DOCUMENT_TEMPLATE__LOCKED'), ['life' => 10]);
                        $this->process_document_templates__list();
                        break;
                    }

                    $daoLock->add_lock(null, 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE', 'isys_document_template', 'isys_document_template__id', $id);
                    $this->process_document_templates__edit($id, $_GET['documentTypeID']);
                } elseif ($_GET['action'] === 'template_preview') {
                    $this->process_document_templates__preview($id);
                } else {
                    $this->process_document_templates__list();
                }
        }
    }

    /**
     * Chapter processor
     *
     * @author  Selcuk Kekec <skekec@i-doit.com>
     */
    protected function process_chapters()
    {
        $l_id = $_POST['id'][0];
        $documentId = (int) $_GET[C__DOCUMENT__GET__DOCUMENT_ID];

        $daoLock = isys_component_dao_lock::instance($this->database);

        // Check if there is any lock existing
        if ($daoLock->is_locked(null, null, null, 'isys_document_template', 'isys_document_template__id', $documentId)) {
            isys_notify::warning($this->language->get('LC__MODULE__DOCUMENT_TEMPLATE__LOCKED'), ['life' => 10]);
            $this->process_document_templates__list();
            return;
        }

        // Create a document lock!
        $daoLock->add_lock(null, 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE', 'isys_document_template', 'isys_document_template__id', $documentId);

        switch (isys_glob_get_param(C__GET__NAVMODE)) {
            case C__NAVMODE__DELETE:
                $this->process_chapters__delete($_POST['id']);
                $this->process_document_templates__edit($documentId);
                break;

            case C__NAVMODE__SAVE:
                $this->process_chapters__save($_POST['chapter_id'], $_POST, $documentId);
                break;

            case C__NAVMODE__EDIT:
                $this->process_chapters__edit($l_id, $documentId);
                break;

            case C__NAVMODE__NEW:
                $this->process_chapters__edit(null, $documentId);
                break;

            default:
                if ($_GET['action'] === 'edit' && (isset($_GET['id']) || isset($_GET['parentChapterID']))) {
                    $this->process_chapters__edit($_GET['id'], $documentId);
                } else {
                    $this->process_document_templates__edit($documentId);
                }
        }

        // @see DOKU-257
        $l_preview_link = isys_helper_link::create_url([
            C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
            C__GET__SETTINGS_PAGE => self::CL__ACTION__TEMPLATE_PREVIEW,
            'id'                  => $documentId
        ]);

        $params = base64_encode(json_encode([
            'multiselection' => false,
            'name' => 'template_preview',
            isys_popup_browser_object_ng::C__CALLBACK__ACCEPT => 'preview_template(\'' . $l_preview_link . '&objectId=\' + selection[0]);'
        ]));

        isys_component_template_navbar::getInstance()->append_button('LC__MODULE__DOCUMENT__DOCUMENT__TEMPLATE_PREVIEW', 'template_preview', [
            'tooltip'    => 'LC__MODULE__DOCUMENT__DOCUMENT__TEMPLATE_CREATE_PREVIEW_FOR_TEMPLATE',
            'active'     => true,
            'visible'    => true,
            'icon'       => 'icons/silk/zoom.png',
            'navmode'    => 'template_preview',
            'js_onclick' => "get_popup('browser_object_ng', '', '882', '550', {params: '$params'}, '');"
        ]);
    }

    /**
     * Removes a document
     *
     * @param   array|int $id
     *
     * @return  void
     * @author  Selcuk Kekec <skekec@i-doit.com>
     */
    protected function process_document_templates__delete($id)
    {
        $auth = isys_auth_document::instance();

        if ($auth->is_allowed_to(isys_auth::DELETE, 'TEMPLATES/' . isys_auth::EMPTY_ID_PARAM)) {
            $this->m_dao_document_templates->deleteTemplate($id);
            return;
        }

        // If the user is not allowed to delete templates, we'll check if he is allowed to delete specific templates.
        $id = (array) $id;

        foreach ($id as $documentTemplateId) {
            if (!$auth->is_allowed_to(isys_auth::DELETE, 'TEMPLATES/' . $documentTemplateId)) {
                isys_notify::error('You are not allowed to delete document template #' . $documentTemplateId);
                continue;
            }

            $this->m_dao_document_templates->deleteTemplate($documentTemplateId);
        }
    }

    /**
     * Removes a chapter.
     *
     * @param   mixed $p_chapter_id
     *
     * @author  Selcuk Kekec <skekec@i-doit.com>
     */
    protected function process_chapters__delete($p_chapter_id)
    {
        $this->m_dao_document_chapters->deleteChapter($p_chapter_id);
    }

    /**
     * Removes a chapter.
     *
     * @param   int|array $id
     *
     * @author  Selcuk Kekec <skekec@i-doit.com>
     */
    protected function process_documents__delete($id)
    {
        $auth = isys_auth_document::instance();

        if ($auth->is_allowed_to(isys_auth::DELETE, 'DOCUMENTS/' . isys_auth::EMPTY_ID_PARAM)) {
            $this->m_dao_documents->deleteDocument($id);
            return;
        }

        $id = (array) $id;

        foreach ($id as $documentId) {
            if (!$auth->is_allowed_to(isys_auth::DELETE, 'TEMPLATES/' . $documentId)) {
                isys_notify::error('You are not allowed to delete document #' . $documentId);
                continue;
            }

            $this->m_dao_documents->deleteDocument($documentId);
        }
    }

    /**
     * Save-action for writing the document to the database. This is used for "CREATE" and "UPDATE".
     *
     * @param  integer $documentId
     * @param  array   $postData
     *
     * @throws isys_exception_auth
     * @author Selcuk Kekec <skekec@i-doit.com>
     */
    protected function process_documents__save($documentId = null, array $postData = [])
    {
        $auth = isys_auth_document::instance();

        if ($documentId !== null) {
            $auth->check(isys_auth::EDIT, 'DOCUMENTS/' . $documentId);
        }

        $this->m_dao_documents->saveDocument($documentId, [
                'title'       => $postData['C__DOCUMENT__TITLE'],
                'description' => $postData['C__DOCUMENT__DESCRIPTION'],
            ]);

        if (isset($postData['custom-field']) && is_array($postData['custom-field'])) {
            $customFieldDocument = new Document($this->database);

            foreach ($postData['custom-field'] as $customFieldId => $customFieldValue) {
                if (isset($postData['custom-field__HIDDEN'][$customFieldId])) {
                    $customFieldValue = $postData['custom-field__HIDDEN'][$customFieldId];
                }

                $customFieldDocument->saveCustomFieldValue($customFieldId, $documentId, $customFieldValue);
            }
        }


        isys_notify::success($this->language->get('LC__MODULE__DOCUMENT__NOTIFICATION__DOCUMENT__SUCCESS'));
    }

    /**
     * Save-action for writing the document template to the database. This is used for "CREATE" and "UPDATE".
     *
     * @param   integer $templateId
     * @param   array   $postData
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    protected function process_document_templates__save($templateId = null, array $postData = [])
    {
        $auth = isys_auth_document::instance();

        if ($templateId !== null) {
            $auth->check(isys_auth::EDIT, 'TEMPLATES/' . $templateId);
        } elseif (
            isset($postData['C__DOCUMENT__TYPE']) &&
            !$auth->is_allowed_to(isys_auth::CREATE, 'TEMPLATES/' . isys_auth::EMPTY_ID_PARAM) &&
            !$auth->is_allowed_to(isys_auth::CREATE, 'TEMPLATES_IN_CATEGORIES/' . isys_auth::EMPTY_ID_PARAM) &&
            !$auth->is_allowed_to(isys_auth::CREATE, 'TEMPLATES_IN_CATEGORIES/' . $postData['C__DOCUMENT__TYPE'])
        ) {
            new isys_exception_auth($this->language->get('LC__AUTH__MANUAL_EXCEPTION__MISSING_RIGHT_FOR_TEMPLATES'));
        }

        // @see  DOKU-308  Sanitize some options.
        foreach($postData['options'] as $key => &$value) {
            if ($key === 'general.toc' && $value > 0) {
                $postData['options']['general.headlines'] = 1;
            }

            if (preg_match('~general\.h[1-6]\.size~', $key)) {
                $value = (int)$value;
            }
        }

        $templateId = $this->m_dao_document_templates->saveTemplate($templateId, [
                'title'                  => $postData['C__DOCUMENT__TITLE'],
                'isys_document_type__id' => $postData['C__DOCUMENT__TYPE'],
                'index'                  => $postData['C__TEMPLATE__INDEX'],
                'header'                 => $postData['C__TEMPLATE__HEADER'],
                'footer'                 => $postData['C__TEMPLATE__FOOTER'],
                'options'                => $postData['options'],
            ]);

        // DOKU-273 Remove selected custom fields.
        if (isset($postData['custom-fields-deletion'])) {
            $delection = isys_format_json::decode($postData['custom-fields-deletion']);

            if (is_array($delection) && count($delection)) {
                Configuration::instance($this->database)->deleteCustomFields($delection);
            }
        }

        // DOKU-273 Save custom fields data.
        if ($templateId && isset($postData['custom-fields'])) {
            $configurationModel = Configuration::instance($this->database);
            $customFieldsConfiguration = isys_format_json::decode($postData['custom-fields']);

            if (!is_array($customFieldsConfiguration)) {
                $customFieldsConfiguration = [];
            }

            foreach ($customFieldsConfiguration as $i => $customField) {
                try {
                    $id = (strpos($customField['id'], 'new') === false) ? $customField['id'] : 0;

                    $result = $configurationModel->saveCustomField(new CustomField(
                        $id,
                        $templateId,
                        $customField['name'],
                        str_replace('.', '\\', $customField['dataType']),
                        $customField['defaultValue'] ?: null
                    ), $i);
                } catch (UnknownDataType $e) {
                    isys_notify::error($e->getMessage(), ['sticky' => true]);
                }
            }
        }

        isys_notify::success($this->language->get('LC__MODULE__DOCUMENT__NOTIFICATION__TEMPLATE__SUCCESS'));

        // DOKU-92 This happenes, when we create a new document template.
        $l_redirect_to = isys_helper_link::create_url([
                C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
                C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENT_TEMPLATES,
                C__GET__TREE_NODE     => C__MODULE__DOCUMENT . '0000' . $templateId,
                'action'              => 'edit',
                C__GET__ID            => $templateId
            ]);

        header('Location: ' . $l_redirect_to);
        die;
    }

    /**
     * Duplicate complete document template(s)
     *
     * @param   array       $p_id
     * @param   integer     $p_document_category
     * @param        string $p_title
     *
     * @author  Van Quyen Hoang <qhoang@i-doit.com>
     */
    protected function process_document_templates__duplicate($p_id = [], $p_document_category = null, $p_title = '')
    {
        isys_auth_document::instance()
            ->check(isys_auth::EDIT, 'TEMPLATES');

        $l_res               = $this->m_dao_document_components->get_data();
        $l_component_mapping = $l_chapter_mapping = $l_chapter_update = $l_component_update = [];
        while ($l_row = $l_res->get_row()) {
            $l_component_mapping[$l_row['isys_document_component__id']] = $l_row['isys_document_component__id'];
            $l_chapter_mapping[$l_row['isys_document_chapter__id']]     = $l_row['isys_document_chapter__id'];
        } // while

        if (!is_array($p_id)) {
            $p_id = [$p_id];
        } // if

        $templates = $this->m_dao_document_templates->get_data($p_id);

        while ($template = $templates->get_row()) {
            $l_template_id            = $template['isys_document_template__id'];
            $l_document_template_data = [
                'title'                  => $p_title,
                'isys_document_type__id' => $p_document_category,
                'index'                  => $template['isys_document_template__index'],
                'header'                 => $template['isys_document_template__header'],
                'footer'                 => $template['isys_document_template__footer'],
                'options'                => $template['isys_document_template__options']
            ];

            $l_document_template_id = $this->m_dao_document_templates->saveTemplate(null, $l_document_template_data);

            $l_res = $this->m_dao_document_components->get_data(
                null,
                null,
                'AND isys_document_template__id = ' . $this->m_dao_documents->convert_sql_id($l_template_id)
            );

            while ($l_row = $l_res->get_row()) {
                // Add chapter entry
                $l_document_chapter_data                                = [
                    'title'                      => $l_row['isys_document_chapter__title'],
                    'isys_document_template__id' => $l_document_template_id,
                    'description'                => $l_row['isys_document_chapter__description'],
                    'sort'                       => $l_row['isys_document_chapter__sort'],
                    'new_page'                   => $l_row['isys_document_chapter__new_page']
                ];
                $l_chapter_id                                           = $this->m_dao_document_chapters->saveChapter(null, $l_document_chapter_data);
                $l_chapter_mapping[$l_row['isys_document_chapter__id']] = $l_chapter_id;

                if ($l_row['isys_document_chapter__isys_document_chapter__id'] !== null) {
                    $l_chapter_update[$l_chapter_id] = $l_row['isys_document_chapter__isys_document_chapter__id'];
                } // if

                // Update component entry
                $l_document_component_data = [
                    'text' => $l_row['isys_document_component__text']
                ];
                $l_component_id            = isys_document_dao_components::get_last_component_id();
                $this->m_dao_document_components->saveComponent($l_component_id, $l_document_component_data);
                $l_component_mapping[$l_row['isys_document_component__id']] = $l_component_id;

                if ($l_row['isys_document_component__isys_document_component__id'] !== null) {
                    $l_component_update[$l_component_id] = $l_row['isys_document_component__isys_document_component__id'];
                } // if
            } // while
        } // foreach

        if (count($l_component_update) > 0) {
            // Update component to component assignment
            foreach ($l_component_update as $l_component_id => $l_assigned_component_id) {
                $l_new_assigned_component_id = $l_component_mapping[$l_assigned_component_id];
                $this->m_dao_document_components->saveComponent($l_component_id, ['isys_document_component__id' => $l_new_assigned_component_id]);
            } // foreach
        } // if

        if (count($l_chapter_update) > 0) {
            // Update chapter to chapter assignment
            foreach ($l_chapter_update as $l_chapter_id => $l_assigned_chapter_id) {
                $l_new_assigned_chapter_id = $l_chapter_mapping[$l_assigned_chapter_id];
                $this->m_dao_document_chapters->saveChapter($l_chapter_id, ['isys_document_chapter__id' => $l_new_assigned_chapter_id]);
            } // foreach
        } // if

        isys_notify::success($this->language->get('LC__MODULE__DOCUMENT__NOTIFICATION__TEMPLATE__SUCCESS'));
    } // function

    /**
     * Save-action for writing the chapter to the database. This is used for "CREATE" and "UPDATE".
     *
     * @param  integer $p_id
     * @param  array   $p_configuration
     * @param  integer $p_document_id
     */
    protected function process_chapters__save($p_id = null, array $p_configuration = [], $p_document_id)
    {
        $p_configuration['C__CHAPTER__PARENT'] = $p_configuration['C__CHAPTER__PARENT'] == -1 ? 'NULL' : $p_configuration['C__CHAPTER__PARENT'];

        $l_id = $this->m_dao_document_chapters->saveChapter($p_id, [
                'title'                       => $p_configuration['C__CHAPTER__TITLE'],
                'isys_document_component__id' => $p_configuration['component_id'],
                'isys_document_chapter__id'   => $p_configuration['C__CHAPTER__PARENT'],
                'isys_document_template__id'  => $p_document_id,
                'new_page'                    => $p_configuration['C__CHAPTER__NEW_PAGE']
            ]);

        $this->m_dao_document_chapters->update_component($l_id, $p_configuration['C__CHAPTER__TEXT'], $p_configuration['C__CHAPTER__TEMPLATE_COMPONENT']);

        isys_notify::success($this->language->get('LC__MODULE__DOCUMENT__NOTIFICATION__CHAPTER__SUCCESS'));

        $parameters = [
            C__GET__MODULE_ID             => C__MODULE__DOCUMENT,
            C__GET__SETTINGS_PAGE         => self::CL__ACTION__CHAPTERS,
            'action'                      => 'edit',
            C__DOCUMENT__GET__DOCUMENT_ID => $p_document_id,
            C__GET__ID                    => $l_id
        ];

        // @see  DOKU-226  Our
        if ($p_id === null) {
            $this->template->assign('redirectAfterSave', isys_helper_link::create_url($parameters, true));
        }
    }

    /**
     * Edit-action for displaying the document form. This is used for "NEW" and "EDIT".
     *
     * @param   integer $documentId
     *
     * @author  Selcuk Kekec <skekec@i-doit.com>
     */
    protected function process_documents__edit($documentId = null)
    {
        global $g_dirs;

        $l_revisions = $l_host = $l_rules = [];
        $auth = isys_auth_document::instance();
        $locales = isys_application::instance()->container->get('locales');

        if (!$auth->has('documents') && !$auth->has('documents_in_categories')) {
            throw new isys_exception_auth($this->language->get('LC__AUTH__MANUAL_EXCEPTION__MISSING_RIGHT_FOR_DOCUMENTS'));
        }

        $allowedToEdit = $auth->is_allowed_to(isys_auth::EDIT, 'DOCUMENTS/' . isys_auth::EMPTY_ID_PARAM);
        $allowedToCreate = $auth->is_allowed_to(isys_auth::CREATE, 'DOCUMENTS/' . isys_auth::EMPTY_ID_PARAM);

        if ($documentId !== null) {
            $auth->check(isys_auth::VIEW, 'DOCUMENTS/' . $documentId);
            $allowedToEdit = $auth->is_allowed_to(isys_auth::EDIT, 'DOCUMENTS/' . $documentId);
        }

        if ($documentId !== null && $documentId > 0) {
            $l_host = $this->m_dao_documents
                ->get_data($documentId)
                ->get_row();

            $l_rules['template_id']['p_strValue'] = $documentId;
            $l_rules['C__DOCUMENT__DESCRIPTION']['p_strValue'] = $l_host['isys_document__description'];
            $l_rules['C__DOCUMENT__TYPE']['p_strSelectedID'] = $l_host['isys_document__isys_document_type__id'];
            $l_rules['C__DOCUMENT__TITLE']['p_strValue'] = $l_host['isys_document__title'];

            if (isset($l_host['isys_obj__id']) && $l_host['isys_obj__id']) {
                $l_quick_info = new isys_ajax_handler_quick_info();
                $l_assigned_object = '<img src="' . $g_dirs['images'] . 'icons/silk/link.png" class="vam mr5" />' .
                    $this->language->get(isys_cmdb_dao::instance($this->database)->get_objtype_name_by_id_as_string($l_host['isys_obj__isys_obj_type__id'])) . ' &raquo; ' . $l_host['isys_obj__title'];

                $this->template->assign('assigned_object_url', $l_quick_info->get_quick_info($l_host['isys_obj__id'], $l_assigned_object, C__LINK__OBJECT));
            }

            /* Get revisions */
            $l_revision_res = $this->m_dao_document_revisions->get_revisions_of_document($documentId);

            if ($l_revision_res->count()) {
                while ($l_row = $l_revision_res->get_row()) {
                    $l_row['isys_document_revision__created'] = $locales->fmt_datetime($l_row['isys_document_revision__created']);

                    try {
                        $l_row['customFields'] = isys_format_json::decode($l_row['isys_document_revision__custom_fields']);
                    } catch (Exception $e) {
                        $l_row['customFields'] = [
                            $this->language->get('LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__CUSTOM_FIELDS') => $this->language->get('LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__CUSTOM_FIELDS__TO_MUCH_DATA')
                        ];
                    }

                    $l_revisions[] = $l_row + static::get_export_links($documentId, $l_row['isys_document_revision__id']);
                }
            }
        }

        $l_template_url = '<i style="color:grey;">' . $this->language->get('LC__MODULE__DOCUMENT__NO_LINKED_TEMPLATE') . '</i>';

        if ($l_host['isys_document_template__id'] !== null) {
            $link = isys_helper_link::create_url([
                C__GET__MODULE_ID      => C__MODULE__DOCUMENT,
                C__GET__SETTINGS_PAGE  => self::CL__ACTION__DOCUMENT_TEMPLATES,
                'action'               => 'edit',
                'id'                   => $l_host['isys_document_template__id'],
                C__CMDB__GET__TREEMODE => C__MODULE__DOCUMENT . '00000' . $l_host['isys_document_type__id'],
            ]);

            $l_template_url = '<a href="' . $link . '" class="vam">' .
                '<img src="' . $g_dirs['images'] . 'icons/silk/link.png" class="vam mr5" /><span>' . $l_host['isys_document_template__title'] . '</span>' .
                '</a>';
        }

        $customFieldData = [];

        if ($documentId > 0) {
            $customFieldRepository = new CustomFieldsRepository();
            $customFields = $customFieldRepository->getCustomFieldsForDocument($documentId);

            foreach ($customFields as $customField) {
                $value = null;

                $customFieldValue = $customField->getValue();
                $customFieldStringValue = $customField->getValue()->getStringValue() ?: $customField->getDefaultValue()->getStringValue();

                $customFieldData[$customField->getName()] = $customFieldValue->getTemplate(
                    'custom-field[' . $customField->getId() . ']',
                    $customFieldStringValue,
                    $allowedToEdit
                );
            }
        }

        $this->template->assign('document_id', $documentId)
            ->assign('custom_field_data', $customFieldData)
            ->assign('object_id', $l_host['isys_obj__id'])
            ->assign('creation_date', $locales->fmt_datetime($l_host['isys_document__created']))
            ->assign('template_url', $l_template_url)
            ->assign('revisions', $l_revisions)
            ->assign('export_links', static::get_export_links($documentId))
            ->smarty_tom_add_rules('tom.content.bottom.content', $l_rules)
            ->include_template('contentbottomcontent', self::get_tpl_dir() . 'documents/documents_form.tpl');

        $_GET[C__CMDB__GET__EDITMODE] = 0;

        if ($documentId) {
            $l_navbar = isys_component_template_navbar::getInstance();

            $l_navbar->set_active(false, C__NAVBAR_BUTTON__SAVE);

            if ($allowedToEdit) {
                $this->template->activate_editmode();

                // Document-Refresh button
                $l_navbar->append_button('LC__MODULE__DOCUMENT__DOCUMENT__REGENERATE_DOCUMENT', 'regenerate_document', [
                        'active'     => true,
                        'visible'    => true,
                        'icon'       => 'icons/silk/arrow_refresh.png',
                        'js_onclick' => ';',
                        'navmode'    => 'regenerate_document'
                    ])
                    ->set_active(true, C__NAVBAR_BUTTON__SAVE);
            }

            if ($auth->is_allowed_to(isys_auth::DELETE, 'DOCUMENTS/' . $documentId)) {
                $l_navbar->set_active(true, C__NAVBAR_BUTTON__DELETE)
                    ->set_title('LC__MODULE__DOCUMENT__REVISION_REMOVE_BUTTON', C__NAVBAR_BUTTON__DELETE)
                    ->set_js_function('onclick=""', C__NAVBAR_BUTTON__DELETE);
            }
        }
    }

    /**
     * Edit-action for displaying the document teplate form. This is used for "NEW" and "EDIT".
     *
     * @param  integer $documentId
     * @param  integer $p_document_type_id
     *
     * @throws isys_exception_auth
     */
    protected function process_document_templates__edit($documentId = null, $p_document_type_id = null)
    {
        $l_rules  = [];
        $auth   = isys_auth_document::instance();
        $l_navbar = isys_component_template_navbar::getInstance();

        if (!$auth->has('templates') && !$auth->has('templates_in_categories')) {
            throw new isys_exception_auth($this->language->get('LC__AUTH__MANUAL_EXCEPTION__MISSING_RIGHT_FOR_TEMPLATES'));
        }

        $allowedToEdit = $auth->is_allowed_to(isys_auth::EDIT, 'TEMPLATES/' . isys_auth::EMPTY_ID_PARAM);
        $allowedToCreate = $auth->is_allowed_to(isys_auth::CREATE, 'TEMPLATES/' . isys_auth::EMPTY_ID_PARAM);

        if ($documentId !== null) {
            $auth->check(isys_auth::VIEW, 'TEMPLATES/' . $documentId);
            $allowedToEdit = $auth->is_allowed_to(isys_auth::EDIT, 'TEMPLATES/' . $documentId);
        }

        $this->template->assign('templateID', $documentId);

        if ($documentId !== null && $documentId > 0) {
            $l_host = isys_factory::get_instance('isys_document_dao_templates', $this->database)
                ->get_data($documentId)
                ->get_row();

            // Define rules.
            $l_rules['template_id']['p_strValue']            = $documentId;
            $l_rules['C__DOCUMENT__TITLE']['p_strValue']     = $l_host['isys_document_template__title'];
            $l_rules['C__DOCUMENT__TYPE']['p_strSelectedID'] = $l_host['isys_document_template__isys_document_type__id'];
            $l_rules['C__TEMPLATE__INDEX']['p_strValue']     = $l_host['isys_document_template__index'];
            $l_rules['C__TEMPLATE__HEADER']['p_strValue']    = $l_host['isys_document_template__header'];
            $l_rules['C__TEMPLATE__FOOTER']['p_strValue']    = $l_host['isys_document_template__footer'];

            // Get chapters of document.
            $l_document_chapterRes = $this->m_dao_document_chapters->get_data(null, $documentId);

            if ($l_document_chapterRes->num_rows()) {
                $l_chapterTree = $this->m_dao_document_chapters->get_chapter_tree($this->m_dao_document_chapters->get_data(null, $documentId));

                if (!is_array($l_chapterTree)) {
                    $l_chapterTree = [];
                }

                $this->template->assign('chapterTree', $l_chapterTree);
            } // if

            // Add preview button
            $l_navbar = isys_component_template_navbar::getInstance();

            // Build link
            $l_preview_link = isys_helper_link::create_url([
                C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
                C__GET__SETTINGS_PAGE => self::CL__ACTION__TEMPLATE_PREVIEW,
                'id'                  => $documentId
            ]);

            $l_back_link    = isys_helper_link::create_url([
                C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
                C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENT_TEMPLATES,
            ]);

            $params = base64_encode(json_encode([
                'multiselection' => false,
                'name' => 'template_preview',
                isys_popup_browser_object_ng::C__CALLBACK__ACCEPT => 'preview_template(\'' . $l_preview_link . '\' + \'&objectId=\' + selection[0]);'
            ]));

            $l_navbar->append_button('LC__MODULE__DOCUMENT__DOCUMENT__TEMPLATE_PREVIEW', 'template_preview', [
                'tooltip'    => 'LC__MODULE__DOCUMENT__DOCUMENT__TEMPLATE_CREATE_PREVIEW_FOR_TEMPLATE',
                'active'     => true,
                'visible'    => true,
                'icon'       => 'icons/silk/zoom.png',
                'navmode'    => 'template_preview',
                'js_onclick' => "get_popup('browser_object_ng', '', '882', '550', {params: '$params'}, '');"
            ]);

            $cancelJs = 'if (! confirm(\'' . $this->language->get('LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE_LIST__CONFIRM_CANCEL') . '\')) {return false;}; ' .
                'window.location = \'' . $l_back_link . '\';';
            $l_navbar->set_js_onclick($cancelJs, C__NAVBAR_BUTTON__CANCEL);
        } else {
            $l_rules['C__DOCUMENT__TYPE']['p_strSelectedID'] = $p_document_type_id;
        } // if

        $l_defaultPDFFormatter = isys_tenantsettings::get('document.default-pdf-formatter', 'isys_document_format_pdf');

        if (!class_exists($l_defaultPDFFormatter)) {
            $l_defaultPDFFormatter = 'isys_document_format_pdf';
        }

        $l_formatOptionDefinition = array_merge($this->getFormatOptionDefinition(), (new $l_defaultPDFFormatter)->getFormatOptionDefinition());

        $l_font_directory = isys_document_format_pdf::getFontUploadDir();

        $saveTreeUrl = isys_helper_link::create_url([
            C__GET__MODULE_ID => C__MODULE__DOCUMENT,
            C__GET__AJAX      => 1,
            C__GET__AJAX_CALL => 'document',
            'method'          => 'saveChapterSorting'
        ]);

        $documentEditUrl = isys_helper_link::create_url([
            C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
            C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENT_TEMPLATES,
            'action'              => 'edit',
            C__GET__ID            => ''
        ]);

        $chapterEditUrl = isys_helper_link::create_url([
            C__GET__MODULE_ID             => C__MODULE__DOCUMENT,
            C__GET__SETTINGS_PAGE         => self::CL__ACTION__CHAPTERS,
            'action'                      => 'edit',
            C__DOCUMENT__GET__DOCUMENT_ID => $documentId,
            C__GET__ID                    => ''
        ]);

        $subChapterAddUrl = isys_helper_link::create_url([
            C__GET__MODULE_ID             => C__MODULE__DOCUMENT,
            C__GET__SETTINGS_PAGE         => self::CL__ACTION__CHAPTERS,
            'documentTypeID'              => $p_document_type_id,
            'action'                      => 'edit',
            C__DOCUMENT__GET__DOCUMENT_ID => $documentId,
            'parentChapterID'             => ''
        ]);

        // Disable Editmode for the moment.
        $_GET[C__CMDB__GET__EDITMODE] = 0;

        // Button-Rights Handling.
        $l_navbar->set_active(false, C__NAVBAR_BUTTON__CANCEL)
            ->set_js_onclick(
                "CKEDITOR.instances['C__TEMPLATE__INDEX'].updateElement(); CKEDITOR.instances['C__TEMPLATE__HEADER'].updateElement(); CKEDITOR.instances['C__TEMPLATE__FOOTER'].updateElement(); $('navMode').setValue('" .
                C__NAVMODE__SAVE . "'); $('isys_form').submit(); ",
                C__NAVBAR_BUTTON__SAVE
            )
            ->append_button(
                $this->language->get('LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__TEMPLATE_EXPORT_BUTTON'),
                'template_export',
                [
                    'navmode' => 'template_export',
                    'tooltip' => $this->language->get('LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__TEMPLATE_EXPORT_TOOLTIP'),
                    'icon'    => 'icons/silk/database_copy.png'
                ]
            );

        $customFieldData = [];
        $customFieldTypes = [];

        if ($documentId) {
            if ($allowedToEdit) {
                $this->template->activate_editmode();

                $l_navbar->set_active(true, C__NAVBAR_BUTTON__SAVE);
                $l_navbar->set_active(true, C__NAVBAR_BUTTON__CANCEL);
            }

            $customFields = (new CustomFieldsRepository())->getCustomFieldsForTemplate($documentId);

            foreach ($customFields as $customField) {
                $customFieldData[] = [
                    'id'           => $customField->getId(),
                    'name'         => $customField->getName(),
                    'dataType'     => str_replace('\\', '.', $customField->getDataType())
                ];
            }
        } elseif ($allowedToCreate) {
            $this->template->activate_editmode();

            $l_navbar->set_active(true, C__NAVBAR_BUTTON__SAVE);
            $l_navbar->set_active(true, C__NAVBAR_BUTTON__CANCEL);
        }

        // DOKU-273 Prepare the custom-field-type format, for the GUI.
        foreach (CustomFieldsRepository::getAvailableValueTypes() as $customFieldValueName => $customFieldValueClass) {
            $customFieldTypes[str_replace('\\', '.', get_class($customFieldValueClass))] = $this->language->get($customFieldValueName);
        }

        asort($customFieldTypes);

        $backgroundImageOptions = [
            isys_document_format_pdf::C__PAGE_TYPE__INDEX => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__UPLOAD_BACKGROUND_TYPE_INDEX',
            isys_document_format_pdf::C__PAGE_TYPE__TOC => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__UPLOAD_BACKGROUND_TYPE_TOC',
            isys_document_format_pdf::C__PAGE_TYPE__CONTENT => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__UPLOAD_BACKGROUND_TYPE_CONTENT'
        ];

        $this->template
            ->assign('ajaxFontUploadUrl', isys_helper_link::create_url([C__GET__AJAX => 1, C__GET__AJAX_CALL => 'document', 'method' => 'import_ttf_font']))
            ->assign('ajaxBackgroundUploadUrl', isys_helper_link::create_url([C__GET__AJAX => 1, C__GET__AJAX_CALL => 'document', 'method' => 'uploadBackgroundImage']))
            ->assign('upload_dir_writable', is_writable($l_font_directory))
            ->assign('upload_ttf_not_writable', $this->language->get('LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__UPLOAD_TTF_DIR_NOT_WRITABLE', str_replace(BASE_DIR, '&lt;i-doit&gt;/', $l_font_directory)))
            ->assign('upload_ttf_description', $this->language->get('LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__UPLOAD_TTF_DESCRIPTION', str_replace(BASE_DIR, '&lt;i-doit&gt;/', $l_font_directory)))
            ->assign('documentOptionDefinition', $l_formatOptionDefinition)
            ->assign('documentOptions', $this->getFormatOptions($documentId))
            ->assign('backgroundImageOptions', $backgroundImageOptions)
            ->assign('uploadDirectory', self::getWwwPath() . 'resources/upload/')
            ->assign('saveTreeUrl', $saveTreeUrl)
            ->assign('chapterEditUrl', $chapterEditUrl)
            ->assign('documentEditUrl', $documentEditUrl)
            ->assign('subchapterAddUrl', $subChapterAddUrl)
            ->assign('ckeditor_config', $this->getWysiwygConfiguration())
            ->assign('custom_fields_types', $customFieldTypes)
            ->assign('custom_fields_data', $customFieldData)
            ->assign('allowedToEdit', $allowedToEdit)
            ->smarty_tom_add_rules('tom.content.bottom.content', $l_rules)
            ->include_template('contentbottomcontent', self::getPath() . 'templates/templates/templates_form.tpl');
    }

    /**
     * Edit-action for displaying the chapter form. This is used for "NEW" and "EDIT".
     *
     * @param   integer $p_chapter_id
     * @param   integer $p_document_id
     *
     * @author  Selcuk Kekec <skekec@i-doit.com>
     */
    protected function process_chapters__edit($p_chapter_id = null, $p_document_id)
    {
        global $index_includes;

        $l_host = $l_rules = [];

        $l_rules['C__CHAPTER__NEW_PAGE']['p_strSelectedID'] = 0;

        // @see DOKU-289 Define the document ID.
        $l_rules['template_id']['p_strValue'] = $p_document_id;

        // ID given?
        if ($p_chapter_id !== null && $p_chapter_id > 0) {
            $l_host = isys_factory::get_instance('isys_document_dao_chapters', $this->database)
                ->get_data($p_chapter_id)
                ->get_row();

            // Define rules.
            $l_rules['chapter_id']['p_strValue'] = $p_chapter_id;
            $l_rules['component_id']['p_strValue'] = $l_host['isys_document_component__id'];
            $l_rules['C__CHAPTER__TITLE']['p_strValue'] = $l_host['isys_document_chapter__title'];
            $l_rules['C__CHAPTER__DOCUMENT']['p_strSelectedID'] = $l_host['isys_document_chapter__isys_document_template__id'];
            $l_rules['C__CHAPTER__PARENT']['p_strSelectedID'] = $l_host['isys_document_chapter__isys_document_chapter__id'];
            $l_rules['C__CHAPTER__TEMPLATE_COMPONENT']['p_strSelectedID'] = $l_host['isys_document_component__isys_document_component__id'];
            $l_rules['C__CHAPTER__TEXT']['p_strValue'] = $l_host['text'];
            $l_rules['C__CHAPTER__NEW_PAGE']['p_strSelectedID'] = $l_host['isys_document_chapter__new_page'];
            $this->template->assign('chapter_text', $l_host['isys_document_component__text']);
        }

        // Parent-Chapter.
        if (isset($_GET['parentChapterID'])) {
            $l_rules['C__CHAPTER__PARENT']['p_strSelectedID'] = $_GET['parentChapterID'];
        }

        // Building array for the parent dialogbox.
        $l_chapterArr = [];
        $l_chaptersData = $this->m_dao_document_chapters->get_chapter_tree_in_single_level($p_document_id);

        if (count($l_chaptersData)) {
            foreach ($l_chaptersData as $s_chapterData) {
                $l_chapterArr[$s_chapterData['id']] = $s_chapterData['position'] . ' ' . $s_chapterData['title'];
            }
        }

        // Get selectable components.
        $l_componentArr = [];
        $l_template_res = $this->m_dao_document_templates->get_data();

        while ($l_template_row = $l_template_res->get_row()) {
            // We use this to display chapter numbers.
            $l_chapters = $this->m_dao_document_chapters->get_chapter_tree_in_single_level($l_template_row['isys_document_template__id']);

            if (is_array($l_chapters) && count($l_chapters)) {
                foreach ($l_chapters as $l_chapter) {
                    $l_componentArr[$l_chapter['id']] = '[' . $l_template_row['isys_document_template__title'] . '] ' . $l_chapter['position'] . ' ' . $l_chapter['title'];
                }
            }
        }

        // Unset self to prevent loops.
        if (isset($l_host['isys_document_chapter__isys_document_component__id'])) {
            unset($l_componentArr[$l_host['isys_document_chapter__isys_document_component__id']]);
        }

        $l_rules['C__CHAPTER__TEMPLATE_COMPONENT']['p_arData'] = $l_rules['C__CHAPTER__TEMPLATE_COMPONENT_COPY']['p_arData'] = $l_componentArr;
        $l_rules['C__CHAPTER__TEMPLATE_COMPONENT']['p_bSort'] = $l_rules['C__CHAPTER__TEMPLATE_COMPONENT_COPY']['p_bSort'] = false;
        $l_rules['C__CHAPTER__NEW_PAGE']['p_arData'] = get_smarty_arr_YES_NO();

        // Unset self to prevent loops.
        unset($l_chapterArr[$p_chapter_id]);

        $l_rules['C__CHAPTER__PARENT']['p_bSort'] = false;
        $l_rules['C__CHAPTER__PARENT']['p_arData'] = $l_chapterArr;

        $this->template->smarty_tom_add_rules('tom.content.bottom.content', $l_rules);

        // Deactivate editmode.
        $_GET[C__CMDB__GET__EDITMODE] = 0;

        $l_navbar = isys_component_template_navbar::getInstance()
            ->set_js_onclick(
                "CKEDITOR.instances['C__CHAPTER__TEXT'].updateElement(); $('navMode').setValue('" . C__NAVMODE__SAVE . "'); save_via_ajax('ajaxReturnNote');",
                C__NAVBAR_BUTTON__SAVE
            );

        $this->template->activate_editmode();

        $l_navbar->set_active(true, C__NAVBAR_BUTTON__CANCEL);
        $l_navbar->set_active(true, C__NAVBAR_BUTTON__SAVE);
        $l_back_link = isys_helper_link::create_url([
                C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
                C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENT_TEMPLATES,
                'action'              => 'edit',
                'id'                  => $p_document_id
            ]);
        $cancelJs = 'if (! confirm(\'' . $this->language->get('LC__MODULE__DOCUMENT__CHAPTER__CONFIRM_CANCEL') . '\')) {return false;}; ';
        $cancelJs .= 'window.location = \'' . $l_back_link . '\';';
        $l_navbar->set_js_onclick($cancelJs, C__NAVBAR_BUTTON__CANCEL);

        $this->template->assign('ckeditor_config', $this->getWysiwygConfiguration());
        $index_includes['contentbottomcontent'] = self::get_tpl_dir() . 'chapters/chapters_form.tpl';
    }

    /**
     * List-action for displaying the documents.
     *
     * @throws  isys_exception_auth
     * @throws  isys_exception_database
     * @author  Selcuk Kekec <skekec@i-doit.com>
     */
    protected function process_documents__list()
    {
        $auth = isys_auth_document::instance();

        $l_list_headers = [
            'isys_document__id'             => 'LC__MODULE__DOCUMENT__DOCUMENT_LIST__ID',
            'isys_document__title'          => 'LC__MODULE__DOCUMENT__DOCUMENT__DOCUMENT_TEMPLATE_TITLE',
            'isys_obj__title'               => 'LC__MODULE__DOCUMENT__CONNECTED_OBJECT',
            'isys_document_template__title' => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE',
            'isys_document_type__title'     => 'LC__MODULE__DOCUMENT__DOCUMENT__TYPE',
            'isys_document__created'        => 'LC__CMDB__CATG__DOCUMENT__CREATION_REFRESH_DATE',
            'export_buttons'                => 'LC__MODULE__DOCUMENT__DOCUMENT__EXPORT_AS'
        ];

        if (!$auth->has('documents') && !$auth->has('documents_in_categories')) {
            throw new isys_exception_auth($this->language->get('LC__AUTH__MANUAL_EXCEPTION__MISSING_RIGHT_FOR_DOCUMENTS'));
        }

        $documentCategoryId = (isset($_GET['documentTypeID'])) ? $_GET['documentTypeID'] : null;
        $documentResult = $this->m_dao_documents->get_data(null, $documentCategoryId);
        $l_documents_count = count($documentResult);

        $allowedToCreateDocuments = $auth->is_allowed_to(isys_auth::CREATE, 'DOCUMENTS/' . isys_auth::EMPTY_ID_PARAM);
        $allowedToEditDocuments   = $auth->is_allowed_to(isys_auth::EDIT, 'DOCUMENTS/' . isys_auth::EMPTY_ID_PARAM);
        $allowedToDeleteDocuments = $auth->is_allowed_to(isys_auth::DELETE, 'DOCUMENTS/' . isys_auth::EMPTY_ID_PARAM);

        if ($documentCategoryId) {
            $allowedToCreateDocuments = $auth->is_allowed_to(isys_auth::CREATE, 'DOCUMENTS_IN_CATEGORIES/' . $documentCategoryId);
        }

        // Check for rights on document types
        // array_values is necessary because isys_component_list will literally access first index 0
        $l_documents = array_values(array_filter($documentResult->__as_array(), function ($row) use ($auth, &$allowedToEditDocuments, &$allowedToDeleteDocuments) {
            if (!$auth->is_allowed_to(isys_auth::VIEW, 'DOCUMENTS/' . $row['isys_document__id'])) {
                return false;
            }

            if ($auth->is_allowed_to(isys_auth::EDIT, 'DOCUMENTS/' . $row['isys_document__id'])) {
                $allowedToEditDocuments = true;
            }

            if ($auth->is_allowed_to(isys_auth::DELETE, 'DOCUMENTS/' . $row['isys_document__id'])) {
                $allowedToDeleteDocuments = true;
            }

            return true;
        }));

        $l_link = isys_helper_link::create_url([
            C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
            C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENTS,
            'action'              => 'edit',
            'id'                  => '[{isys_document__id}]'
        ]);

        $l_list = new isys_component_list($l_documents, null, null, C__RECORD_STATUS__NORMAL);
        $l_list->set_row_modifier($this, 'document_list_modifier')
            ->config($l_list_headers, $l_link, '[{isys_document__id}]');

        if ($l_list->createTempTable()) {
            $this->template->assign('configuration_table', $l_list->getTempTableHtml());
        }

        $l_navbar = isys_component_template_navbar::getInstance()
            ->set_active($allowedToCreateDocuments, C__NAVBAR_BUTTON__NEW)
            ->set_active(($l_documents_count > 0 && $allowedToEditDocuments), C__NAVBAR_BUTTON__EDIT)
            ->set_active(($l_documents_count > 0 && $allowedToDeleteDocuments), C__NAVBAR_BUTTON__DELETE)
            ->set_visible(true, C__NAVBAR_BUTTON__NEW)
            ->set_visible(true, C__NAVBAR_BUTTON__EDIT)
            ->set_visible(true, C__NAVBAR_BUTTON__DELETE)
            ->set_js_onclick(
                "get_popup('document_create_document', 'editMode=1&documentTypeID=" . $_GET['documentTypeID'] . "', 1000, 430, null, 'popup_commentary')",
                C__NAVBAR_BUTTON__NEW
            )
            ->add_onclick_prepend(C__NAVBAR_BUTTON__DELETE, 'if (!$(\'scroller\').select(\'input[type = "checkbox"]:checked\').length || ! confirm(\'' . $this->language->get('LC__MODULE__DOCUMENT__DOCUMENT_LIST__CONFIRM_DELETE') . '\')) { $(\'navbar_item_C__NAVMODE__DELETE\').removeClassName(\'navbar_item_inactive\').down(\'img\').writeAttribute(\'src\', window.dir_images + \'icons/silk/page_delete.png\'); return false;}')
            ->append_button('LC__MODULE__DOCUMENT__DOCUMENT__REGENERATE_DOCUMENT', 'regenerate_document', [
                'active'     => true,
                'visible'    => true,
                'icon'       => 'icons/silk/arrow_refresh.png',
                'js_onclick' => ';',
                'navmode'    => 'regenerate_document'
            ]);

        $this->template->include_template('contentbottomcontent', self::get_tpl_dir() . 'documents/documents_list.tpl');
    }

    /**
     * Row-Modifier for the documents list
     *
     * @param   array &$p_row
     *
     * @author  Selcuk Kekec <skekec@i-doit.com>
     */
    public function document_list_modifier(&$p_row)
    {
        global $g_dirs;

        // Format date of creation to user defined format.
        $p_row['isys_document__created'] = isys_application::instance()->container->get('locales')->fmt_datetime($p_row['isys_document__created']);

        // Link to connected object.
        if ($p_row['isys_obj__id']) {
            $p_row['isys_obj__title'] = '<a href="?' . C__CMDB__GET__OBJECT . '=' . $p_row['isys_obj__id'] . '">' . '<img src="' . $g_dirs["images"] .
                'icons/silk/link.png" alt="Link" class="vam mr5" /><span class="vam">' . $p_row['isys_obj__title'] . '</span></a>';
        } else {
            $p_row['isys_obj__title'] = "<i style='color:grey'>" . $this->language->get('LC__MODULE__DOCUMENT__NO_CONNECTED_OBJECT') . "</i>";
        } // if

        /* Link to document template */
        if (!is_null($p_row['isys_document_template__id'])) {
            $p_row['isys_document_template__title'] = '<a href="' . isys_helper_link::create_url(
                    [
                        C__GET__MODULE_ID      => C__MODULE__DOCUMENT,
                        C__GET__SETTINGS_PAGE  => self::CL__ACTION__DOCUMENT_TEMPLATES,
                        'action'               => 'edit',
                        'id'                   => $p_row['isys_document_template__id'],
                        C__CMDB__GET__TREEMODE => C__MODULE__DOCUMENT . '00000' . $p_row['isys_document_type__id'],
                    ]
                ) . '">
                            <img src="' . $g_dirs["images"] . 'icons/silk/link.png" alt="Link" class="vam" /> <span class="vam">' . $p_row['isys_document_template__title'] .
                '</span></a>';
        } else {
            $p_row['isys_document_template__title'] = '<i style="color:grey">' . $this->language->get('LC__MODULE__DOCUMENT__NO_LINKED_TEMPLATE') . '</i>';
        }

        $l_links = static::get_export_links($p_row['isys_document__id']);

        // Export-Links.
        $p_row['export_buttons'] = '<a class="btn btn-small mr5" href="' . $l_links['html_download'] . '">' .
            '<img src="images/icons/silk/page_white_code.png" class="mr5" /><span>' . $this->language->get('LC__MODULE__DOCUMENT__FORMAT__HTML') . '</span>' .
            '</a><a class="btn btn-small mr5" href="' . $l_links['pdf_download'] . '">' .
            '<img src="images/icons/silk/page_white_acrobat.png" class="mr5"><span>' . $this->language->get('LC__MODULE__DOCUMENT__FORMAT__PDF') . '</span>' .
            '</a><a class="btn btn-small" href="' . $l_links['pdf_inline'] . '">' .
            '<img src="images/icons/silk/page_white_acrobat.png" class="mr5"><span>' .
                $this->language->get('LC__MODULE__DOCUMENT__FORMAT__PDF') . ' ' . $this->language->get('LC__MODULE__DOCUMENT__EXPORT_INLINE') .
            '</span></a>';
    }

    /**
     * List-action for displaying the documents
     *
     * @author  Selcuk Kekec <skekec@i-doit.com>
     */
    protected function process_document_templates__list()
    {
        $auth = isys_auth_document::instance();

        $l_list_headers = [
            'isys_document_template__id'         => 'LC__MODULE__DOCUMENT__DOCUMENT_LIST__ID',
            'isys_document_template__title'      => 'LC__MODULE__DOCUMENT__DOCUMENT_LIST__TITLE',
            'isys_document_type__title'          => 'LC__MODULE__DOCUMENT__DOCUMENT_LIST__TYPE',
            'isys_document_template__created_at' => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__CREATED_DATE',
            'isys_document_template__updated_at' => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__LAST_CHANGE'
        ];

        if (!$auth->has('templates') && !$auth->has('templates_in_categories')) {
            throw new isys_exception_auth($this->language->get('LC__AUTH__MANUAL_EXCEPTION__MISSING_RIGHT_FOR_TEMPLATES'));
        }

        $documentCategoryId = (isset($_GET['documentTypeID'])) ? $_GET['documentTypeID'] : null;
        $templateResult   = $this->m_dao_document_templates->get_data(null, $documentCategoryId);
        $templateCount = count($templateResult);

        $allowedToCreateTemplates = $auth->is_allowed_to(isys_auth::CREATE, 'TEMPLATES/' . isys_auth::EMPTY_ID_PARAM);
        $allowedToEditTemplates   = $auth->is_allowed_to(isys_auth::EDIT, 'TEMPLATES/' . isys_auth::EMPTY_ID_PARAM);
        $allowedToDeleteTemplates = $auth->is_allowed_to(isys_auth::DELETE, 'TEMPLATES/' . isys_auth::EMPTY_ID_PARAM);

        if ($documentCategoryId) {
            $allowedToCreateTemplates = $auth->is_allowed_to(isys_auth::CREATE, 'TEMPLATES_IN_CATEGORIES/' . $documentCategoryId);
        }

        $templateList = array_values(array_filter($templateResult->__as_array(), function ($row) use ($auth, &$allowedToEditTemplates, &$allowedToDeleteTemplates) {
            if (!$auth->is_allowed_to(isys_auth::VIEW, 'TEMPLATES/' . $row['isys_document_template__id'])) {
                return false;
            }

            if ($auth->is_allowed_to(isys_auth::EDIT, 'TEMPLATES/' . $row['isys_document_template__id'])) {
                $allowedToEditTemplates = true;
            }

            if ($auth->is_allowed_to(isys_auth::DELETE, 'TEMPLATES/' . $row['isys_document_template__id'])) {
                $allowedToDeleteTemplates = true;
            }

            return true;
        }));

        $l_list_url = isys_helper_link::create_url([
            C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
            C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENT_TEMPLATES,
            'action'              => 'edit',
            'id'                  => '[{isys_document_template__id}]'
        ]);

        $l_list = isys_component_list::factory($templateList)
            ->config($l_list_headers, $l_list_url, '[{isys_document_template__id}]')
            ->set_row_modifier($this, 'documentTemplateListModifyRow');

        if ($l_list->createTempTable()) {
            $this->template->assign('configuration_table', $l_list->getTempTableHtml());
        }

        $tree = isys_component_tree::factory();

        (new isys_module_dialog_admin())->build_tree($tree);

        $editCategoriesLink = isys_helper_link::create_url([
            C__GET__MODULE_ID     => C__MODULE__SYSTEM,
            C__GET__MODULE_SUB_ID => C__MODULE__DIALOG_ADMIN,
            C__GET__TREE_NODE     => $tree->find_id_by_title(isys_helper_textformat::clean_string(strip_tags($this->language->get('isys_document_type')))),
            'table'              => 'isys_document_type'
        ]);

        isys_component_template_navbar::getInstance()
            ->set_active($allowedToCreateTemplates, C__NAVBAR_BUTTON__NEW)
            ->set_active(($templateCount > 0 && $allowedToEditTemplates), C__NAVBAR_BUTTON__EDIT)
            ->set_active(($templateCount > 0 && $allowedToDeleteTemplates), C__NAVBAR_BUTTON__DELETE)
            ->set_visible(true, C__NAVBAR_BUTTON__NEW)
            ->set_visible(true, C__NAVBAR_BUTTON__EDIT)
            ->set_visible(true, C__NAVBAR_BUTTON__DELETE)
            ->append_button(
                'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__TEMPLATE_IMPORT_BUTTON',
                'template_import',
                [
                    'navmode' => 'template_import',
                    'tooltip' => $this->language->get('LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__TEMPLATE_IMPORT_TOOLTIP'),
                    'icon'    => 'icons/silk/database_copy.png'
                ]
            )
            ->append_button(
                'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__TEMPLATE_DUPLICATE_BUTTON',
                'template_duplicate',
                [
                    'navmode' => 'template_duplicate',
                    'tooltip' => $this->language->get('LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__TEMPLATE_DUPLICATE_TOOLTIP'),
                    'icon'    => 'icons/silk/page_copy.png'
                ]
            )
            ->append_button(
                'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__TEMPLATE_EDIT_CATEGORIES_BUTTON',
                'edit_categories',
                [
                    'url' => $editCategoriesLink,
                    'tooltip' => $this->language->get('LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__TEMPLATE_EDIT_CATEGORIES_BUTTON_TOOLTIP'),
                    'icon'    => 'icons/silk/book_edit.png'
                ]
            )
            ->add_onclick_prepend(C__NAVBAR_BUTTON__DELETE, 'if (!$(\'scroller\').select(\'input[type="checkbox"]:checked\').length || !confirm(\'' . $this->language->get('LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE_LIST__CONFIRM_DELETE') . '\')) { $(\'navbar_item_C__NAVMODE__DELETE\').removeClassName(\'navbar_item_inactive\').down(\'img\').writeAttribute(\'src\', window.dir_images + \'icons/silk/page_delete.png\'); return false; }');

        $this->template->include_template('contentbottomcontent', self::getPath() . 'templates/templates/templates_list.tpl');
    }

    /**
     * Modify row method for template list.
     *
     * @param array &$row
     *
     * @see   DOKU-245  Set "updated" when saving a chapter.
     */
    public function documentTemplateListModifyRow(&$row)
    {
        $locales = isys_application::instance()->container->get('locales');
        $dao = isys_cmdb_dao::instance($this->database);
        $quickinfo = new isys_ajax_handler_quick_info();

        $row['created'] = $row['updated'] = '<em class="text-grey">' . isys_tenantsettings::get('gui.empty_value', '-') . '</em>';

        if ($row['isys_document_template__created_at'] !== null) {
            $user = $row['created'];

            if ($row['isys_document_template__created_by'] !== null) {
                $user = $quickinfo->get_quick_info(
                    $row['isys_document_template__created_by'],
                    $dao->get_obj_name_by_id_as_string($row['isys_document_template__created_by']),
                    C__LINK__OBJECT
                );
            }

            $row['created'] = $locales->fmt_datetime($row['isys_document_template__created_at']) . ' (' . $user . ')';
        }

        if ($row['isys_document_template__updated_at'] !== null) {
            $user = $row['updated'];

            if ($row['isys_document_template__updated_by'] !== null) {
                $user = $quickinfo->get_quick_info(
                    $row['isys_document_template__updated_by'],
                    $dao->get_obj_name_by_id_as_string($row['isys_document_template__updated_by']),
                    C__LINK__OBJECT
                );
            }

            $row['updated'] = $locales->fmt_datetime($row['isys_document_template__updated_at']) . ' (' . $user . ')';
        }
    }

    /**
     * List-action for displaying the chapters
     *
     * @author  Selcuk Kekec <skekec@i-doit.com>
     */
    protected function process_chapters__list()
    {
        global $index_includes;
        $l_list_headers = [
            'isys_document_chapter__id'    => 'LC__MODULE__DOCUMENT__CHAPTER_LIST__ID',
            'isys_document_chapter__title' => 'LC__MODULE__DOCUMENT__CHAPTER_LIST__TITLE',
            'isys_chapter_pattern__title'  => 'LC__MODULE__DOCUMENT__CHAPTER_LIST__PATTERN',
        ];

        $l_documentsRes    = $this->m_dao_document_chapters->get_data(null, $_GET[C__DOCUMENT__GET__DOCUMENT_ID]);
        $l_documents_count = $l_documentsRes->num_rows();

        /**
         * @var $l_list isys_component_list
         */
        $l_list = isys_factory::get_instance('isys_component_list')
            ->set_data(null, $l_documentsRes)
            ->config(
                $l_list_headers,
                "?" . C__GET__MODULE_ID . "=" . C__MODULE__DOCUMENT . "&" . C__GET__SETTINGS_PAGE . "=" . self::CL__ACTION__CHAPTERS .
                "&action=edit&id=[{isys_document_chapter__id}]",
                '[{isys_document_chapter__id}]'
            );

        if ($l_list->createTempTable()) {
            $this->template->assign('configuration_table', $l_list->getTempTableHtml());
        } // if

        isys_component_template_navbar::getInstance()
            ->set_active(true, C__NAVBAR_BUTTON__NEW)
            ->set_active(($l_documents_count > 0), C__NAVBAR_BUTTON__EDIT)
            ->set_active(($l_documents_count > 0), C__NAVBAR_BUTTON__DELETE)
            ->set_visible(true, C__NAVBAR_BUTTON__NEW)
            ->set_visible(true, C__NAVBAR_BUTTON__EDIT)
            ->set_visible(true, C__NAVBAR_BUTTON__DELETE);

        $index_includes['contentbottomcontent'] = self::get_tpl_dir() . 'chapters/chapters_list.tpl';
    }

    /**
     * Build breadcrumb navigation
     *
     * @param &$p_gets
     *
     * @return array|null
     */
    public function breadcrumb_get(&$p_gets)
    {
        /* Gets */
        $l_gets = $this->m_request->get_gets();

        /* Result-Array */
        $l_res = [];

        switch ($l_gets[C__GET__SETTINGS_PAGE]) {
            case self::CL__ACTION__DOCUMENT_TEMPLATES:
                /* Templates-Main-Node */
                $l_res[] = [
                    $this->language->get('LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATES') => [
                        C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
                        C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENT_TEMPLATES,
                    ]
                ];

                /* Templates-Type-Node */
                if (isset($l_gets['documentTypeID'])) {
                    $l_document_type_res = $this->m_dao_documents->retrieve(
                        "SELECT isys_document_type__title FROM isys_document_type" . " WHERE isys_document_type__id = " .
                        $this->m_dao_documents->convert_sql_id($l_gets['documentTypeID'])
                    );

                    if ($l_document_type_res->count()) {
                        $l_document_type_title = $this->language->get($l_document_type_res->get_row_value('isys_document_type__title'));

                        $l_res[] = [
                            $l_document_type_title => [
                                C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
                                C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENT_TEMPLATES,
                                'documentTypeID'      => $l_gets['documentTypeID']
                            ]
                        ];
                    }
                }

                /* Templates-Edit-Node */
                if (isset($l_gets[C__GET__ID])) {
                    $l_template_res = $this->m_dao_document_templates->get_data($l_gets[C__GET__ID]);

                    if ($l_template_res->count()) {
                        $l_template_row = $l_template_res->get_row();

                        $l_res[] = [
                            $this->language->get($l_template_row['isys_document_type__title']) => [
                                C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
                                C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENT_TEMPLATES,
                                'documentTypeID'      => $l_template_row['isys_document_type__id']
                            ]
                        ];

                        $l_res[] = [
                            $l_template_row['isys_document_template__title'] => [
                                C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
                                C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENT_TEMPLATES,
                                'action'              => 'edit',
                                'id'                  => $l_gets[C__GET__ID]
                            ]
                        ];
                    }
                }

                // Templates-New-Node.
                if (!isset($l_gets[C__GET__ID]) && $_POST['navMode'] == C__NAVMODE__EDIT) {
                    $l_res[] = [$this->language->get('LC__MODULE__DOCUMENT__NEW_TEMPLATE') => null];
                }

                break;
            case self::CL__ACTION__DOCUMENTS:
                /* Documents-Main-Node */
                $l_res[] = [
                    $this->language->get('LC__MODULE__DOCUMENT__DOCUMENTS') => [
                        C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
                        C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENTS,
                    ]
                ];

                /* Documents-Type-Node */
                if (isset($l_gets['documentTypeID'])) {
                    $l_type_res = $this->m_dao_documents->retrieve(
                        'SELECT isys_document_type__title 
                        FROM isys_document_type
                        WHERE isys_document_type__id = ' . $this->m_dao_documents->convert_sql_id($l_gets['documentTypeID']) . ';'
                    );

                    if ($l_type_res->count()) {
                        $l_type_row = $l_type_res->get_row();

                        $l_res[] = [
                            $this->language->get($l_type_row['isys_document_type__title']) => [
                                C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
                                C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENTS,
                                'documentTypeID'      => $l_gets['documentTypeID'],
                            ]
                        ];
                    }
                }

                /* Documents-Edit-Node */
                if (isset($l_gets['id'])) {
                    $l_document_res = $this->m_dao_documents->get_data($l_gets['id']);

                    if ($l_document_res->count()) {
                        $l_document_row = $l_document_res->get_row();

                        /* Documents-Type-Node */
                        $l_res[] = [
                            $this->language->get($l_document_row['isys_document_type__title']) => [
                                C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
                                C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENTS,
                                'documentTypeID'      => $l_document_row['isys_document_type__id'],
                            ]
                        ];

                        $l_res[] = [$l_document_row['isys_document_template__title'] => null];
                    }
                }

                break;
            case self::CL__ACTION__CHAPTERS:
                $l_res[] = [
                    $this->language->get('LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATES') => [
                        C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
                        C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENT_TEMPLATES,
                    ]
                ];

                /* Chapters-Node */
                if (isset($l_gets[C__DOCUMENT__GET__DOCUMENT_ID])) {
                    $l_template_res = $this->m_dao_document_templates->get_data($l_gets[C__DOCUMENT__GET__DOCUMENT_ID]);

                    if ($l_template_res->count()) {
                        $l_template_row = $l_template_res->get_row();
                        $l_res[]        = [
                            $this->language->get($l_template_row['isys_document_type__title']) => [
                                C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
                                C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENT_TEMPLATES,
                                'documentTypeID'      => $l_template_row['isys_document_type__id'],
                            ]
                        ];
                        $l_res[]        = [
                            $l_template_row['isys_document_template__title'] => [
                                C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
                                C__GET__SETTINGS_PAGE => self::CL__ACTION__DOCUMENT_TEMPLATES,
                                'action'              => 'edit',
                                'id'                  => $l_template_row['isys_document_template__id'],
                            ]
                        ];

                        if (isset($l_gets['parentChapterID'])) {
                            $l_chapter_res = $this->m_dao_document_chapters->get_data($l_gets['parentChapterID']);

                            if ($l_chapter_res->count()) {
                                $l_chapter_row = $l_chapter_res->get_row();

                                $l_res[] = [
                                    $l_chapter_row['isys_document_chapter__title'] => [
                                        C__GET__MODULE_ID             => C__MODULE__DOCUMENT,
                                        C__GET__SETTINGS_PAGE         => self::CL__ACTION__CHAPTERS,
                                        'action'                      => 'edit',
                                        'id'                          => $l_chapter_row['isys_document_chapter__id'],
                                        C__DOCUMENT__GET__DOCUMENT_ID => $l_gets[C__DOCUMENT__GET__DOCUMENT_ID]
                                    ]
                                ];
                            }
                        }

                        if (isset($l_gets['id'])) {
                            $l_chapter_res = $this->m_dao_document_chapters->get_data($l_gets['id']);

                            if ($l_chapter_res->count()) {
                                $l_chapter_row = $l_chapter_res->get_row();

                                $l_res[] = [$l_chapter_row['isys_document_chapter__title'] => null];
                            }
                        }

                        // Templates-New-Node.
                        if (!isset($l_gets[C__GET__ID]) && $_POST['navMode'] == C__NAVMODE__EDIT) {
                            $l_res[] = [$this->language->get('LC__MODULE__DOCUMENT__NEW_CHAPTER') => null];
                        }
                    }
                }

                break;
        }

        return $l_res;
    }

    /**
     * Method for downloading files.
     *
     * @param   array $p_files
     *
     * @return  array
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public static function download_files(array $p_files = [])
    {
        if (!count($p_files)) {
            return [];
        } // if

        $l_return = [];
        $l_use_curl = !ini_get('allow_url_fopen');

        foreach ($p_files as $l_file) {
            try {
                if ($l_use_curl) {
                    $l_curl = curl_init();

                    if (isys_settings::get('proxy.active', false)) {
                        //curl_setopt($l_curl, CURLOPT_HTTPPROXYTUNNEL, 1);
                        curl_setopt($l_curl, CURLOPT_PROXY, isys_settings::get('proxy.host') . ":" . isys_settings::get('proxy.port'));

                        if (isys_settings::get('proxy.username', false)) {
                            curl_setopt($l_curl, CURLOPT_PROXYUSERPWD, isys_settings::get('proxy.username') . ":" . isys_settings::get('proxy.password'));
                        }
                    }

                    curl_setopt($l_curl, CURLOPT_URL, $l_file);
                    curl_setopt($l_curl, CURLOPT_RETURNTRANSFER, 1);

                    if (substr($l_file, 0, 5) === 'https') {
                        curl_setopt($l_curl, CURLOPT_SSLVERSION, 3);
                        curl_setopt($l_curl, CURLOPT_SSL_VERIFYPEER, false);
                    }

                    $l_filecontent = curl_exec($l_curl);
                    $l_error = curl_error($l_curl);
                    curl_close($l_curl);

                    if (!empty($l_error)) {
                        throw new isys_exception_general('The file "' . $l_file . '" could not be downloaded.');
                    }

                    $l_return[$l_file] = $l_filecontent;
                } else {
                    // @see  DOKU-145  Temporary set an own error handler to prevent any PHP errors reaching the frontend.
                    $errorReporting = E_ALL & ~E_NOTICE;

                    if (defined('E_DEPRECATED')) {
                        $errorReporting &= ~E_DEPRECATED;
                    }

                    if (defined('E_STRICT')) {
                        $errorReporting &= ~E_STRICT;
                    }

                    set_error_handler(function ($severity, $message, $file, $line) {
                        throw new isys_exception_general($message . ' (' . $file . ' in line ' . $line . ')');
                    }, $errorReporting);

                    try {
                        $contextOptions = [
                            'ssl' => [
                                'verify_peer'      => false,
                                'verify_peer_name' => false
                            ]
                        ];

                        // @see DOKU-95 Include Proxy-data with the stream context.
                        if (isys_settings::get('proxy.active', false)) {
                            $contextOptions['http'] = [
                                'proxy'           => 'tcp://' . isys_settings::get('proxy.host') . ':' . isys_settings::get('proxy.port'),
                                'request_fulluri' => true
                            ];

                            if (!empty(isys_settings::get('proxy.username'))) {
                                $auth = base64_encode(isys_settings::get('proxy.username') . ':' . isys_settings::get('proxy.password'));
                                $contextOptions['http']['header'] = "Proxy-Authorization: Basic $auth";
                            }
                        }

                        // Alternative?
                        // stream_context_set_default($contextOptions);

                        $l_return[$l_file] = file_get_contents($l_file, false, stream_context_create($contextOptions));
                    } catch (isys_exception_general $e) {
                        $e->write_log();
                    }

                    restore_error_handler();
                }
            } catch (Exception $e) {
                $l_return[$l_file] = $e;
            }
        }

        return $l_return;
    }

    /**
     * Generate preview pdf for template
     *
     * @param   int $p_template_id
     *
     * @return isys_document_export_pdf|false
     * @throws \Exception
     * @throws \isys_exception_general
     * @author Selcuk Kekec <skekec@i-doit.com>
     */
    protected function process_document_templates__preview($p_template_id)
    {
        try {
            if (isset($_GET['objectId'])) {
                $document = new \idoit\Module\Document\Document\Document();
                $previewDocumentId = $document->create('Preview of ' . $_GET['objectId'], $p_template_id, 'Preview of ' . $_GET['objectId'], $_GET['objectId']);

                $this->process_document_export('pdf', $previewDocumentId, null, true, true);

                return false;
            }

            if (!$p_template_id || !is_numeric($p_template_id)) {
                throw new isys_exception_general('ID of document template is not setted or invalid.');
            }

            // Init.
            $l_dao_chapter    = isys_document_dao_chapters::instance($this->database);
            $l_dao_document   = isys_document_dao_templates::instance($this->database);
            $l_documentRow    = $l_dao_document->get_data($p_template_id)->get_row();
            $l_export_options = [];

            if (isys_format_json::is_json($l_documentRow['isys_document_template__options'])) {
                $l_export_options = isys_format_json::decode($l_documentRow['isys_document_template__options']);
            }

            if ($l_export_options['general.toc'] > 0) {
                $l_export_options['general.headlines'] = 1;
            }

            // 1. Prepare document model.
            $l_model_document = new isys_document_compiler_model_document($l_export_options);

            // 2. Load chapter models.
            $l_model_document->load($p_template_id, $l_dao_chapter);

            // 3. Get chapters.
            $l_compiler_chapter_models = $l_model_document->get_chapters();
            $l_export_chapter_models   = [];

            if (!count($l_compiler_chapter_models)) {
                throw new isys_exception_general($this->language->get('LC__MODULE__DOCUMENT__ERROR__NO_CHAPTERS_DEFINED'));
            }

            // 4. Transform compiler models into export models
            foreach ($l_compiler_chapter_models as $l_chapter_model) {
                // Create new export model
                $l_export_chapter_models[] = isys_document_export_model::factory()
                    ->set_level($l_chapter_model->get_level())
                    ->set_pos($l_chapter_model->get_chapter_num())
                    ->set_text($l_chapter_model->get_raw_text())
                    ->set_title($l_chapter_model->get_title())
                    ->set_new_page($l_chapter_model->get_new_page());
            }

            // 5. Initialize the pdf exporter
            $l_defaultPDFFormatter = isys_tenantsettings::get('document.default-pdf-formatter', 'isys_document_format_pdf');
            if (!class_exists($l_defaultPDFFormatter)) {
                $l_defaultPDFFormatter = 'isys_document_format_pdf';
            }

            // Create the exporter and inject the lib.
            $l_export = new isys_document_export_pdf(
                $l_export_chapter_models,
                new $l_defaultPDFFormatter()
            );

            // 6. Export the document
            $l_export->set_title($l_documentRow['isys_document_template__title'])
                ->set_style('.document-wysiwyg-placeholder {font-style: italic;}')
                ->set_options($l_export_options + [
                        'index'           => $l_documentRow['isys_document_template__index'],
                        'header'          => $l_documentRow['isys_document_template__header'],
                        'footer'          => $l_documentRow['isys_document_template__footer'],
                        'revisionId'      => 0,
                        'revisionComment' => ''
                    ])
                ->export()
                ->send($l_documentRow['isys_document_template__title'] . '.' . $l_export->get_file_extension(), isset($_GET['inline']));

            return $l_export;
        } catch (Exception $e) {
            global $g_error;

            isys_notify::error($this->language->get('LC__MODULE__DOCUMENT__ERROR__EXPORT', $e->getMessage()));
            $g_error = $this->language->get('LC__MODULE__DOCUMENT__ERROR__EXPORT', $e->getMessage());

            $this->process_documents__edit($_GET['id']);
        }

        return false;
    }

    /**
     * Uniform method for generating document export links.
     *
     * @param   integer $documentId
     * @param   integer $revisionId
     *
     * @return  array
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public static function get_export_links($documentId, $revisionId = null)
    {
        $baseUrl = [
            C__GET__MODULE_ID             => C__MODULE__DOCUMENT,
            C__GET__SETTINGS_PAGE         => 'document_export',
            C__DOCUMENT__GET__DOCUMENT_ID => $documentId
        ];

        if ($revisionId !== null) {
            $baseUrl[C__DOCUMENT__GET__REVISION_ID] = $revisionId;
        }

        return [
            'html_download' => isys_helper_link::create_url($baseUrl + ['type' => 'html']),
            'pdf_download'  => isys_helper_link::create_url($baseUrl + ['type' => 'pdf']),
            // 'docx_download' => isys_helper_link::create_url($baseUrl + array('type' => 'docx')), @todo  Jira ticket: DOKU-18
            'pdf_inline'    => isys_helper_link::create_url($baseUrl + ['type' => 'pdf', 'inline' => 1])
        ];
    }

    /**
     * Get related auth class for module.
     *
     * @return  isys_auth_document
     */
    public static function get_auth()
    {
        return isys_auth_document::instance();
    }
}
