<?php

use idoit\Module\Document\Compiler\Placeholder\Floorplan;
use idoit\Module\Document\Compiler\Placeholder\PageBreak;

/**
 * i-doit Placeholder
 *
 * @package    Modules
 * @subpackage Document
 * @author     Selcuk Kekec <skekec@synetics.de>
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */
abstract class isys_document_compiler_placeholder
{
    /**
     * Compiled Placholder ready for output.
     *
     * @var  string
     */
    protected $m_compiled_placeholder = null;

    /**
     * Data.
     *
     * @var  array
     */
    protected $m_data = null;

    /**
     * Compiler.
     *
     * @var  isys_document_compiler
     */
    protected $m_compiler = null;

    /**
     * Property DAO.
     *
     * @var  isys_cmdb_dao_category_property
     */
    protected $m_dao_property = null;

    /**
     * Cache. In this form: array("md5 hash of m_datas json representation" => "compiled placeholder")
     *
     * @var  array
     */
    public static $m_cache = [];

    /**
     * md5 hash of m_datas json representation.
     *
     * @var  string
     */
    protected static $m_cache_key = null;

    /**
     * @deprecated This seems to be never used
     * @var        string
     */
    protected $m_custom_style = null;

    /**
     * Cache a placeholder
     *
     * @param string $p_return Compiled placeholder
     * @param string $p_key    md5 hash
     */
    protected static function cache($p_return, $p_key = null)
    {
        $l_key = $p_key ?: static::get_cache_key();
        static::$m_cache[$l_key] = $p_return;
    }

    /**
     * Set Data
     *
     * @param array $p_data
     */
    protected function set_data(array $p_data)
    {
        $this->m_data = $p_data;
    }

    /**
     * Get compiler unit
     *
     * @return isys_document_compiler_chapter
     */
    protected function get_compiler()
    {
        return $this->m_compiler;
    }

    /**
     * Set compiler unit
     *
     * @param isys_document_compiler $p_compiler
     */
    protected function set_compiler(isys_document_compiler $p_compiler)
    {
        $this->m_compiler = $p_compiler;
    }

    /**
     * Get Data
     *
     * @return array
     */
    protected function get_data()
    {
        return $this->m_data;
    }

    /**
     * Set cache key
     *
     * @param string $p_cache_key
     */
    protected static function set_cache_key($p_cache_key)
    {
        static::$m_cache_key = $p_cache_key;
    }

    /**
     * Get cache key
     *
     * @return string
     */
    protected static function get_cache_key()
    {
        return static::$m_cache_key;
    }

    /**
     * Retrieve placeholder from cache by key
     *
     * @param string $p_key
     *
     * @return null
     */
    protected static function get_cached_placeholder($p_key = null)
    {
        $l_key = $p_key ?: static::get_cache_key();

        if (isset(static::$m_cache[$l_key])) {
            return static::$m_cache[$l_key];
        }

        return null;
    }

    /**
     * Check whether request is cached allready
     *
     * @param string $p_key
     *
     * @return bool true | false
     */
    protected static function cached($p_key = null)
    {
        $l_key = $p_key ?: static::get_cache_key();

        return isset(static::$m_cache[$l_key]);
    }

    /**
     * Get compiled string.
     *
     * @param   string $p_custom_style
     *
     * @return  string  compiled placeholder
     */
    public function compile($p_custom_style = null)
    {
        $this->m_custom_style = $p_custom_style;

        // Trigger pre-compile routine
        $this->pre();

        // Call compiler
        $this->m_compiled_placeholder = $this->get();

        // Trigger post-compile routine
        $this->post();

        return $this->m_compiled_placeholder;
    }

    /**
     * Constructor.
     *
     * @param  array                          $p_data
     * @param  isys_document_compiler_chapter $p_compiler
     */
    protected function __construct(array $p_data, isys_document_compiler_chapter $p_compiler)
    {
        $this->m_dao_property = isys_cmdb_dao_category_property::instance(isys_application::instance()->container->get('database'));
        $this->set_data($p_data);
        $this->set_compiler($p_compiler);
    }

    /**
     * Placeholder factory.
     *
     * @param   string                         $type
     * @param   array                          $data
     * @param   isys_document_compiler_chapter $compiler
     *
     * @return  mixed
     * @throws  isys_exception_general
     */
    public static function factory($type, array $data, isys_document_compiler_chapter $compiler)
    {
        // Instantiate desired placeholder object.
        switch ($type) {
            case 'property':
                return new isys_document_compiler_placeholder_property($data, $compiler);

            case 'category':
                return new isys_document_compiler_placeholder_category($data, $compiler);

            case 'report':
                return new isys_document_compiler_placeholder_report($data, $compiler);

            case 'templateVar':
                return new isys_document_compiler_placeholder_template_var($data, $compiler);

            case 'image':
                return new isys_document_compiler_placeholder_image($data, $compiler);

            case 'floorplan':
                return new Floorplan($data, $compiler);

            case 'pageBreak':
                return new PageBreak($data, $compiler);

            default:
                throw new isys_exception_general('Placeholder "' . $type . '" not found.');
        }
    }

    /**
     * Normalize a string to prevent conflicts with the export libs or DomDocument.
     *
     * @param  string $placeholderValue
     *
     * @return string
     */
    protected function normalize($placeholderValue)
    {
        // @see  DOKU-301 reverted DOKU-127
        return preg_replace(['/<script(.*?)<\/script>/i', '/\{[0-9]*\}/'], '', $placeholderValue);
    }

    /**
     * Returns the compiled placeholder
     *
     * @return string
     */
    abstract protected function get();

    /**
     * Pre-Compilation
     *
     * @return mixed
     */
    abstract protected function pre();

    /**
     * Post-Compilation
     *
     * @return mixed
     */
    abstract protected function post();
}
