<?php

/**
 * i-doit document module
 *
 * Installation routines
 *
 * @package     i-doit
 * @subpackage  Modules
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class isys_module_document_install
{
    /**
     * Executes module installation.
     */
    public static function init()
    {
        // Migrate the fonts - don't catch any exceptions... Or else we'll delete everything!
        self::migrate_fonts();

        self::move_tcpdf_fonts();

        // Prepare two variables for the recursive deletion.
        $l_deleted = $l_undeleted = 0;

        // After the migration, remove the vendor directory.
        isys_glob_delete_recursive(dirname(__DIR__) . '/vendor', $l_deleted, $l_undeleted);
    }

    /**
     * Small method to migrate the fonts from the document vendor to the core vendor.
     *
     * @throws  isys_exception_filesystem
     */
    protected static function migrate_fonts()
    {
        global $g_absdir;

        $l_dirs = [
            'vendor/',
            'vendor/tecnick.com/',
            'vendor/tecnick.com/tcpdf/',
            'vendor/tecnick.com/tcpdf/fonts/'
        ];

        $l_document_module = dirname(__DIR__) . '/';
        $l_core_dir = str_replace(['/', '\\'], '/', $g_absdir);
        $l_target_dir = end($l_dirs);

        if (!file_exists($l_core_dir . 'upload/fonts/')) {
            if (!is_writeable($l_core_dir . 'upload/')) {
                throw new isys_exception_filesystem('The directory "' . $l_core_dir .
                    'upload/" is not writable. Please set the correct rights for PHP to write files in this directory!');
            }

            if (!mkdir($l_core_dir . 'upload/fonts')) {
                throw new isys_exception_filesystem('The directory "' . $l_core_dir .
                    'upload/fonts" could not be written. Please set the correct rights for PHP to write files in the parent directory or create it by hand!');
            }
        }

        foreach ($l_dirs as $l_dir) {
            if (!file_exists($l_document_module . $l_dir)) {
                // One of the given directories does not exist... Skip the migration.
                return;
            }

            if (!file_exists($l_core_dir . str_replace('tecnick.com', 'tecnickcom', $l_dir))) {
                // One of the given directories does not exist... Skip the migration.
                return;
            }

            if (!is_readable($l_document_module . $l_dir)) {
                throw new isys_exception_filesystem('The directory "' . $l_document_module . $l_dir .
                    '" is not readable. Please set the correct rights for PHP to read the files in this directory!');
            }

            if (!is_writeable($l_core_dir . str_replace('tecnick.com', 'tecnickcom', $l_dir))) {
                throw new isys_exception_filesystem('The directory "' . $l_document_module . str_replace('tecnick.com', 'tecnickcom', $l_dir) .
                    '" is not writeable. Please set the correct rights for PHP to write files in this directory!');
            }
        }

        // Okay - the font-directories seem to exist. Now we can migrate the missing fonts (if there are any).
        $l_fonts_to_migrate = [];
        $l_document_fonts = array_map('basename', glob($l_document_module . $l_target_dir . '*'));
        $l_core_fonts = array_flip(array_map('basename', glob($l_core_dir . str_replace('tecnick.com', 'tecnickcom', $l_target_dir) . '*')));

        foreach ($l_document_fonts as $l_font) {
            // Don't try to copy directories...
            if (!isset($l_core_fonts[$l_font]) && !is_dir($l_document_module . $l_target_dir . $l_font)) {
                $l_fonts_to_migrate[$l_document_module . $l_target_dir . $l_font] = $l_core_dir . 'upload/fonts/' . $l_font;
            }
        }

        // We should now have a array of fonts, which are not included in the i-doit core.
        if (count($l_fonts_to_migrate)) {
            foreach ($l_fonts_to_migrate as $l_from => $l_to) {
                // Copy the found fonts to our new directory underneath "<i-doit>/upload/fonts/".
                copy($l_from, $l_to);
            }
        }
    }

    /**
     * Small method to migrate the fonts from the vendor/font to the upload/font directory.
     */
    protected static function move_tcpdf_fonts()
    {
        $l_core_font_dir = isys_application::instance()->app_path . '/upload/fonts/';
        $l_vendor_font_dir = isys_application::instance()->app_path . '/vendor/tecnickcom/tcpdf/fonts/';

        $l_vendor_font_files = glob($l_vendor_font_dir . '*', GLOB_MARK);

        if (count($l_vendor_font_files)) {
            foreach ($l_vendor_font_files as $l_font_file) {
                if (substr($l_font_file, -1) == '/') {
                    continue;
                }

                rename($l_font_file, $l_core_font_dir . basename($l_font_file));
            }
        }
    }
}