<?php

namespace idoit\Module\Document\Compiler\Table;

use DOMDocument;
use DOMElement;
use isys_application;

/**
 * i-doit Table Compiler.
 *
 * @package     document
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       1.2.1
 */
class Table
{
    /**
     * @var array
     */
    private $header;

    /**
     * @var array
     */
    private $content;

    /**
     * @var Config
     */
    private $configuration;

    /**
     * @var DOMDocument
     */
    private $dom;

    /**
     * @var DOMElement
     */
    private $domTable;

    /**
     * @var DOMElement
     */
    private $domTableHeader;

    /**
     * @var DOMElement
     */
    private $domTableBody;

    /**
     * @var string
     */
    private $columnStyling;

    /**
     * Table constructor.
     *
     * @param array  $header
     * @param array  $content
     * @param Config $configuration
     */
    public function __construct(array $header, array $content, Config $configuration)
    {
        $this->header = $header;
        $this->content = $content;
        $this->configuration = $configuration;
        $this->dom = new DOMDocument();
        $this->dom->loadHTML('<div></div>', LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);

        $this->domTable = $this->dom->createElement('table');
        $this->domTableHeader = $this->dom->createElement('thead');
        $this->domTableBody = $this->dom->createElement('tbody');

        $this->domTable->setAttribute('style', 'padding:2px 5px;');

        $this->dom->appendChild($this->domTable);
    }

    /**
     * @return string
     */
    private function getColumnStyling()
    {
        if ($this->columnStyling === null) {
            $style = [];

            switch ((string)$this->configuration->getGrid()) {
                case Config::GRID_VERTICAL:
                    $style[] = 'border-left:0.5pt solid #000; border-right:0.5pt solid #000;';
                    break;
                case Config::GRID_HORIZONTAL:
                    $style[] = 'border-top:0.5pt solid #000; border-bottom:0.5pt solid #000;';
                    break;
                case Config::GRID_ALL:
                    $style[] = 'border:0.5pt solid #000;';
                    break;
                default:
                case Config::GRID_NONE:
                    break;
            }

            $this->columnStyling = implode('', $style);
        }

        return $this->columnStyling;
    }

    /**
     * @return string
     */
    public function compile()
    {
        if ($this->configuration->showHeader()) {
            $this->compileHeader();
        }

        $this->compileBody();

        return $this->dom->saveHTML($this->domTable);
    }

    /**
     * Compile method for the table header.
     */
    private function compileHeader()
    {
        $tableRow = $this->dom->createElement('tr');
        $columnStyle = [
            'color:#' . $this->configuration->getHeaderTextColor() . ';',
            'background-color:#' . $this->configuration->getHeaderBackgroundColor() . ';',
            'text-align:' . $this->configuration->getHeaderTextAlign() . ';',
            $this->getColumnStyling()
        ];

        $headerTextStyle = $this->configuration->getHeaderTextStyle();

        if (strpos($headerTextStyle, 'b') !== false) {
            $columnStyle[] = 'font-weight:bold;';
        }

        if (strpos($headerTextStyle, 'i') !== false) {
            $columnStyle[] = 'font-style:italic;';
        }

        if (strpos($headerTextStyle, 'u') !== false) {
            $columnStyle[] = 'text-decoration:underline;';
        }

        $columnStyle = implode('', $columnStyle);

        foreach ($this->header as $header) {
            $tableHeader = $this->dom->createElement('th');
            $tableHeader->setAttribute('style', $columnStyle);

            // We use this in case of "HTML" inside our column.
            $this->appendHTML($tableHeader, $header);

            $tableRow->appendChild($tableHeader);
        }

        if ($this->configuration->showHeaderAfterPageBreak()) {
            $this->domTableHeader->appendChild($tableRow);

            $this->domTable->appendChild($this->domTableHeader);
        } else {
            $this->domTableBody->appendChild($tableRow);
        }
    }

    /**
     * Compile method for the table body.
     */
    private function compileBody()
    {
        $language = isys_application::instance()->container->get('language');
        $columnStyle = $this->getColumnStyling();
        $zebraStyle = strtolower($this->configuration->getZebraRowColor());

        if ($zebraStyle !== 'fff' && $zebraStyle !== 'ffffff') {
            $zebraStyle = 'background-color:#' . $zebraStyle . ';';
        } else {
            $zebraStyle = '';
        }

        if (count($this->content)) {
            foreach ($this->content as $i => $row) {
                $rowZebraStyle = ($i % 2 !== 0 ? $zebraStyle : '');
                $tableRow = $this->dom->createElement('tr');

                if ($rowZebraStyle) {
                    $tableRow->setAttribute('style', $rowZebraStyle);
                }

                $possibleContent = '';

                foreach ($row as $column) {
                    $tableColumn = $this->dom->createElement('td');

                    // We use this in case of "HTML" inside our column.
                    $this->appendHTML($tableColumn, $column);

                    if ($columnStyle) {
                        $tableColumn->setAttribute('style', $columnStyle);
                    }

                    $tableRow->appendChild($tableColumn);
                    $possibleContent .= $column;
                }

                if (!empty($possibleContent)) {
                    $this->domTableBody->appendChild($tableRow);
                }
            }
        } else {
            $tableRow = $this->dom->createElement('tr');

            $tableColumn = $this->dom->createElement('td');
            $tableColumn->setAttribute('colspan', count($this->header));
            $tableColumn->textContent = '-- ' . $language->get('LC__MODULE__DOCUMENT__NO_CONTENT') . ' --';

            if ($columnStyle) {
                $tableColumn->setAttribute('style', $columnStyle);
            }

            $tableRow->appendChild($tableColumn);

            $this->domTableBody->appendChild($tableRow);
        }

        $this->domTable->appendChild($this->domTableBody);
    }

    /**
     * @param \DOMNode $parent
     * @param string   $source
     */
    private function appendHTML(\DOMNode $parent, $source)
    {
        $tmpDoc = new DOMDocument();
        $tmpDoc->loadHTML('<meta http-equiv="Content-Type" content="text/html; charset=utf-8">' . $source);

        $childNodes = $tmpDoc->getElementsByTagName('body')->item(0)->childNodes;

        if ($childNodes !== null && count($childNodes)) {
            foreach ($childNodes as $node) {
                $node = $parent->ownerDocument->importNode($node, true);
                $parent->appendChild($node);
            }
        }
    }
}
