<?php

namespace idoit\Module\Document\Console\Command;

use idoit\Console\Command\AbstractCommand;
use idoit\Module\Document\Document\Document;
use isys_application;
use isys_document_compiler_document;
use isys_settings;
use isys_tenantsettings;
use Symfony\Component\Console\Input\InputDefinition;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class DocumentsCommand extends AbstractCommand
{
    const COMMAND_NAME = 'documents';

    /**
     * @var OutputInterface
     */
    private $output;

    /**
     * Error Messages, key is documentId
     *
     * @var array
     */
    private $errors = [];

    /**
     * Get name for command
     *
     * @return string
     */
    public function getCommandName()
    {
        return self::COMMAND_NAME;
    }

    /**
     * Get description for command
     *
     * @return string
     */
    public function getCommandDescription()
    {
        return '';
    }

    /**
     * Retrieve Command InputDefinition
     *
     * @return InputDefinition
     *
     * @throws \InvalidArgumentException
     */
    public function getCommandDefinition()
    {
        $definition = new InputDefinition();

        // Command modes
        $definition->addOption(new InputOption(
            'create_revision',
            null,
            InputOption::VALUE_NONE,
            'Creates one or more revisions for given documents'
        ));

        $definition->addOption(new InputOption(
            'export',
            null,
            InputOption::VALUE_NONE,
            'Exports one or more documents to a file'
        ));

        // Document filtering options
        $definition->addOption(new InputOption(
            'id',
            null,
            InputOption::VALUE_IS_ARRAY | InputOption::VALUE_REQUIRED,
            'Select one or more documents by id'
        ));

        $definition->addOption(new InputOption(
            'all_documents',
            null,
            InputOption::VALUE_NONE,
            'Selects all documents'
        ));

        $definition->addOption(new InputOption(
            'all_category',
            null,
            InputOption::VALUE_REQUIRED,
            'Selects all documents in category'
        ));

        $definition->addOption(new InputOption(
            'all_template',
            null,
            InputOption::VALUE_REQUIRED,
            'Selects all documents with template'
        ));

        $definition->addOption(new InputOption(
            'all_object',
            null,
            InputOption::VALUE_REQUIRED,
            'Selects all documents with object'
        ));

        // Export Options
        $definition->addOption(new InputOption(
            'export_format',
            null,
            InputOption::VALUE_REQUIRED,
            'Export format: pdf or html'
        ));

        $definition->addOption(new InputOption(
            'export_folder',
            null,
            InputOption::VALUE_REQUIRED,
            'Export format: pdf or html',
            isys_settings::get('system.dir.file-upload', isys_application::instance()->app_path . '/upload/files/')
        ));

        return $definition;
    }

    /**
     * Checks if a command can have a config file via --config
     *
     * @return bool
     */
    public function isConfigurable()
    {
        return false;
    }

    /**
     * Returns an array of command usages
     *
     * @return string[]
     */
    public function getCommandUsages()
    {
        return [];
    }

    /**
     * @param InputInterface  $input
     * @param OutputInterface $output
     *
     * @return int
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $this->output = $output;

        if ($input->getOption('export') && $input->getOption('create_revision')) {
            throw new \InvalidArgumentException("Only provide either --create_revision or --export");
        }

        if ($input->getOption('export') && $input->getOption('export_format') && !in_array($input->getOption('export_format'), ['pdf', 'html'])) {
            throw new \InvalidArgumentException("Export format needs to be either pdf or html");
        }

        if ($input->getOption('export') && $input->getOption('export_format') === null) {
            throw new \InvalidArgumentException("When exporting you need to provide a export_format");
        }

        if (
            $input->getOption('export') && !(is_dir($input->getOption('export_folder')) && is_writable($input->getOption('export_folder')))
        ) {
            throw new \InvalidArgumentException("Export folder needs to be a directory and be writeable");
        }

        $documents = [];

        $documentService = new Document();

        if ($input->getOption('all_documents')) {
            $documents = $documentService = $documentService->retrieveDocuments();
        } else {
            if (!empty($input->getOption('id'))) {
                $documents = array_merge($documents, $documentService->retrieveDocuments($input->getOption('id')));
            }

            if ($input->getOption('all_category') !== null) {
                $documents = array_merge($documents, $documentService->retrieveDocumentsByCategory($input->getOption('all_category')));
            }

            if ($input->getOption('all_object') !== null) {
                $documents = array_merge($documents, $documentService->retrieveDocumentsByObject($input->getOption('all_object')));
            }

            if ($input->getOption('all_template') !== null) {
                $documents = array_merge($documents, $documentService->retrieveDocumentsByTemplate($input->getOption('all_template')));
            }
        }

        if ($input->getOption('create_revision')) {
            $this->createRevisionsForDocuments($documents);

            foreach ($this->errors as $documentId => $error) {
                $this->output->writeln("<error>Error while creating revisions for document ({$documentId})</error>");
                $this->output->writeln("<error>{$error}</error>");
            }

            return 0;
        }

        if ($input->getOption('export')) {
            $this->exportDocuments($documents, $input->getOption('export_format'), $input->getOption('export_folder'));

            foreach ($this->errors as $documentId => $error) {
                $this->output->writeln("<error>Error while exporting document ({$documentId})</error>");
                $this->output->writeln("<error>{$error}</error>");
            }

            return 0;
        }

        return 0;
    }

    private function createRevisionsForDocuments(array $documents)
    {
        $documentService = new Document();

        foreach ($documents as $document) {
            try {
                isys_document_compiler_document::refresh_document($document['isys_document__id']);
                $this->output->writeln("Refreshed document ({$document['isys_document__id']})");
            } catch (\isys_exception_database $e) {
                $this->errors[$document['isys_document__id']] = 'Revision could not be created because of: [Database] ' . $e->getMessage();
            } catch (\isys_exception_general $e) {
                $this->errors[$document['isys_document__id']] = 'Revision could not be created because of: [General]  ' . $e->getMessage();
            } catch (\Exception $e) {
                $this->errors[$document['isys_document__id']] = 'Revision could not be created because of: [General]  ' . $e->getMessage();
            }
        }
    }

    private function exportDocuments($documents, $exportFormat, $exportFolder)
    {
        $documentService = new Document();

        foreach ($documents as $document) {

            try {
                $filePath = $documentService->export($document['isys_document__id'], $exportFormat, $exportFolder);
            } catch (\Exception $e) {
                $this->errors[$document['isys_document__id']] = 'Document could not be exported: [General]  ' . $e->getMessage();
            }

            $this->output->writeln('File exported to ' . $filePath);

            if (is_file($filePath)) {
                $this->output->writeln("Exported document ({$document['isys_document__id']}) to {$filePath}");
            } else {
                $this->errors[$document['isys_document__id']] = 'Document could not be exported: [General]  Unknown error';
            }
        }
    }
}
