<?php

use idoit\Module\Document\CustomFields\Model\Document;

/**
 * i-doit
 *
 * DAO: Global category document.
 *
 * @package     i-doit
 * @subpackage  CMDB_Categories
 * @author      Selcuk Kekec <skekec@i-doit.org>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class isys_cmdb_dao_category_g_document extends isys_cmdb_dao_category_global
{
    /**
     * Category's name. Will be used for the identifier, constant, main table, and many more.
     *
     * @var string
     */
    protected $m_category = 'document';

    /**
     * Is category multi-valued or single-valued?
     *
     * @var boolean
     */
    protected $m_multivalued = true;

    /**
     * Method for returning the properties.
     *
     * @return array
     * @author Selcuk Kekec <skekec@i-doit.org>
     */
    protected function properties()
    {
        return [];
    }

    /**
     * Method for retrieving the dynamic properties of this category dao.
     *
     * @author Leonard Fischer <lfischer@i-doit.com>
     * @return array
     */
    protected function dynamic_properties()
    {
        return [
            '_revision_download' => [
                C__PROPERTY__INFO     => [
                    C__PROPERTY__INFO__TITLE       => 'LC__CMDB__CATG__DOCUMENT__DOWNLOAD_LATEST_REVISION',
                    C__PROPERTY__INFO__DESCRIPTION => 'Last document revision'
                ],
                C__PROPERTY__FORMAT   => [
                    C__PROPERTY__FORMAT__CALLBACK => [$this, 'dynamic_property_callback_revision_download']
                ],
                C__PROPERTY__PROVIDES => [
                    C__PROPERTY__PROVIDES__LIST => false
                ]
            ]
        ];
    }

    /**
     * Dynamic property handling for displaying the "download last revision" button.
     *
     * @param  array $p_row
     *
     * @return string
     */
    public function dynamic_property_callback_revision_download($p_row)
    {
        global $g_dirs;

        $l_doc = $this->get_last_revisons($p_row['isys_obj__id'], C__RECORD_STATUS__NORMAL, 1)->get_row();

        if (is_array($l_doc)) {
            $links = isys_module_document::get_export_links($l_doc['isys_document__id'], $l_doc['isys_document_revision__id']);
            $created = isys_application::instance()->container->get('locales')
                ->fmt_datetime($l_doc['isys_document_revision__created']);
            $language = isys_application::instance()->container->get('language');

            return '<a href="' . $links['html_download'] . '" class="btn btn-small mr5" title="' . _L('LC__CMDB__CATS__FILE_DOWNLOAD') . ' HTML &quot;' . $l_doc['isys_document_template__title'] . '&quot; (' . $created . ')">' .
                '<img src="' . $g_dirs['images'] . 'icons/silk/page_white_code.png" class="mr5" /><span>' . _L('LC__MODULE__DOCUMENT__FORMAT__HTML') . '</span>' .
                '</a><a href="' . $links['pdf_download'] . '" class="btn btn-small mr5" title="' . _L('LC__CMDB__CATS__FILE_DOWNLOAD') . ' PDF &quot;' . $l_doc['isys_document_template__title'] . '&quot; (' . $created . ')">' .
                '<img src="' . $g_dirs['images'] . 'icons/silk/page_white_acrobat.png" class="mr5" /><span>' . _L('LC__MODULE__DOCUMENT__FORMAT__PDF') . '</span>' .
                '</a><a href="' . $links['pdf_inline'] . '" class="btn btn-small" title="' . _L('LC__MODULE__EXPORT') . ' PDF &quot;' . $l_doc['isys_document_template__title'] . '&quot; (' . $created . ')">' .
                '<img src="' . $g_dirs['images'] . 'icons/silk/page_white_acrobat.png" class="mr5" /><span>' . _L('LC__MODULE__DOCUMENT__FORMAT__PDF') . ' ' . _L('LC__MODULE__DOCUMENT__EXPORT_INLINE') . '</span>' .
                '</a>';
        }

        return isys_tenantsettings::get('gui.empty_value', '-');
    }

    /**
     * Sync method for import, export and duplicating.
     *
     * @param  array   $p_category_data
     * @param  integer $p_object_id
     * @param  integer $p_status
     *
     * @return mixed
     */
    public function sync($p_category_data, $p_object_id, $p_status)
    {
    }

    /**
     * Get data method for retrieving data.
     *
     * @param  integer $categoryEntryId
     * @param  integer $objectId
     * @param  string  $condition
     * @param  array   $filter
     * @param  integer $status
     *
     * @return isys_component_dao_result
     */
    public function get_data($categoryEntryId = null, $objectId = null, $condition = "", $filter = null, $status = null)
    {
        $l_sql = 'SELECT * FROM isys_catg_document_list
			INNER JOIN isys_obj ON isys_catg_document_list__isys_obj__id = isys_obj__id
			INNER JOIN isys_document ON isys_catg_document_list__isys_document__id = isys_document__id
			LEFT JOIN isys_document_template ON isys_document_template__id = isys_document__isys_document_template__id
			LEFT JOIN isys_document_type ON isys_document__isys_document_type__id = isys_document_type__id
			WHERE TRUE ' . $condition . ' ' . $this->prepare_filter($filter) . ' ';

        if ($objectId !== null) {
            $l_sql .= $this->get_object_condition($objectId);
        }

        if ($categoryEntryId !== null) {
            $l_sql .= ' AND isys_catg_document_list__id = ' . $this->convert_sql_id($categoryEntryId);
        }

        if ($status !== null) {
            $l_sql .= ' AND isys_catg_document_list__status = ' . $this->convert_sql_int($status);
        }

        return $this->retrieve($l_sql);
    }

    /**
     * Method for retrieving the latest document revisions of the given object.
     *
     * @param  array|int $p_obj_id
     * @param  integer   $p_status
     * @param  integer   $p_limit
     *
     * @return isys_component_dao_result
     * @throws Exception
     * @throws isys_exception_database
     * @author Leonard Fischer <lfischer@i-doit.com>
     */
    public function get_last_revisons($p_obj_id = null, $p_status = C__RECORD_STATUS__NORMAL, $p_limit = null)
    {
        $l_limit = $l_object_condition = '';

        if ($p_obj_id !== null && ($p_obj_id > 0 || is_array($p_obj_id))) {
            if (is_array($p_obj_id)) {
                $l_object_condition = 'AND isys_catg_document_list__isys_obj__id ' . $this->prepare_in_condition($p_obj_id);
            } else {
                $l_object_condition = 'AND isys_catg_document_list__isys_obj__id = ' . $this->convert_sql_id($p_obj_id);
            }
        }

        if ($p_limit !== null && $p_limit > 0) {
            $l_limit = 'LIMIT ' . $this->convert_sql_int($p_limit);
        }

        $l_sql = 'SELECT * FROM isys_catg_document_list
			INNER JOIN isys_obj ON isys_catg_document_list__isys_obj__id = isys_obj__id
			INNER JOIN isys_document ON isys_catg_document_list__isys_document__id = isys_document__id
			INNER JOIN isys_document_revision ON isys_document_revision__isys_document__id = isys_document__id
			LEFT JOIN isys_document_template ON isys_document_template__id = isys_document__isys_document_template__id
			LEFT JOIN isys_document_type ON isys_document__isys_document_type__id = isys_document_type__id
			WHERE TRUE
			' . $l_object_condition . '
			AND isys_catg_document_list__status = ' . $this->convert_sql_int($p_status) . '
			ORDER BY isys_document_revision__created DESC ' . $l_limit . ';';

        return $this->retrieve($l_sql);
    }

    /**
     * Creates the condition to the object table.
     *
     * @param  mixed  $p_obj_id
     * @param  string $p_alias
     *
     * @return string
     * @author Van Quyen Hoang <qhoang@i-doit.de>
     */
    public function get_object_condition($p_obj_id = null, $p_alias = 'isys_obj')
    {
        if (!empty($p_obj_id)) {
            if (is_array($p_obj_id)) {
                return ' AND (isys_catg_document_list__isys_obj__id ' . $this->prepare_in_condition($p_obj_id) . ') ';
            }

            return ' AND (isys_catg_document_list__isys_obj__id = ' . $this->convert_sql_id($p_obj_id) . ') ';
        }

        return '';
    }

    /**
     * Import-Handler for this category.
     *
     * @param  array $p_data
     *
     * @return void
     */
    public function import($p_data)
    {
    }

    /**
     * Executes the query to save the category entry given by its ID $p_cat_level.
     *
     * @param       $p_cat_level
     * @param       $p_newRecStatus
     * @param       $p_document_description
     * @param       $p_description
     * @param array $customFields
     *
     * @return bool
     * @throws isys_exception_dao
     * @throws isys_exception_database
     */
    public function save($p_cat_level, $p_newRecStatus, $p_document_description, $p_description, array $customFields = [])
    {
        /**
         * @see  DOKU-355 Implement saving logic for custom fields.
         */
        $documentIdQuery = 'SELECT isys_document__id AS id
            FROM isys_document 
            INNER JOIN isys_catg_document_list ON isys_catg_document_list__isys_document__id = isys_document__id 
            WHERE isys_catg_document_list__id = ' . $this->convert_sql_id($p_cat_level) . '
            LIMIT 1;';

        $documentId = $this->retrieve($documentIdQuery)->get_row_value('id');

        if ($documentId) {
            $customFieldDocument = new Document($this->m_db);

            foreach ($customFields as $id => $value) {
                $customFieldDocument->saveCustomFieldValue($id, $documentId, $value);
            }
        }

        // Resume with the normal 'save' logic.
        $saveQuery = 'UPDATE isys_catg_document_list 
            INNER JOIN isys_document ON isys_catg_document_list__isys_document__id = isys_document__id 
            SET isys_catg_document_list__description = ' . $this->convert_sql_text($p_description) . ',
			isys_catg_document_list__status = ' . $this->convert_sql_id($p_newRecStatus) . ',
			isys_document__description = ' . $this->convert_sql_text($p_document_description) . '
			WHERE isys_catg_document_list__id = ' . $this->convert_sql_id($p_cat_level) . ';';

        return $this->update($saveQuery) && $this->apply_update();
    }

    /**
     * Save element method.
     *
     * @param    integer $p_cat_level
     * @param    integer $p_intOldRecStatus
     * @param    boolean $p_create
     *
     * @return   mixed  Integer with last inserted ID or boolean false.
     * @version  Niclas Potthast <npotthast@i-doit.org>
     */
    public function save_element(&$p_cat_level, &$p_intOldRecStatus, $p_create = false)
    {
        $l_ret = false;

        if ($_GET[C__CMDB__GET__CATLEVEL] != -1 && $_GET[C__CMDB__GET__CATLEVEL] > 0) {
            /**
             * @see DOKU-355  Implement data handling for custom fields.
             */
            $customFieldValues = [];

            if (isset($_POST['custom-field']) && is_array($_POST['custom-field'])) {
                foreach ($_POST['custom-field'] as $customFieldId => $customFieldValue) {
                    if (isset($_POST['custom-field__HIDDEN'][$customFieldId])) {
                        $customFieldValue = $_POST['custom-field__HIDDEN'][$customFieldId];
                    }

                    $customFieldValues[$customFieldId] = $customFieldValue;
                }
            }

            $l_ret = $this->save(
                $_GET[C__CMDB__GET__CATLEVEL],
                C__RECORD_STATUS__NORMAL,
                $_POST['C__DOCUMENT__DESCRIPTION'],
                $_POST["C__CMDB__CAT__COMMENTARY_" . $this->get_category_type() . $this->get_category_id()],
                $customFieldValues
            );
        }

        return $l_ret;
    }

    /**
     * @param int    $p_objID
     * @param int    $p_newRecStatus
     * @param string $p_description
     *
     * @return bool
     */
    public function create($p_objID, $p_newRecStatus, $p_description)
    {
        return true;
    }

    /**
     * @param array  $p_objects
     * @param int    $p_direction
     * @param string $p_table
     * @param null   $p_checkMethod
     * @param bool   $p_purge
     *
     * @return bool
     */
    public function rank_records($p_objects, $p_direction = C__CMDB__RANK__DIRECTION_DELETE, $p_table = "isys_obj", $p_checkMethod = null, $p_purge = false)
    {
        if (in_array($_POST[C__GET__NAVMODE], [C__NAVMODE__QUICK_PURGE, C__NAVMODE__PURGE])) {
            foreach ($p_objects as $objectId) {
                // Delete document associated to object
                isys_document_dao::instance($this->m_db)
                    ->deleteDocumentsAssociatedToObject($objectId, [$_GET[C__CMDB__GET__OBJECT]]);
            }
        } else {
            parent::rank_records($p_objects, $p_direction, $p_table, $p_checkMethod, $p_purge);
        }

        return true;
    }
}
