<?php

/**
 * i-doit
 *
 * Document templates DAO.
 *
 * @package     i-doit
 * @subpackage  Modules
 * @author      Selcuk Kekec <skekec@i-doit.com>
 * @version     1.0.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.3.0
 */
class isys_document_dao_templates extends isys_component_dao
{
    /**
     * Returns options for template.
     *
     * @param  integer $p_template_id
     *
     * @return array
     * @throws Exception
     */
    public function get_options($p_template_id)
    {
        $l_options = [];

        // Get template data
        $l_res = $this->get_data($p_template_id);

        if ($l_res->num_rows()) {
            $l_row = $l_res->get_row();

            // Is coulumn setted.
            if (isset($l_row['isys_document_template__options']) && is_string($l_row['isys_document_template__options'])) {
                $l_options = isys_format_json::decode($l_row['isys_document_template__options']);
            }
        }

        if (is_array($l_options)) {
            return $l_options;
        }

        isys_notify::warning('The document options seem incompatible (' . serialize($l_options) . '). Please re-save the document options and try again!', ['sticky' => true]);

        return [];
    }

    /**
     * Method for retrieving document templates.
     *
     * @param  integer $p_id
     * @param  string  $p_type
     *
     * @return isys_component_dao_result
     * @throws isys_exception_database
     * @author Selcuk Kekec <skekec@i-doit.com>
     */
    public function get_data($p_id = null, $p_type = null)
    {
        $l_sql = 'SELECT * FROM isys_document_template
            INNER JOIN isys_document_type ON isys_document_template__isys_document_type__id = isys_document_type__id
            WHERE TRUE';

        if ($p_id !== null) {
            if (!is_array($p_id)) {
                $p_id = [$p_id];
            }

            $l_sql .= ' AND isys_document_template__id ' . $this->prepare_in_condition($p_id);
        }

        if ($p_type !== null) {
            $l_sql .= ' AND isys_document_template__isys_document_type__id = ' . $this->convert_sql_text($p_type);
        }

        return $this->retrieve($l_sql . ';');
    }

    /**
     * Method for creating/saving a document templates
     *
     * @param  integer $p_id
     * @param  array   $p_values
     *
     * @return integer
     * @throws isys_exception_dao
     * @author Selcuk Kekec <skekec@i-doit.com>
     */
    public function saveTemplate($p_id = null, array $p_values = [])
    {
        $session = isys_application::instance()->container->get('session');

        $l_data = [];

        if ($p_id === null || empty($p_id)) {
            // @see  DOKU-245  Set "updated" when saving a chapter.
            $p_values['created_at'] = date('Y-m-d H:i:s');
            $p_values['created_by'] = $session->get_user_id();

            $l_sql = 'INSERT INTO isys_document_template SET %s;';
        } else {
            // @see  DOKU-245  Set "updated" when saving a chapter.
            $p_values['updated_at'] = date('Y-m-d H:i:s');
            $p_values['updated_by'] = $session->get_user_id();

            $l_sql = 'UPDATE isys_document_template SET %s WHERE isys_document_template__id = ' . $this->convert_sql_id($p_id) . ';';
        }

        // Sanitize and encode options
        if (isset($p_values['options']) && is_array($p_values['options'])) {
            $p_values['options'] = isys_format_json::encode($p_values['options']);
        }

        if (count($p_values) > 0) {
            foreach ($p_values as $l_key => $l_value) {
                $l_data[] = 'isys_document_template__' . $l_key . ' = ' . $this->convert_sql_text($l_value);
            }

            $this->update(str_replace('%s', implode(', ', $l_data), $l_sql)) && $this->apply_update();
        }

        return $p_id ?: $this->get_last_insert_id();
    }

    /**
     * Remove a document templates.
     *
     * @param  integer|array $id
     *
     * @return boolean
     * @throws isys_exception_dao
     * @author Selcuk Kekec <skekec@i-doit.com>
     */
    public function deleteTemplate($id)
    {
        if (!is_array($id)) {
            $id = [$id];
        }

        return ($this->update('DELETE FROM isys_document_template WHERE isys_document_template__id ' . $this->prepare_in_condition($id) . ';') && $this->apply_update());
    }

    /**
     * Import method for a local ZIP file (May be uploaded prior).
     *
     * @param   string $p_zip_filepath
     *
     * @throws  Exception
     * @throws  isys_exception_filesystem
     * @throws  isys_exception_general
     * @return  array
     */
    public function import_by_zip($p_zip_filepath)
    {
        $lang = isys_application::instance()->container->get('language');
        $dbSystem = isys_application::instance()->container->get('database_system');

        $l_log = isys_log::get_instance('document-template-import');
        $l_log->info('Starting import procedure...');

        $l_dao_chapter = new isys_document_dao_chapters($this->m_db);
        $l_dao_components = new isys_document_dao_components($this->m_db);
        $l_dao_dialog = new isys_cmdb_dao_dialog_admin($this->m_db);
        $l_dao_report = isys_report_dao::instance($dbSystem);

        $l_result = [];
        $l_result['upload-name'] = substr(strrchr($p_zip_filepath, DS), 1);
        $l_result['upload-filepath'] = $p_zip_filepath;
        $l_result['upload-files-dir'] = substr($l_result['upload-filepath'], 0, -4);

        if (substr($l_result['upload-name'], -3) != 'zip') {
            $l_log->error('The uploaded file "' . $l_result['upload-name'] . '" needs to be a ZIP archive.');
            throw new isys_exception_general('The uploaded file "' . $l_result['upload-name'] . '" needs to be a ZIP archive.');
        }

        if (!file_exists($l_result['upload-filepath'])) {
            $l_log->error('The file "' . $l_result['upload-filepath'] . '" does not exist!');
            throw new isys_exception_filesystem('The file "' . $l_result['upload-filepath'] . '" does not exist!', '');
        }

        if (!class_exists("ZipArchive") || !extension_loaded('zlib')) {
            $l_log->error('It seems as if the "zlib" extension is not installed! Please do so and try again.');
            throw new isys_exception_general('It seems as if the "zlib" extension is not installed! Please do so and try again.');
        }

        $l_files = new isys_update_files();

        if (!$l_files->read_zip($l_result['upload-filepath'], $l_result['upload-files-dir'], false, true)) {
            $l_log->error('The given ZIP file could not be unzipped! Please try again.');
            throw new isys_exception_general('The given ZIP file could not be unzipped! Please try again.');
        }

        $l_log->info('File successfully uploaded!');

        $l_json_data = isys_format_json::decode(file_get_contents($l_result['upload-files-dir'] . '/data.json'));
        $l_images = glob($l_result['upload-files-dir'] . '/images/*');

        if (count($l_images)) {
            $l_log->info('Copying the images (found ' . count($l_images) . ')');

            if (is_writable(isys_module_document::get_upload_dir())) {
                // Move all images to the upload directory.
                foreach ($l_images as $l_image) {
                    $l_image = str_replace(['\\', '/'], DS, $l_image);

                    try {
                        $l_log->info('Copying "' . $l_image . '"...');
                        copy($l_image, isys_module_document::get_upload_dir() . DS . substr(strrchr($l_image, DS), 1));
                    } catch (Exception $e) {
                        $l_log->error('The image "' . isys_module_document::get_upload_dir() . '" could not be copied!');
                    }
                }
            } else {
                $l_log->warning('Attention! The directory "' . isys_module_document::get_upload_dir() . '" is not writable. The images could not be imported!');
            }
        }

        // @see  DOKU-205  Adding the document title to the log.
        $l_log->info('Importing the document "' . $lang->get($l_json_data['document']['title']) . '".');

        // @see  DOKU-205  Adding the category name to the log.
        $l_log->info('Using template category "' . $lang->get($l_json_data['document']['category']) . '".');

        // Check, if the given category already exists.
        $l_category_id = $l_dao_dialog
            ->get_by_title('isys_document_type', $l_json_data['document']['category'])
            ->get_row_value('isys_document_type__id');

        if ($l_category_id === null) {
            $l_category_id = $l_dao_dialog->create('isys_document_type', $l_json_data['document']['category'], null, '', C__RECORD_STATUS__NORMAL);

            $l_log->info('Could not find category "' . $l_json_data['document']['category'] . '", created it (ID #' . $l_category_id . ').');
        }

        $l_template_id = $this->saveTemplate(null, [
            'title'                  => $l_json_data['document']['title'],
            'isys_document_type__id' => $l_category_id,
            'header'                 => $l_json_data['document']['header'],
            'footer'                 => $l_json_data['document']['footer'],
            'options'                => $l_json_data['document']['options']
        ]);

        $i = 0;
        $l_chapter_matching = [];
        $l_report_matching = [];

        // Now we add exported reports to the system.
        $l_log->debug('Importing prior exported reports');
        foreach ($l_json_data['reports'] as $l_report_data) {
            $l_log->debug('Creating report "' . $l_report_data['name'] . '"');
            $l_report_matching[$l_report_data['name']] = $l_dao_report->createReport($l_report_data['name'], $l_report_data['description'], $l_report_data['sql'], '', false,
                false, null, $l_report_data['data']);
        }

        $l_log->info('Importing chapters');
        foreach ($l_json_data['chapter'] as &$l_chapter) {
            $l_log->info('Adding chapter "' . $l_chapter['title'] . '"');
            $l_chapter_id = $l_dao_chapter->saveChapter(null, [
                'sort'                       => $i++,
                'title'                      => $l_chapter['title'],
                'isys_document_template__id' => $l_template_id
            ]);

            $l_component_id = $l_dao_chapter->get_data($l_chapter_id)
                ->get_row_value('isys_document_chapter__isys_document_component__id');

            // Because the local Report ID might differ to the imported one, we re-set the ID.
            $l_found = !!preg_match_all('~\<span.*?data-json=\"(.*?)\".*?data-widget-type=\"report\".*?\>~', $l_chapter['html'], $l_matches);

            if ($l_found) {
                foreach ($l_matches[1] as $l_report_json) {
                    $l_report_data = isys_format_json::decode(str_replace("'", '"', $l_report_json));

                    $l_report_data['data']['id'] = $l_report_matching[$l_report_data['data']['name']];

                    $l_new_report_json = str_replace('"', "'", isys_format_json::encode($l_report_data));

                    // We simply replace the "original" found JSON with our newly created JSON.
                    $l_chapter['html'] = str_replace($l_report_json, $l_new_report_json, $l_chapter['html']);
                }
            }

            $l_dao_components->saveComponent($l_component_id, [
                'text' => $l_chapter['html']
            ]);

            $l_chapter['id'] = $l_chapter_id;
            $l_chapter_matching[$l_chapter['title']] = $l_chapter_id;
        }

        // After we created the chapters, we update the parents.
        $l_log->debug('Setting chapter parents');
        foreach ($l_json_data['chapter'] as $l_chapter_data) {
            if ($l_chapter_data['parent'] !== null && isset($l_chapter_matching[$l_chapter_data['parent']])) {
                $l_log->debug('Setting chapter parent "' . $l_chapter_data['parent'] . '" (#' . $l_chapter_matching[$l_chapter_data['parent']] . ') > "' .
                    $l_chapter_data['title'] . '" (#' . $l_chapter_data['id'] . ')');

                $l_dao_chapter->update_parent_chapter($l_chapter_data['id'], $l_chapter_matching[$l_chapter_data['parent']]);
            }
        }

        $l_result['log'] = $l_log->get_log(true);

        return $l_result;
    }
}
