<?php

use idoit\Module\Document\Document\Document;

/**
 * i-doit
 *
 * Ajax Handler for Manual-Module.
 *
 * @package     Modules
 * @subpackage  Document
 * @author      Selcuk Kekec <skekec@i-doit.com>
 * @version     1.0.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.3.0
 */
class isys_ajax_handler_document extends isys_ajax_handler
{
    /**
     * Call the retrieved Method and die
     *
     * @throws \idoit\Exception\JsonException
     * @throws isys_exception_database
     * @throws isys_exception_general
     */
    public function init()
    {
        try {
            if (method_exists($this, $_GET['method'])) {
                switch ($_GET['method']) {
                    case 'saveChapterSorting':
                        header('Content-Type: application/json; charset=UTF-8');
                        try {
                            $l_return = [
                                'success' => true,
                                'data'    => $this->saveChapterSorting($_POST['documentId'], isys_format_json::decode($_POST['sorting'])),
                                'message' => ''
                            ];
                        } catch (Exception $e) {
                            $l_return = [
                                'success' => false,
                                'data'    => null,
                                'message' => $e->getMessage()
                            ];
                        }

                        echo isys_format_json::encode($l_return);
                        break;

                    case 'upload_import_template':
                        header('Content-Type: application/json; charset=UTF-8');

                        try {
                            $l_return = [
                                'success' => true,
                                'data'    => $this->upload_import_template(),
                                'message' => null
                            ];
                        } catch (Exception $e) {
                            $l_return = [
                                'success' => false,
                                'data'    => null,
                                'message' => $e->getMessage()
                            ];
                        }

                        echo isys_format_json::encode($l_return);
                        die;

                    case 'export_template':
                        header('Content-Type: application/json; charset=UTF-8');

                        try {
                            $l_return = [
                                'success' => true,
                                'data'    => $this->export_template($_POST['id']),
                                'message' => null
                            ];
                        } catch (Exception $e) {
                            $l_return = [
                                'success' => false,
                                'data'    => null,
                                'message' => $e->getMessage()
                            ];
                        }

                        echo isys_format_json::encode($l_return);
                        die;

                    case 'import_ttf_font':
                        header('Content-Type: application/json; charset=UTF-8');

                        try {
                            $l_return = [
                                'success' => true,
                                'data'    => $this->import_ttf_font(),
                                'message' => null
                            ];
                        } catch (Exception $e) {
                            $l_return = [
                                'success' => false,
                                'data'    => null,
                                'message' => $e->getMessage()
                            ];
                        }

                        echo isys_format_json::encode($l_return);
                        die;

                    case 'uploadBackgroundImage':
                        header('Content-Type: application/json; charset=UTF-8');

                        try {
                            $l_return = [
                                'success' => true,
                                'data'    => $this->uploadBackgroundImage(),
                                'message' => null
                            ];
                        } catch (Exception $e) {
                            $l_return = [
                                'success' => false,
                                'data'    => null,
                                'message' => $e->getMessage()
                            ];
                        }

                        echo isys_format_json::encode($l_return);
                        die;

                    case 'regenerateDocument':
                        $reason = isys_cmdb_dao_dialog_admin::instance($this->m_database_component)
                            ->get_data('isys_document_revision_reasons', ((int)$_POST['revisionReason']))
                            ->__as_array();

                        $comment = $_POST['comment'];

                        if (!empty($reason)) {
                            $comment = $reason[0]['isys_document_revision_reasons__title'] . ': <br /> ' . $comment;
                        }

                        $this->regenerateDocument($_POST[C__DOCUMENT__GET__DOCUMENT_ID], ((!empty($_POST['comment']) || !empty($reason)) ? $comment : ''));
                        break;

                    case 'regenerateDocuments':
                        $reason = isys_cmdb_dao_dialog_admin::instance($this->m_database_component)
                            ->get_data('isys_document_revision_reasons', (int)$_POST['revisionReason'])
                            ->__as_array();

                        $comment = $_POST['comment'];

                        if (!empty($reason)) {
                            $comment = $reason[0]['isys_document_revision_reasons__title'] . ': <br /> ' . $comment;
                        }

                        $this->regenerateDocuments(
                            isys_format_json::decode($_POST[C__DOCUMENT__GET__DOCUMENT_ID]),
                            ((!empty($_POST['comment']) || !empty($reason)) ? $comment : '')
                        );
                        break;

                    case 'createDocument':
                        /**
                         * @see DOKU-354  Add 'custom fields' related data to the document creation.
                         */
                        $customFieldData = [];

                        if (isset($_POST['custom-field']) && is_array($_POST['custom-field'])) {
                            foreach ($_POST['custom-field'] as $customFieldId => $customFieldValue) {
                                if (isset($_POST['custom-field__HIDDEN'][$customFieldId])) {
                                    $customFieldValue = $_POST['custom-field__HIDDEN'][$customFieldId];
                                }

                                $customFieldData[$customFieldId] = $customFieldValue;
                            }
                        }

                        $this->createDocument(
                            $_POST['documentTitle'],
                            $_POST['documentTemplateID'],
                            $_POST['description'],
                            isys_format_json::decode($_POST['object']),
                            $customFieldData
                        );
                        break;

                    default:
                        $this->{$_GET['method']}();
                }
            } else {
                throw new isys_exception_general("Method '" . $_GET['method'] . "' does not exist.");
            }
        } catch (Exception $e) {
            http_response_code(500);
            isys_notify::error($e->getMessage());
        }

        die;
    }

    /**
     * Save a Chapter
     *
     * @author Selcuk Kekec <skekec@i-doit.com>
     */
    private function saveChapter()
    {
        $l_dao_chapter = new isys_document_dao_chapters($this->m_database_component);

        try {
            $l_dao_chapter->update_title($_POST['id'], $_POST['title']);

            echo 1;
        } catch (Exception $e) {
            echo 0;
        }

        die;
    }

    /**
     * Add a chapter
     *
     * @author Selcuk Kekec <skekec@i-doit.com>
     */
    private function addChapter()
    {
        $l_dao_chapter = new isys_document_dao_chapters($this->m_database_component);

        try {
            $l_chapterID = $l_dao_chapter->saveChapter(null, [
                'title'                      => $_POST['title'],
                'isys_document_chapter__id'  => $_POST['parentChapterID'] ?: 'NULL',
                'isys_document_template__id' => $_POST['documentTemplateID'],
            ]);

            $l_dao_chapter->update_component($l_chapterID, null, $_POST['componentID'], $_POST['copy']);

            echo 1;
        } catch (Exception $e) {
            echo 0;
        }

        die;
    }

    /**
     * Remove a Chapter
     *
     * @author Selcuk Kekec <skekec@i-doit.com>
     */
    private function removeChapter()
    {
        $l_dao_chapter = new isys_document_dao_chapters($this->m_database_component);

        try {
            $l_dao_chapter->deleteChapter($_POST['chapterID']);

            echo 1;
        } catch (Exception $e) {
            echo 0;
        }

        die;
    }

    /**
     * Save chapter sorting.
     *
     * @param   integer $documentId
     * @param   array   $p_sorting_data
     *
     * @return  array
     * @throws  isys_exception_auth
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function saveChapterSorting($documentId, array $p_sorting_data)
    {
        isys_auth_document::instance()
            ->check(isys_auth::EDIT, 'TEMPLATES');

        $l_dao_chapter = isys_document_dao_chapters::instance($this->m_database_component);

        if (count($p_sorting_data)) {
            foreach ($p_sorting_data as $l_index => $l_item) {
                $l_dao_chapter->update_position($l_item['chapter_id'], $l_index + 1);
                $l_dao_chapter->update_parent_chapter($l_item['chapter_id'], $l_item['parent_chapter_id']);
            }
        }

        $l_chapterTree = $l_dao_chapter->get_chapter_tree($l_dao_chapter->get_data(null, $documentId));

        if (!is_array($l_chapterTree)) {
            $l_chapterTree = [];
        }

        return $l_chapterTree;
    }

    /**
     * Retrieve a components text
     *
     * @author Selcuk Kekec <skekec@i-doit.com>
     */
    private function getComponentText()
    {
        echo isys_document_dao_components::instance($this->m_database_component)
            ->get_data($_POST['componentID'])
            ->get_row_value('isys_document_component__text');

        die;
    }

    /**
     * Create a new document.
     *
     * @param  string    $documentTitle
     * @param  integer   $documentTemplateID
     * @param  string    $description
     * @param  int|array $object
     *
     * @throws Exception
     * @throws isys_exception_database
     * @throws isys_exception_general
     */
    private function createDocument($documentTitle, $documentTemplateID, $description, $object, array $customFields = [])
    {
        (new Document())->create($documentTitle, $documentTemplateID, $description, $object, $customFields);

        die;
    }

    /**
     * Regenerate the given document.
     *
     * @param  integer $documentId
     * @param  string  $comment
     *
     * @return void
     * @author Leonard Fischer <lfischer@i-doit.com>
     */
    private function regenerateDocument($documentId, $comment = '')
    {
        try {
            if (!is_numeric($documentId)) {
                throw new isys_exception_general('No numeric document ID given');
            }

            isys_document_compiler_document::refresh_document((int)$documentId, $comment);
        } catch (Exception $e) {
            echo $e->getMessage();
        }
    }

    /**
     * Regenerate all given document.
     *
     * @param  array  $documentIds
     * @param  string $comment
     *
     * @author Leonard Fischer <lfischer@i-doit.com>
     * @throws isys_exception_dao
     * @throws isys_exception_database
     * @throws isys_exception_general
     * @return void
     */
    private function regenerateDocuments(array $documentIds, $comment = '')
    {
        foreach ($documentIds as $documentId) {
            isys_document_compiler_document::refresh_document((int)$documentId, $comment);
        }
    }

    /**
     * Get the content of a compiled document.
     */
    private function getDocumentContent()
    {
        echo isys_document_dao::instance($this->m_database_component)
            ->get_data($_GET['id'])
            ->get_row_value('isys_document__text');

        die;
    }

    /**
     * Deletes a given revision
     */
    private function deleteRevision()
    {
        $l_dao = isys_document_dao_revision::instance($this->m_database_component);

        if (isys_format_json::is_json($_POST['revisionIDs'])) {
            $_POST['revisionIDs'] = isys_format_json::decode($_POST['revisionIDs']);
        }

        $l_dao->deleteRevision($_POST['revisionIDs']);
        die;
    }

    /**
     * Method for uploading and importing a document template.
     *
     * @throws  isys_exception_filesystem
     * @throws  isys_exception_general
     * @return  array
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function upload_import_template()
    {
        $l_uploader = new isys_library_fileupload;

        $l_result = $l_uploader->set_prefix(substr(md5(microtime(true)), 0, 8) . '__')
            ->handleUpload(isys_module_document::get_import_dir() . DS);

        if ($l_result['success'] === true) {
            return isys_document_dao_templates::instance($this->m_database_component)
                ->import_by_zip(isys_module_document::get_import_dir() . DS . $l_uploader->getUploadName());
        }

        throw new isys_exception_general($l_result['error']);
    }

    /**
     * Method for exporting the given template.
     *
     * @param   integer $p_id
     *
     * @return  string
     */
    private function export_template($p_id)
    {
        $l_data = isys_factory::get_instance('isys_document_export_template_json', $p_id)
            ->export(true);

        $l_data['download_url'] = isys_helper_link::create_url([
            C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
            C__GET__SETTINGS_PAGE => isys_module_document::CL__ACTION__TEMPLATE_EXPORT_DOWNLOAD,
            'file'                => $l_data['filename'],
        ]);

        return $l_data;
    }

    /**
     * Method for uploading and installing a TTF font to TCPDF.
     *
     * @return  array
     * @throws  isys_exception_filesystem
     * @throws  isys_exception_general
     */
    private function import_ttf_font()
    {
        global $g_dirs;

        $l_upload_dir = realpath($g_dirs['temp']) . DS;
        $l_uploader = new isys_library_fileupload;
        $l_filename = $l_uploader->getName();

        $l_result = $l_uploader->handleUpload($l_upload_dir);

        if ($l_result['success'] !== true || (isset($l_result['error']) && !empty($l_result['error']))) {
            throw new isys_exception_filesystem($l_result['error']);
        }

        // If the "just uploaded" file could not be found, we throw an error.
        if (!file_exists($l_upload_dir . $l_filename)) {
            throw new isys_exception_filesystem(_L('LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__UPLOAD_TTF_FILE_NOT_FOUND', $l_upload_dir . $l_filename));
        }

        // Install the font!
        $l_fontname = TCPDF_FONTS::addTTFfont($l_upload_dir . $l_filename, '', '', 32, isys_document_format_pdf::getFontUploadDir() . DS);

        $l_filemanager = new isys_component_filemanager();
        $l_filemanager->delete($l_filename, $l_upload_dir);

        // If the font has been installed, we remove it from the "temp" directory.
        if ($l_fontname == false) {
            throw new isys_exception_general(_L('LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__UPLOAD_TTF_FAILURE'));
        }

        $l_result['fontname'] = $l_fontname;

        return $l_result;
    }

    /**
     * Method for uploading background images.
     *
     * @return array
     * @throws isys_exception_filesystem
     */
    private function uploadBackgroundImage()
    {
        $uploadDirectory = isys_module_document::get_upload_dir() . DS;
        $uploader = new isys_library_fileupload;
        $originalFilename = $uploader->getName();
        $newFilename = substr(md5($originalFilename), 0, 8) . '_' . isys_helper_upload::prepare_filename($originalFilename);

        $result = $uploader->handleUpload($uploadDirectory);

        rename($uploadDirectory . $originalFilename, $uploadDirectory . $newFilename);

        if ($result['success'] !== true || (isset($result['error']) && !empty($result['error']))) {
            throw new isys_exception_filesystem($result['error']);
        }

        $result['filename'] = $newFilename;

        return $result;
    }
}
