'use strict';

/**
 * Chapter tree class, built for the i-doit Document Add-on
 *
 * @author     Leonard Fischer <lfischer@i-doit.com>
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 * @since      1.2.1
 * @version    1.0
 */

window.DocumentChapterTree = Class.create({
    initialize: function ($tree, options) {
        this.$tree = $tree;
        this.$droppableContainer = new Element('div');
        
        this.$tree.insert({after: this.$droppableContainer});
        
        this.$droppable = new Element('li', {className: 'chapter-droppable hide'});
        
        this.options = {
            chapterData:       [],
            saveTreeUrl:       '',
            chapterEditUrl:    '',
            documentEditUrl:   '',
            subChapterAddUrl:  '',
            documentId:        0
        };
        
        Object.extend(this.options, options || {});
        
        this.buildTree();
    },
    
    buildTree: function () {
        var i, nodes = {}, $parent;
        
        this.$tree.update();
        
        for (i in this.options.chapterData) {
            if (!this.options.chapterData.hasOwnProperty(i)) {
                continue;
            }
            
            nodes[this.options.chapterData[i].id] = this.prepareNodes(this.options.chapterData[i]);
        }
    
        for (i in this.options.chapterData) {
            if (!this.options.chapterData.hasOwnProperty(i)) {
                continue;
            }
            
            $parent = this.$tree;
            
            if (nodes.hasOwnProperty(this.options.chapterData[i].parent)) {
                $parent = nodes[this.options.chapterData[i].parent].down('.innerChapters');
            }
            
            $parent.insert(nodes[this.options.chapterData[i].id]);
        }
        
        this.updateChapterNumbers();
        
        this.addEvents();
    },
    
    addEvents: function () {
        var $listPoints = this.$tree.select('li.chapterListNode'),
            i;
        
        // We'll create the draggables.
        for (i in $listPoints) {
            if (!$listPoints.hasOwnProperty(i)) {
                continue;
            }
            
            new Draggable($listPoints[i], {
                revert:  true,
                onStart: this.displayDroppables.bindAsEventListener(this),
                onEnd:   this.hideDroppables.bindAsEventListener(this),
                handle:  'drag-handle',
                snap: function (x, y, drag) {
                    return [
                        (x + drag.offset[0]) + $('contentWrapper').scrollLeft - 13,
                        (y + drag.offset[1]) + $('contentWrapper').scrollTop - 13
                    ];
                }
            });
        }
    },
    
    displayDroppables: function (drag) {
        var that      = this,
            rootPos   = this.$tree.getBoundingClientRect(),
            $nodeList = this.$tree.select('.chapterListNode'),
            $droppable, i, zIndex = 100;
        
        drag.element.addClassName('dragging');
        
        this.$droppableContainer.update();
        
        // We create invisible drop-zones above the tree.
        for (i in $nodeList) {
            if (!$nodeList.hasOwnProperty(i)) {
                continue;
            }
            
            if ($nodeList[i].up('.dragging')) {
                continue;
            }
            
            var pos = $nodeList[i].getBoundingClientRect();
            
            if (drag.element.next() !== $nodeList[i] && $nodeList[i].match(':first-child')) {
                $droppable = new Element('div')
                    .writeAttribute({
                        className:    'chapter-droppable',
                        'data-before': $nodeList[i].readAttribute('data-chapter-id')
                    })
                    .setStyle({
                        top:    1 + (pos.top - rootPos.top) + 'px',
                        left:   10 + (pos.left - rootPos.left) + 'px',
                        width:  pos.width + 'px',
                        height: '13px',
                        zIndex: zIndex
                    });
                zIndex ++;
        
                this.$droppableContainer.insert($droppable);
            }
            
            $droppable = new Element('div')
                .writeAttribute({
                    className:     'chapter-droppable',
                    'data-parent': $nodeList[i].readAttribute('data-chapter-id')
                })
                .setStyle({
                    top:    15 + (pos.top - rootPos.top) + 'px',
                    left:   15 + (pos.left - rootPos.left) + 'px',
                    width:  (pos.width - 5) + 'px',
                    height: ((pos.bottom - pos.top) - 10) + 'px', //'21px',
                    zIndex: zIndex
                });
            zIndex ++;
            
            this.$droppableContainer.insert($droppable);
    
            $droppable = new Element('div')
                .writeAttribute({
                    className:     'chapter-droppable',
                    'data-after': $nodeList[i].readAttribute('data-chapter-id')
                })
                .setStyle({
                    top:    6 + (pos.bottom - rootPos.top) + 'px',
                    left:   10 + (pos.left - rootPos.left) + 'px',
                    width:  pos.width + 'px',
                    height: '13px',
                    zIndex: zIndex
                });
            zIndex++;
    
            this.$droppableContainer.insert($droppable);
        }
    
        this.$droppableContainer.removeClassName('hide');
        
        this.$droppableContainer.select('.chapter-droppable').reverse().each(function($node) {
            Droppables.add($node, {
                hoverclass: 'hover',
                onDrop:     that.saveTreeState.bind(that)
            });
        })
    },
    
    saveTreeState: function ($draggable, $droppable) {
        var beforeId = $droppable.readAttribute('data-before'),
            parentId = $droppable.readAttribute('data-parent'),
            afterId = $droppable.readAttribute('data-after'),
            $chapterNodes, $parent, data = [];
        
        try {
            if (beforeId !== null) {
                this.$tree.down('[data-chapter-id="' + beforeId + '"]').insert({before: $draggable});
            }
    
            if (parentId !== null) {
                this.$tree.down('[data-chapter-id="' + parentId + '"] .innerChapters').insert($draggable);
            }
    
            if (afterId !== null) {
                this.$tree.down('[data-chapter-id="' + afterId + '"]').insert({after: $draggable});
            }
        } catch (e) {
            // This helps to save checks for "prevent a element to be dragged in itself".
            return;
        }
    
        $chapterNodes = this.$tree.select('.chapterListNode');
        
        for (i in $chapterNodes) {
            if (! $chapterNodes.hasOwnProperty(i)) {
                continue;
            }
    
            $parent = $chapterNodes[i].up('li.chapterListNode');
            
            data.push({
                chapter_id: $chapterNodes[i].readAttribute('data-chapter-id'),
                parent_chapter_id: ($parent ? $parent.readAttribute('data-chapter-id') : null)
            });
        }
    
        new Ajax.Request(this.options.saveTreeUrl, {
            method: 'post',
            parameters: {
                documentId: this.options.documentId,
                sorting: JSON.stringify(data)
            },
            onComplete: function (xhr) {
                var json = xhr.responseJSON;
            
                if (json.success) {
                    this.options.chapterData = json.data;
    
                    this.buildTree();
                
                    idoit.callbackManager.triggerCallback('document-refresh-tree');
                } else {
                    idoit.Notify.error(json.message || xhr.responseText, {sticky: true});
                }
            }.bind(this)
        });
    },
    
    hideDroppables: function () {
        this.$tree.select('.dragging').invoke('removeClassName', 'dragging');
        this.$droppableContainer.addClassName('hide');
    },
    
    prepareNodes: function (node) {
        var $node = new Element('li', {id:'chapterID_' + node.id, 'data-chapter-id':node.id, 'data-parent-chapter-id':node.parent, className:'chapterListNode'}),
            $nodeContent = new Element('div', {className:'chapterContent'}),
            $childList = new Element('ul', {id:'chapterChilds_' + node.id, 'data-chapter-id':node.id, className:'innerChapters'}),
            $operatorContainer = new Element('div', {className:'operatorContainer'}),
            nodeTitle = node.title;
    
        if (node.component && nodeTitle) {
            nodeTitle += " (" + idoit.Translate.get('LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__INHERITS') + " " +
                (new Element('a', {href: this.options.documentEditUrl.replace(/\{doc-id\}/g, node.component.templateID)}).update(node.component.templateTitle).outerHTML) + " &raquo; " +
                (new Element('a', {href: this.options.chapterEditUrl.replace(/\{id\}/g, node.component.chapterID)}).update(node.component.chapterTitle).outerHTML) + ")";
        }
        
        $nodeContent
            .update(new Element('div', {className:'fl drag-handle'})
                .update(new Element('span', {className:'resize mr5'}).update('&nbsp;&nbsp;&nbsp;'))
                .insert(new Element('span', {className:'chapterNumber mr5'}))
                .insert(new Element('span', {className:'chapterTitle'}).update(nodeTitle)));
    
        $operatorContainer
            .insert(new Element('a', {href:this.options.chapterEditUrl.replace(/\{id\}/g, node.id), className:'chapterOperator mr5 chapterEditor', title:idoit.Translate.get('LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__EDIT_CHAPTER')})
                .update(new Element('img', {src:window.dir_images + 'icons/silk/pencil.png'})))
            .insert(new Element('a', {href:this.options.subChapterAddUrl.replace(/\{id\}/g, node.id), className:'chapterOperator mr5 addChapter', title:idoit.Translate.get('LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__ADD_SUB_CHAPTER')})
                .update(new Element('img', {src:window.dir_images + 'icons/silk/add.png'})))
            .insert(new Element('a', {href:'javascript:', className:'chapterOperator mr5 chapterRemover', title:idoit.Translate.get('LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__DELETE_CHAPTER')})
                .update(new Element('img', {src:window.dir_images + 'icons/silk/cross.png'})));
        
        $nodeContent.insert($operatorContainer);
        
        $node.update($nodeContent).insert($childList);
    
        return $node;
    },
    
    updateChapterNumbers: function () {
        var that = this;
        
        this.$tree.select('li.chapterListNode').each(function ($li) {
            $li.down('span.chapterNumber').update(that.getChapterNumber($li));
        });
    },
    
    getChapterNumber: function ($startLi) {
        var path = [],
            $li  = $startLi;
        
        while ($li && $li.match('li')) {
            path.push($li.up().childElements().filter(function ($el) {
                return $el.hasClassName('chapterListNode');
            }).indexOf($li) + 1);
            
            $li = $li.up('li');
        }
        
        return path.reverse().join('.');
    }
});
