'use strict';

/**
 * CustomFields class, built for the i-doit Document Add-on.
 *
 * @author     Leonard Fischer <lfischer@i-doit.com>
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 * @since      1.3.0
 * @version    1.0
 */

window.CustomFields = Class.create({
    initialize: function ($container, options, data) {
        this.$container = $container;
        
        this.data = data || [];
        this.deletion = [];
        
        this.options = {
            dataTypes: []
        };
        
        Object.extend(this.options, options || {});
        
        this.process();
        
        this.$dataField.setValue(JSON.stringify(this.data));
        
        this.setObserver();
        this.refreshSortable();
    },
    
    setObserver: function () {
        this.$container.on('click', 'button[data-action]', this.processButtonPress.bindAsEventListener(this));
        this.$container.on('change', 'input,select', this.processDataChange.bindAsEventListener(this));
    },
    
    refreshSortable: function () {
        Sortable.create(this.$container.down('.content'), {
            tag:      'li',
            handle:   'handle',
            onChange: function () {
                this.processDataChange();
            }.bind(this)
        });
    },
    
    processButtonPress: function (ev) {
        var $button = ev.findElement('button'),
            action  = $button.readAttribute('data-action');
        
        switch (action) {
            case 'add-row':
                this.addCustomFieldRow();
                break;
            
            case 'remove-row':
                this.removeCustomFieldRow($button.up('li').readAttribute('data-id'));
                break;
        }
    },
    
    processDataChange: function (ev) {
        var newData          = [],
            $customFieldRows = this.$container.select('.content li'),
            tupel, i;
        
        if (ev && ev.target && $(ev.target).matches('[name^="data-type"]'))
        {
            // @todo  Here we reload the "default value" smarty plugin.
        }
        
        for (i in $customFieldRows) {
            if (!$customFieldRows.hasOwnProperty(i)) {
                continue;
            }
            
            newData.push({
                id:           $customFieldRows[i].readAttribute('data-id'),
                name:         $customFieldRows[i].down('[name^="name"]').getValue(),
                dataType:     $customFieldRows[i].down('[name^="data-type"]').getValue(),
                defaultValue: null // @todo  Default values!
            });
        }
        
        this.data = newData;
        this.$dataField.setValue(JSON.stringify(this.data));
        this.$deleteField.setValue(JSON.stringify(this.deletion))
    },
    
    process: function () {
        var $header    = new Element('ul')
                .writeAttribute('class', 'border header')
                .setStyle({borderBottom: 'none'}),
            $content   = new Element('ul')
                .writeAttribute('class', 'border content')
                .setStyle({borderTop: 'none'}),
            $addButton = new Element('button')
                .writeAttribute('class', 'btn mt5')
                .writeAttribute('data-action', 'add-row')
                .writeAttribute('type', 'button')
                .update(new Element('img')
                    .writeAttribute('class', 'mr5')
                    .writeAttribute('src', window.dir_images + 'icons/silk/add.png'))
                .insert(new Element('span')
                    .update(idoit.Translate.get('LC__MODULE__DOCUMENT__CUSTOM_FIELDS__ADD_NEW_FIELD'))),
            i;
        
        this.$dataField = new Element('input')
            .writeAttribute('type', 'hidden')
            .writeAttribute('name', 'custom-fields');
    
        this.$deleteField = new Element('input')
            .writeAttribute('type', 'hidden')
            .writeAttribute('name', 'custom-fields-deletion');
        
        $header
            .update(new Element('li')
                .writeAttribute('class', 'gradient')
                .update(new Element('div')
                    .writeAttribute('class', 'label-a ml15')
                    .update(idoit.Translate.get('LC__SYSTEM__CUSTOM_CATEGORIES__FIELD_TITLE')))
                .insert(new Element('div')
                    .writeAttribute('class', 'label-b')
                    .update(idoit.Translate.get('LC__SYSTEM__CUSTOM_CATEGORIES__FIELD_TYPE')))
                .insert(new Element('div')
                    .writeAttribute('class', 'label-c')
                    .update(idoit.Translate.get('LC__SYSTEM__CUSTOM_CATEGORIES__FIELD_ACTION')))
                .insert(new Element('br')
                    .writeAttribute('class', 'cb')));
        
        for (i in this.data) {
            if (!this.data.hasOwnProperty(i)) {
                continue;
            }
            
            $content.insert(this.renderCustomFieldRow(this.data[i]));
        }
        
        this.$container
            .update(this.$dataField)
            .insert(this.$deleteField)
            .insert($header)
            .insert($content)
            .insert($addButton);
    },
    
    addCustomFieldRow: function () {
        var identifier = new Date();
        
        this.$container.down('.content').insert(this.renderCustomFieldRow({
            id: 'new' + (identifier.getTime() + ("000" + identifier.getMilliseconds()).substr(-3))
        }));
        
        this.processDataChange();
        this.refreshSortable();
    },
    
    removeCustomFieldRow: function (id) {
        this.deletion.push(id);
        
        this.$container.down('[data-id="' + id + '"]').remove();
        this.processDataChange();
        this.refreshSortable();
    },
    
    renderCustomFieldRow: function (data) {
        var $row          = new Element('li'),
            $dragHandle   = new Element('div')
                .writeAttribute('class', 'handle'),
            $fieldName    = new Element('input')
                .writeAttribute('class', 'input input-small ml15 mr15')
                .writeAttribute('type', 'text')
                .writeAttribute('name', 'name[' + data.id + ']')
                .setValue(data.name || ''),
            $fieldType    = this.renderCustomFieldTypeSelect()
                .writeAttribute('name', 'data-type[' + data.id + ']')
                .setValue(data.dataType || ''),
            $fieldRemover = new Element('button')
                .writeAttribute('class', 'btn ml15 text-red')
                .writeAttribute('type', 'button')
                .writeAttribute('data-action', 'remove-row')
                .update(new Element('img')
                    .writeAttribute('class', 'mr5')
                    .writeAttribute('src', window.dir_images + 'icons/silk/cross.png'))
                .insert(new Element('span').update(idoit.Translate.get('LC__UNIVERSAL__REMOVE')));
        
        return $row
            .writeAttribute('data-id', data.id || 0)
            .update($dragHandle)
            .insert($fieldName)
            .insert($fieldType)
            .insert($fieldRemover);
    },
    
    renderCustomFieldTypeSelect: function () {
        var $select = new Element('select')
                .writeAttribute('class', 'input input-small'),
            i;
        
        for (i in this.options.dataTypes) {
            if (!this.options.dataTypes.hasOwnProperty(i)) {
                continue;
            }
            
            $select.insert(
                new Element('option')
                    .writeAttribute('value', i)
                    .update(this.options.dataTypes[i])
            );
        }
        
        return $select;
    }
});
