<?php

namespace idoit\Module\Document\Document;

use idoit\Module\Document\CustomFields\Repository\CustomFieldsRepository;
use idoit\Module\Document\CustomFields\Model\Document AS CustomFieldsModel;
use isys_application;
use isys_cmdb_dao_category_data_value;
use isys_cmdb_dao_category_s_person_master;
use isys_document_compiler_chapter;
use isys_document_compiler_document;
use isys_document_compiler_model_document;
use isys_document_dao;
use isys_document_dao_chapters;
use isys_document_dao_revision;
use isys_document_dao_templates;
use isys_document_export;
use isys_document_export_html;
use isys_document_export_pdf;
use isys_document_format_html;
use isys_format_json;
use isys_helper_upload;
use isys_tenantsettings;

class Document
{
    /**
     * @param       $documentTitle
     * @param       $documentTemplateID
     * @param       $description
     * @param       $object
     * @param array $customFields
     *
     * @return false|int|null
     * @throws \idoit\Exception\JsonException
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     * @throws \isys_exception_general
     */
    public function create($documentTitle, $documentTemplateID, $description, $object, array $customFields = [])
    {
        $l_documentID = false;

        $database = isys_application::instance()->container->get('database');
        $session = isys_application::instance()->container->get('session');

        if (!is_array($object) && is_numeric($object)) {
            $object = [$object];
        }

        $l_dao = isys_document_dao::instance($database);
        $l_dao_chapter = isys_document_dao_chapters::instance($database);
        $l_dao_document = isys_document_dao_templates::instance($database);
        $l_dao_person = isys_cmdb_dao_category_s_person_master::instance($database);

        $l_person_data = $l_dao_person->get_data(null, $session->get_user_id())->get_row();
        $l_documentRow = $l_dao_document->get_data($documentTemplateID)->get_row();
        $l_options = [];

        if (isys_format_json::is_json_array($l_documentRow['isys_document_template__options'])) {
            $l_options = isys_format_json::decode($l_documentRow['isys_document_template__options']);
        }

        $l_model_document = new isys_document_compiler_model_document($l_options);

        // Prepare document model.
        $l_model_document
            ->load($documentTemplateID, $l_dao_chapter)
            ->set_author($l_person_data['isys_cats_person_list__first_name'] . ' ' . $l_person_data['isys_cats_person_list__last_name'])
            ->set_title($documentTitle)
            ->set_created(time())
            ->set_raw_index($l_documentRow['isys_document_template__index'])
            ->set_raw_header($l_documentRow['isys_document_template__header'])
            ->set_raw_footer($l_documentRow['isys_document_template__footer'])
            ->set_document_type(_L($l_documentRow['isys_document_type__title']))
            ->setDescription($description);

        // Create document compiler.
        $l_compiler_document = new isys_document_compiler_document($l_model_document);
        isys_cmdb_dao_category_data_value::$m_store_data = true;

        if (is_array($object) && count($object)) {
            foreach ($object as $l_object) {
                isys_document_compiler_document::set_global_master_object($l_object);
                isys_document_compiler_chapter::set_global_master_object($l_object);

                // Save the document.
                $l_documentID = $l_dao->saveDocument(null, [
                    'title'                      => $documentTitle,
                    'description'                => $description,
                    'isys_document_template__id' => $documentTemplateID,
                    'created'                    => date('Y-m-d H:i:s')
                ], $l_object);

                if ($l_documentID) {
                    /**
                     * @see DOKU-354  Add 'custom fields' related logic, also move 'compilation' here.
                     */
                    $customFieldDocument = new CustomFieldsModel($database);

                    foreach ($customFields as $customFieldId => $customFieldValue) {
                        $customFieldDocument->saveCustomFieldValue($customFieldId, $l_documentID, $customFieldValue);
                    }

                    $l_model_document
                        ->setCustomFields((new CustomFieldsRepository())->getCustomFieldsForDocument($l_documentID));

                    // Compile our document.
                    $l_compiler_document->compile();

                    isys_document_dao_revision::create_revision($l_documentID, $l_compiler_document->get_model(), $session->get_user_id());
                }
            }
        } else {
            // Save the document.
            $l_documentID = $l_dao->saveDocument(null, [
                'title'                      => $documentTitle,
                'description'                => $description,
                'isys_document_template__id' => $documentTemplateID,
                'created'                    => date('Y-m-d H:i:s')
            ]);

            if ($l_documentID) {
                /**
                 * @see DOKU-354  Add 'custom fields' related logic, also move 'compilation' here.
                 */
                $customFieldDocument = new CustomFieldsModel($database);

                foreach ($customFields as $customFieldId => $customFieldValue) {
                    $customFieldDocument->saveCustomFieldValue($customFieldId, $l_documentID, $customFieldValue);
                }

                $l_model_document
                    ->setCustomFields((new CustomFieldsRepository())->getCustomFieldsForDocument($l_documentID));

                // Compile our document.
                $l_compiler_document->compile();

                isys_document_dao_revision::create_revision($l_documentID, $l_compiler_document->get_model(), $session->get_user_id());
            }
        }

        return $l_documentID;
    }

    /**
     * Exports the last document revision and returns its filepath
     *
     * @param int $documentId
     * @param string $exportType
     * @param string $exportFolder
     *
     * @return string
     *
     * @throws \Exception
     */
    public function export($documentId, $exportType, $exportFolder)
    {
        $daoRevisions = new \isys_document_dao_revision(isys_application::instance()->container->database);
        $daoTemplates = new \isys_document_dao_templates(isys_application::instance()->container->database);

        $l_revision_id = $daoRevisions->get_last_revision($documentId);

        $l_document_result = $daoRevisions->get_data($l_revision_id);

        if ($l_document_result->num_rows() > 0) {
            $l_document_data = $l_document_result->get_row();

            $l_models = $daoRevisions->load_models_for_revision($documentId, $l_revision_id);

            if (is_array($l_models) && count($l_models)) {
                switch ($exportType) {
                    case 'pdf':
                        $l_defaultPDFFormatter = isys_tenantsettings::get('document.default-pdf-formatter', 'isys_document_format_pdf');

                        if (!class_exists($l_defaultPDFFormatter)) {
                            $l_defaultPDFFormatter = 'isys_document_format_pdf';
                        }

                        // Create the exporter and inject the lib.
                        $l_export = new isys_document_export_pdf($l_models, new $l_defaultPDFFormatter());
                        break;

                    case 'html':
                        $l_export = new isys_document_export_html($l_models, new isys_document_format_html());
                        break;

                    default:
                        throw new \Exception("No valid export type provided");
                        break;
                }

                // Get export configuration from template.
                $l_options = $daoTemplates->get_options($l_document_data['isys_document_template__id']);

                $fileName = $this->prepareExportFilename($documentId, $l_revision_id);

                /**
                 * @var $l_export isys_document_export
                 */
                $l_export->set_title($l_document_data['isys_document__title'])
                    ->set_options($l_options + [
                            'index'           => $l_document_data['isys_document_revision__index'],
                            'header'          => $l_document_data['isys_document_revision__header'],
                            'footer'          => $l_document_data['isys_document_revision__footer'],
                            'revisionId'      => $l_revision_id,
                            'revisionComment' => $l_document_data['isys_document_revision__comment']
                        ])
                    ->export()
                    ->write($fileName, $exportFolder);

                return $exportFolder . '/' . $fileName . '.' . $l_export->get_file_extension();
            }

            throw new \Exception("Document does not have any chapters.");
        }

        throw new \Exception("Document does not exist.");
    }

    /**
     * This method returns a filename, like "<document-template-name>_<object-name>_<timestamp>".
     *
     * @param   integer $documentId
     * @param   integer $revisionId
     *
     * @return  string
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    protected function prepareExportFilename($documentId, $revisionId)
    {
        $l_doc = isys_document_dao::instance(isys_application::instance()->container->get('database'))
            ->get_data($documentId)
            ->get_row();

        $filename = isys_application::instance()->container->get('language')->get($l_doc['isys_document_type__title']) .
            ' - ' . $l_doc['isys_document_template__title'] .
            ' - ' . $l_doc['isys_obj__title'] .
            ' - Revision ' . $revisionId;

        return isys_helper_upload::prepare_filename($filename);
    }

    /**
     * Retrieve documents filtered by given ids
     *
     * @param null|int[] $ids
     *
     * @return array
     */
    public function retrieveDocuments($ids = null)
    {
        $dao = new isys_document_dao(isys_application::instance()->container->get('database'));

        return $dao->get_data($ids)->__as_array();
    }

    /**
     * Retrieve documents filtered by given ids
     *
     * @param int $template
     *
     * @return array
     */
    public function retrieveDocumentsByTemplate($template)
    {
        $dao = new isys_document_dao(isys_application::instance()->container->get('database'));

        return $dao->get_data(null, null, " AND isys_document_template__id = " . $template)->__as_array();
    }

    /**
     * Retrieve documents filtered by given ids
     *
     * @param int $objectId
     *
     * @return array
     */
    public function retrieveDocumentsByObject($objectId)
    {
        $dao = new isys_document_dao(isys_application::instance()->container->get('database'));

        return $dao->get_data(null, null, " AND isys_obj__id = " . $objectId)->__as_array();
    }

    /**
     * Retrieve documents filtered by given ids
     *
     * @param int $category
     *
     * @return array
     */
    public function retrieveDocumentsByCategory($category)
    {
        $dao = new isys_document_dao(isys_application::instance()->container->get('database'));

        return $dao->get_data(null, $category)->__as_array();
    }
}
