<?php

/**
 * i-doit
 *
 * Exports HTML to PDF.
 *
 * @package     i-doit
 * @subpackage  Modules
 * @author      Selcuk Kekec <skekec@i-doit.com>
 * @version     1.0.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.4.0
 */
class isys_document_export_pdf extends isys_document_export
{
    /**
     * Export Document to pdf.
     *
     * @param string $p_filename
     *
     * @return  isys_document_export_pdf
     */
    public function export($p_filename = '')
    {
        // Initialize formatter with options.
        $this->m_formatter->initialize($this->m_options);

        $this->prepareStylesheet();

        $this->m_formatter->beforeExport();

        // Set CSS styling in the formatter, since TCPDF will need to append this chunk in every "writeHTML" call.
        $this->m_formatter->setStyle($this->get_style());

        // @see DOKU-395 Reset the page number to work with the document page order.
        $this->m_formatter->resetPageNumber();

        /**
         * Add chapters html text to our document
         *
         * @var $l_model isys_document_export_model
         */
        foreach ($this->m_models as $l_model) {
            // Determine if chapter shall be in a new page or not
            if ($l_model->get_new_page() && ($l_model->get_pos() != '1.')) {
                $this->m_formatter->addNewPage();
            }

            // Attach headline to formatted document.
            if ($this->m_options['general.headlines']) {
                $this->m_formatter->addHeadline($l_model->get_title(), $l_model->get_pos(), $l_model->get_level());
            }

            // @see ID-413 No longer wrap the content in a DIV container
            $l_text = $l_model->get_text();

            // Correct right alignment
            $doc = new DOMDocument('1.0', 'utf-8');
            $doc->loadHTML('<?xml encoding="utf-8" ?>' . $l_text);
            $this->domAlignReplace($doc);
            $l_text = $doc->saveHTML($doc->lastChild); // we take last child because that's where the text is

            // Attach content to formatted document.
            $this->m_formatter->addContent($l_text);
        }

        // Call 'endPage' to include the footer on the last page.
        $this->m_formatter->endPage();

        $this->m_formatter->afterExport();

        return $this;
    }

    /**
     *
     * @param string      $p_filename
     * @param string|null $filePath
     *
     * @return  isys_document_export_pdf
     */
    public function write($p_filename, $filePath = null)
    {
        $this->m_formatter->save(($filePath !== null ? rtrim($filePath, '/') . '/' . $p_filename . '.' . $this->get_file_extension() : $this->get_filepath($p_filename)));

        return $this;
    }

    /**
     * @param string $filename
     * @param bool   $inline
     *
     * @return $this
     */
    public function send(string $filename, bool $inline = false): self
    {
        $this->m_formatter->send($filename . '.pdf', $inline);

        return $this;
    }

    /**
     * Mimetype of our file.
     *
     * @return  string
     */
    public function get_mimetype(): string
    {
        return 'application/pdf';
    }

    /**
     * Extension.
     *
     * @return string
     */
    public function get_file_extension(): string
    {
        return 'pdf';
    }

    /**
     * Initialize.
     */
    protected function init()
    {
        // Setting default styles.
        $this->prepareStylesheet();
    }

    /**
     * @return void
     */
    private function prepareStylesheet()
    {
        // @see  DOKU-304 Preparing some default styles.
        $style = 'p { margin: 0 0 10px 0; }
            table { width: 100%; }
            code { font-family: courier; }
            ins { text-decoration: underline; }
            blockquote { padding: 5px 20px; margin-left: 20px; border-left: 3px solid #ccc; }
            .marker { background-color: #ff0; }
            .text-monospace, .font-monospace { font-family: courier; } ';

        // @see  DOKU-308  Pass the defined headline-sizes.
        foreach ($this->m_options as $key => $value) {
            if (preg_match('~general\.h([1-6])\.size~', $key, $matches)) {
                $style .= 'h' . $matches[1] . ' { font-size: ' . $value . 'pt; } ';
            }
        }

        $this->set_style($style);
    }

    /**
     * @param DOMNode $domNode
     *
     * @see   DOKU-359
     */
    private function domAlignReplace(DOMNode $domNode)
    {
        if ($domNode->hasChildNodes()) {
            foreach ($domNode->childNodes as $child) {
                if ($child->nodeType === XML_ELEMENT_NODE && $child->hasAttributes()) {
                    foreach ($child->attributes as $attr) {
                        if ($attr->nodeName === 'style' && preg_match('/float\s*:\s*right/', $attr->nodeValue)) {
                            $domNode->setAttribute('align', 'right');

                            return;
                        }
                    }
                }
                $this->domAlignReplace($child);
            }
        }
    }
}
