<?php

/**
 *
 * @package     i-doit
 * @subpackage  Modules
 * @version     1.1
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class isys_document_format_pdf extends TCPDF implements isys_document_formattable
{
    /*
     * @see  DOKU-158  Constant for calculating the footer scale.
     */
    const C__FOOTER__PDF__SCALER = 1.4;

    /*
     * @see  DOKU-309  Constants for the current "page type".
     */
    const C__PAGE_TYPE__INDEX   = 'index';
    const C__PAGE_TYPE__TOC     = 'toc';
    const C__PAGE_TYPE__CONTENT = 'content';

    /**
     * PDF Options.
     *
     * @var array
     */
    private array $m_options = [];

    /**
     * Footer height.
     *
     * @var int
     */
    private int $footerHeight = 0;

    /**
     * In order to be styled correctly, we need to pass the styles with every "writeHTML" call.
     *
     * @var string
     */
    private string $style = '';

    /**
     * Variable contains the current "page type".
     *
     * @var string
     */
    private string $pageType = self::C__PAGE_TYPE__CONTENT;

    /**
     * @var int
     */
    private int $lastIndexPage = 0;

    /**
     * @var  array
     */
    protected $m_formatOptions = [
        'pdf.title'              => [
            'title'       => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__OPTION_TITLE',
            'type'        => 'text',
            'default'     => '',
            'placeholder' => '',
            'p_strClass'  => 'input-small'
        ],
        'pdf.subject'            => [
            'title'       => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__OPTION_SUBJECT',
            'type'        => 'text',
            'default'     => '',
            'placeholder' => '',
            'p_strClass'  => 'input-small'
        ],
        'pdf.keywords'           => [
            'title'       => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__OPTION_KEYWORDS',
            'type'        => 'textarea',
            'default'     => '',
            'placeholder' => '',
            'p_strClass'  => 'input-small'
        ],
        'pdf.margin-top'         => [
            'title'       => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__EXPORT_M_TOP',
            'type'        => 'text',
            'default'     => '20',
            'placeholder' => '20',
            'p_strStyle'  => 'width:70px;',
            'p_nMaxLen'   => 3,
            'description' => 'mm'
        ],
        'pdf.margin-bottom'      => [
            'title'       => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__EXPORT_M_BOTTOM',
            'type'        => 'text',
            'default'     => '20',
            'placeholder' => '20',
            'p_strStyle'  => 'width:70px;',
            'p_nMaxLen'   => 3,
            'description' => 'mm'
        ],
        'pdf.margin-left'        => [
            'title'       => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__EXPORT_M_LEFT',
            'type'        => 'text',
            'default'     => '20',
            'placeholder' => '20',
            'p_strStyle'  => 'width:70px;',
            'p_nMaxLen'   => 3,
            'description' => 'mm'
        ],
        'pdf.margin-right'       => [
            'title'       => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__EXPORT_M_RIGHT',
            'type'        => 'text',
            'default'     => '20',
            'placeholder' => '20',
            'p_strStyle'  => 'width:70px;',
            'p_nMaxLen'   => 3,
            'description' => 'mm'
        ],
        'pdf.font.family'        => [
            'title'      => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__DEFAULT_FONT',
            'type'       => 'select',
            'default'    => 'helvetica',
            'options'    => [],
            'p_strClass' => 'input-mini'
        ],
        'pdf.font.size'          => [
            'title'       => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__DEFAULT_FONT_SIZE',
            'type'        => 'text',
            'default'     => '10',
            'placeholder' => '10',
            'p_strStyle'  => 'width:70px;',
            'p_nMaxLen'   => 3
        ],
        'pdf.orientation'        => [
            'title'      => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__ORIENTATION',
            'type'       => 'select',
            'default'    => 'p',
            'options'    => [
                'p' => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__ORIENTATION_PORTRAIT',
                'l' => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__ORIENTATION_LANDSCAPE'
            ],
            'p_strClass' => 'input-mini'
        ],
        'pdf.background.index'   => [
            'type'   => 'text',
            'hidden' => true
        ],
        'pdf.background.toc'     => [
            'type'   => 'text',
            'hidden' => true
        ],
        'pdf.background.content' => [
            'type'   => 'text',
            'hidden' => true
        ]
    ];

    /**
     * PDF default font.
     *
     * @var string
     */
    protected $m_defaultFont = 'helvetica';

    /**
     * Default page unit.
     *
     * @var string
     */
    protected $m_defaultPageUnit = 'mm';

    /**
     * Retrieves the format option definition for TCPDF.
     *
     * @return array
     */
    public function getFormatOptionDefinition(): array
    {
        $l_fontList = [
            'helvetica' => 'helvetica'
        ];

        foreach ($this->getFontsList() as $l_font) {
            $l_fontList[$l_font] = $l_font;
        }

        ksort($l_fontList);

        $this->m_formatOptions['pdf.font.family']['options'] = $l_fontList;

        return [
            'LC__DOCUMENT__PDF_SETTINGS' => $this->m_formatOptions
        ];
    }

    /**
     * @param string $style
     *
     * @return isys_document_format_pdf
     */
    public function setStyle(string $style): isys_document_format_pdf
    {
        $this->style = $style;

        return $this;
    }

    /**
     * Process the PDF header.
     *
     * @return void
     */
    public function Header()
    {
        $image = null;
        $imagePath = isys_module_document::get_upload_dir() . '/';
        $optionKey = 'pdf.background.' . $this->pageType;

        if (isset($this->m_options[$optionKey]) && !empty($this->m_options[$optionKey])) {
            $image = $this->m_options[$optionKey];

            if (!file_exists($imagePath . $image)) {
                $image = null;
            }
        }

        // @see  DOKU-309  If we defined a background image (that also exists in the filesystem) we display it underneath the content.
        if ($image) {
            $width = 210;
            $height = 297;

            // Swap the width and height for landscape orientation.
            if (isset($this->m_options['pdf.orientation']) && $this->m_options['pdf.orientation'] === 'l') {
                $width = 297;
                $height = 210;
            }

            // Get the current page break margin and 'auto-page-break' mode.
            $margin = $this->getBreakMargin();
            $autoPageBreak = $this->AutoPageBreak;

            // Disable auto-page-break.
            $this->SetAutoPageBreak(false, 0);

            // Set background image.
            $this->Image($imagePath . $image, 0, 0, $width, $height);

            // Restore auto-page-break status and set the starting point for the page content.
            $this->SetAutoPageBreak($autoPageBreak, $margin);
            $this->setPageMark();
        }

        if ($this->pageType === self::C__PAGE_TYPE__CONTENT && isset($this->m_options['header'])) {
            // This needs to be done.
            $fontSize = isset($this->m_options['pdf.font.size'])
                ? ($this->m_options['pdf.font.size'] > 5
                    ? $this->m_options['pdf.font.size']
                    : 10)
                : 10;

            $this->SetFont($this->m_options['pdf.font.family'], '', $fontSize);

            $topMargin = ((int)$this->m_options['pdf.margin-top']);

            if ($topMargin > 0) {
                $this->setY($topMargin);
            }

            $this->writeHTML($this->style . $this->replace_variables($this->m_options['header']), true, false, true, false, '');
        }
    }

    /**
     * We overwrite this method to correctly handle fonts, that are included from other directories.
     *
     * @param string    $family
     * @param string    $style
     * @param null      $size
     * @param string    $fontfile
     * @param string    $subset
     * @param bool|true $out
     *
     * @return mixed|void
     */
    public function SetFont($family, $style = '', $size = null, $fontfile = '', $subset = 'default', $out = true)
    {
        try {
            return parent::SetFont($family, $style, $size, '', $subset, $out);
        } catch (Exception $e) {
            isys_notify::warning($e->getMessage(), ['life' => 10]);

            return parent::SetFont('helvetica', $style, $size, '', $subset, $out);
        }
    }

    /**
     * Process the PDF footer.
     *
     * @return void
     */
    public function Footer()
    {
        if ($this->pageType === self::C__PAGE_TYPE__CONTENT && isset($this->m_options['footer'])) {
            // This needs to be done.
            $fontSize = isset($this->m_options['pdf.font.size'])
                ? ($this->m_options['pdf.font.size'] > 5
                    ? $this->m_options['pdf.font.size']
                    : 10)
                : 10;

            $this->SetFont($this->m_options['pdf.font.family'], '', $fontSize);

            // @see  DOKU-158
            $footer = str_replace('<p>&nbsp;</p>', '', $this->m_options['footer']);

            $this->setY('-' . $this->footerHeight);
            $this->SetAutoPageBreak(false);
            $this->writeHTML($this->style . $this->replace_variables($footer));
            $this->SetAutoPageBreak(true, $this->footerHeight);
        }
    }

    /**
     * @param array $options
     *
     * @return void
     */
    public function initialize(array $options): void
    {
        $this->m_options = $options;

        $l_orientation = '';

        if (isset($this->m_options['pdf.orientation'])) {
            $l_orientation = $this->m_options['pdf.orientation'];
        }

        if (isset($this->m_options['general.index-page']) && $this->m_options['general.index-page']) {
            $this->pageType = self::C__PAGE_TYPE__INDEX;
        }

        // Initialize configured font
        $this->initFont();

        // @See DOKU-158
        $footerLineCount = 1;
        $footer = str_replace('<p>&nbsp;</p>', '', $options['footer']);

        if (!empty($footer)) {
            $domObj = new DOMDocument();
            $domObj->loadHTML($footer);

            if ($domObj->getElementsByTagName('table')->length > 0) {
                $footerLineCount = $domObj->getElementsByTagName('tr')->length;
            } else {
                $footerLineCount = $domObj->getElementsByTagName('p')->length;
            }
        }

        // @See DOKU-158 Calculate footer height which considers the height of the footer content
        if (isset($options['footer'])) {
            $this->footerHeight = (($this->getFontSize() * $this->getScaleFactor() * $footerLineCount) / self::C__FOOTER__PDF__SCALER) + $options['pdf.margin-bottom'];
        }

        // Page orientation
        $this->setPageOrientation($l_orientation, '', $this->footerHeight);

        // Set default page unit
        $this->setPageUnit($this->m_defaultPageUnit);

        if (isset($this->m_options['header'])) {
            $options['pdf.margin-top'] *= 2;
        }

        // Default margins
        $this->SetMargins($options['pdf.margin-left'], $options['pdf.margin-top'], $options['pdf.margin-right']);

        // Set PDF title
        if (isset($options['pdf.title'])) {
            $this->SetTitle($options['pdf.title']);
        }

        // Set PDF subject
        if (isset($options['pdf.subject'])) {
            $this->SetSubject($options['pdf.subject']);
        }

        // Set PDF keywords
        if (isset($options['pdf.keywords'])) {
            $this->SetSubject($options['pdf.keywords']);
        }

        $this->AddPage();
    }

    /**
     * Fill the list of available fonts ($this->fontlist).
     *
     * @return array
     */
    public function getFontsList(): array
    {
        parent::getFontsList();

        $l_font_upload_dir = self::getFontUploadDir();

        // @see ID-2265 / DOKU-60
        if (file_exists($l_font_upload_dir) && is_readable($l_font_upload_dir)) {
            $l_fonts = array_map('basename', glob($l_font_upload_dir . '/*.php'));

            if (count($l_fonts)) {
                foreach ($l_fonts as $l_font) {
                    $this->fontlist[] = substr($l_font, 0, -4);
                }
            }
        }

        return $this->fontlist;
    }

    /**
     * Initialize configured font from options.
     *
     * @return $this
     */
    protected function initFont(): self
    {
        if (!isset($this->m_options['pdf.font.family'])) {
            $this->m_options['pdf.font.family'] = $this->m_defaultFont;
        }

        // Set font family.
        $fontSize = isset($this->m_options['pdf.font.size']) ? ($this->m_options['pdf.font.size'] > 5 ? $this->m_options['pdf.font.size'] : 10) : 10;

        $this->SetFont($this->m_options['pdf.font.family'], '', $fontSize);

        return $this;
    }

    /**
     * Write formatted content to $p_filename.
     *
     * @param string $filename
     *
     * @return $this
     */
    public function save(string $filename): self
    {
        $this->lastPage();
        $this->output($filename, 'F');

        return $this;
    }

    /**
     * Send formatted content to browser.
     *
     * @param string $filename
     * @param bool   $inline
     *
     * @return $this
     */
    public function send(string $filename, bool $inline = false): self
    {
        $this->lastPage();
        $this->output($filename, $inline ? 'I' : 'D');

        return $this;
    }

    /**
     * Add Chapter content.
     *
     * @param string $text
     *
     * @return $this
     */
    public function addContent(string $text): self
    {
        $this->writeHTML($this->style . $this->replace_variables($text), true, false, true);

        return $this;
    }

    /**
     * Add Chapter heading.
     *
     * @param string $title
     * @param string $pos
     * @param int    $level
     *
     * @return $this
     */
    public function addHeadline(string $title, string $pos, int $level = 0): self
    {
        $position = $this->m_options['general.hx.show_numbers'] ? $pos : '';

        if ($position || $position == 0) {
            $title = $position . ' ' . $title;
        }

        $this->Bookmark($title, ($level - 1), 0, '', 'B', [0, 0, 0]);

        $fontColor = '#000000';
        $fontColorKey = 'general.h' . $level . '.color';

        if (isset($this->m_options[$fontColorKey])) {
            $fontColor = $this->m_options[$fontColorKey];
        }

        // @see  DOKU-308  Pass the defined headline-sizes.
        $fontSize = '';
        $fontSizeKey = 'general.h' . $level . '.size';

        if (isset($this->m_options[$fontSizeKey]) && is_numeric($this->m_options[$fontSizeKey])) {
            $fontSize = 'font-size:' . $this->m_options[$fontSizeKey] . 'pt;';
        }

        $this->writeHTML('<h' . $level . ' style="color:' . $fontColor . ';' . $fontSize . '">' . $title . '</h' . $level . '>', true, false, true);

        return $this;
    }

    /**
     * @return $this
     */
    public function afterExport(): self
    {
        if (isset($this->m_options['general.toc']) && $this->m_options['general.toc']) {
            $this->pageType = self::C__PAGE_TYPE__TOC;

            // Add a new page for TOC.
            $this->addTOCPage();

            // Initialize configured font.
            $this->initFont();

            // @see DOKU-395 Add the TOC page(s) at the end of the "index page".
            $this->addTOC($this->lastIndexPage + 1, $this->m_options['pdf.font.family'], '.');

            // End of TOC page.
            $this->endTOCPage();

            $this->pageType = self::C__PAGE_TYPE__CONTENT;
        }

        return $this;
    }

    /**
     * Adds a new page to the document.
     *
     * @return $this
     */
    public function addNewPage(): self
    {
        $this->AddPage();

        return $this;
    }

    /**
     * @return $this
     */
    public function beforeExport(): self
    {
        if (isset($this->m_options['general.index-page']) && $this->m_options['general.index-page']) {
            $this->pageType = self::C__PAGE_TYPE__INDEX;

            // Add the 'index' page content.
            $this->addContent($this->m_options['index']);
            $this->endPage();

            $this->pageType = self::C__PAGE_TYPE__CONTENT;

            // Add page if we show the TOC page so we can overwrite it later.
            $this->AddPage();
        }

        $this->lastIndexPage = $this->getPage() - 1;

        return $this;
    }

    /**
     * Method for replacing some variables.
     *
     * @param string $p_text
     *
     * @return  string
     */
    protected function replace_variables($p_text)
    {
        $p_replacements = [
            '%%CURRENT_PAGE%%'     => $this->getAliasNumPage(),
            '%%TOTAL_PAGES%%'      => $this->getAliasNbPages(),
            '%%REVISION_ID%%'      => $this->m_options['revisionId'],
            '%%REVISION_COMMENT%%' => $this->m_options['revisionComment'],
        ];

        // On the "table of contents" page, the current page is "FALSE".
        if ($this->pageType !== self::C__PAGE_TYPE__CONTENT) {
            $p_replacements['%%CURRENT_PAGE%%'] = '';
            $p_replacements['%%TOTAL_PAGES%%'] = '';
        }

        return strtr($p_text, $p_replacements);
    }

    /**
     * Method for getting the i-doit font upload directory.
     *
     * @static
     * @return string
     */
    public static function getFontUploadDir(): string
    {
        return isys_application::instance()->app_path . '/upload/fonts';
    }

    /**
     * @return string
     */
    public function getContent(): string
    {
        return '';
    }

    /**
     * Helper function to reset the paging to 1 at the current page.
     *
     * @return void
     */
    public function resetPageNumber(): void
    {
        if (!isset($this->m_options['general.toc']) || !$this->m_options['general.toc']) {
            $this->starting_page_number = 2 - $this->getPage();
        } else {
            $this->starting_page_number = 1 - $this->getPage();
        }
    }
}
