<?php

/**
 * i-doit
 *
 * Super export class
 *
 * @package     i-doit
 * @subpackage  Modules
 * @author      Selcuk Kekec <skekec@i-doit.com>
 * @version     1.0.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.4.0
 */
abstract class isys_document_export
{
    /**
     * Document export formatter.
     *
     * @var  isys_document_formattable
     */
    protected $m_formatter;

    /**
     * Document title.
     *
     * @var  string
     */
    protected $m_title;

    /**
     * Chapter-Models.
     * @var  array
     */
    protected $m_models;

    /**
     * Optional options.
     *
     * @var  array
     */
    protected $m_options = [];

    /**
     * Styles.
     *
     * @var  string
     */
    protected $m_style;

    /**
     * Set style rules.
     *
     * @return  string
     */
    protected function get_style()
    {
        return '<style type="text/css">' . $this->m_style . '</style>';
    }

    /**
     * Set style.
     *
     * @param   string $p_style
     *
     * @return  $this
     */
    public function set_style($p_style)
    {
        $this->m_style = $p_style;

        return $this;
    }

    /**
     * Append style rules to existing.
     *
     * @param   string $p_style
     *
     * @return  $this
     */
    public function append_style($p_style)
    {
        $this->m_style .= $p_style;

        return $this;
    }

    /**
     * Define some optional export options.
     *
     * @param   array $p_options
     *
     * @return  $this
     */
    public function set_options(array $p_options)
    {
        if (count($p_options) > 0) {
            foreach ($p_options as $l_key => $l_value) {
                $this->m_options[$l_key] = $l_value;
            }
        }

        return $this;
    }

    /**
     * Format setter.
     *
     * @param   isys_document_formattable $p_formatter
     *
     * @return  $this
     */
    private function set_formatter(isys_document_formattable $p_formatter)
    {
        $this->m_formatter = $p_formatter;

        return $this;
    }

    /**
     * Instantiate and initilize the exporter.
     *
     * @param  array $p_models
     */
    public function __construct(array $p_models, isys_document_formattable $p_formatter)
    {
        $this->set_models($p_models);
        $this->set_formatter($p_formatter);

        $this->init();
    }

    /**
     * Get title.
     *
     * @return  string
     */
    public function get_title()
    {
        return $this->m_title;
    }

    /**
     * Set title.
     *
     * @param   string $p_title
     *
     * @return  isys_document_export
     */
    public function set_title($p_title)
    {
        $this->m_title = $p_title;

        return $this;
    } // function

    /**
     * Set whole models array.
     *
     * @param   array $p_models
     *
     * @return  isys_document_export
     */
    public function set_models(array $p_models)
    {
        $this->m_models = $p_models;

        return $this;
    }

    /**
     * Get whole models array.
     *
     * @return  array
     */
    public function get_models()
    {
        return $this->m_models;
    }

    /**
     * Add export model to exporter.
     *
     * @param  isys_document_export_model $p_model
     */
    public function add_model(isys_document_export_model $p_model)
    {
        $this->m_models[] = $p_model;
    }

    /**
     * Export the HTML stored in $m_raw_text to the desiered format.
     *
     * @return  isys_document_export
     */
    abstract public function export();

    /**
     * Write export to file.
     *
     * @param string      $p_filename
     * @param null|string $filePath
     *
     * @return  isys_document_export
     */
    abstract public function write($p_filename, $filePath = null);

    /**
     * Send export to browser using a suggested filename.
     *
     * @return  isys_document_export
     */
    abstract public function send($p_filename);

    /**
     * Get file extension.
     */
    abstract public function get_file_extension();

    /**
     * Returns the mimetype.
     */
    abstract public function get_mimetype();

    /**
     * Initialize the exporter.
     */
    abstract protected function init();

    /**
     * Path to our exported file.
     *
     * @param   string $p_filename
     *
     * @return  string
     */
    public function get_filepath($p_filename = '')
    {
        if (empty($p_filename)) {
            $p_filename = 'export';
        }

        return isys_module_document::get_export_dir() . DS . $p_filename . '.' . $this->get_file_extension();
    }
}
