<?php

/**
 *
 * @package     i-doit
 * @subpackage  Modules
 * @author      Dennis Stücken <dstuecken@syneticd.de>
 * @version     1.1
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class isys_document_format_pdf extends TCPDF implements isys_document_formattable
{
    /*
     * @see  DOKU-158  Constant for calculating the footer scale.
     */
    const C__FOOTER__PDF__SCALER = 1.4;

    /*
     * @see  DOKU-309  Constants for the current "page type".
     */
    const C__PAGE_TYPE__INDEX   = 'index';
    const C__PAGE_TYPE__TOC     = 'toc';
    const C__PAGE_TYPE__CONTENT = 'content';

    /**
     * PDF Options.
     *
     * @var array
     */
    private $m_options = [];

    /**
     * Footer heigth.
     *
     * @var int
     */
    private $footerHeight = 0;

    /**
     * In order to be styled correctly, we need to pass the styles with every "writeHTML" call.
     *
     * @var string
     */
    private $style = '';

    /**
     * Variable contains the current "page type".
     *
     * @var string
     */
    private $pageType = self::C__PAGE_TYPE__CONTENT;

    /**
     * @var  array
     */
    protected $m_formatOptions = [
        'pdf.title'              => [
            'title'       => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__OPTION_TITLE',
            'type'        => 'text',
            'default'     => '',
            'placeholder' => '',
            'p_strClass'  => 'input-small'
        ],
        'pdf.subject'            => [
            'title'       => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__OPTION_SUBJECT',
            'type'        => 'text',
            'default'     => '',
            'placeholder' => '',
            'p_strClass'  => 'input-small'
        ],
        'pdf.keywords'           => [
            'title'       => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__OPTION_KEYWORDS',
            'type'        => 'textarea',
            'default'     => '',
            'placeholder' => '',
            'p_strClass'  => 'input-small'
        ],
        'pdf.margin-top'         => [
            'title'       => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__EXPORT_M_TOP',
            'type'        => 'text',
            'default'     => '20',
            'placeholder' => '20',
            'p_strStyle'  => 'width:70px;',
            'p_nMaxLen'   => 3,
            'description' => 'mm'
        ],
        'pdf.margin-bottom'      => [
            'title'       => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__EXPORT_M_BOTTOM',
            'type'        => 'text',
            'default'     => '20',
            'placeholder' => '20',
            'p_strStyle'  => 'width:70px;',
            'p_nMaxLen'   => 3,
            'description' => 'mm'
        ],
        'pdf.margin-left'        => [
            'title'       => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__EXPORT_M_LEFT',
            'type'        => 'text',
            'default'     => '20',
            'placeholder' => '20',
            'p_strStyle'  => 'width:70px;',
            'p_nMaxLen'   => 3,
            'description' => 'mm'
        ],
        'pdf.margin-right'       => [
            'title'       => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__EXPORT_M_RIGHT',
            'type'        => 'text',
            'default'     => '20',
            'placeholder' => '20',
            'p_strStyle'  => 'width:70px;',
            'p_nMaxLen'   => 3,
            'description' => 'mm'
        ],
        'pdf.font.family'        => [
            'title'      => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__DEFAULT_FONT',
            'type'       => 'select',
            'default'    => 'helvetica',
            'options'    => [],
            'p_strClass' => 'input-mini'
        ],
        'pdf.font.size'          => [
            'title'       => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__DEFAULT_FONT_SIZE',
            'type'        => 'text',
            'default'     => '10',
            'placeholder' => '10',
            'p_strStyle'  => 'width:70px;',
            'p_nMaxLen'   => 3
        ],
        'pdf.orientation'        => [
            'title'      => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__ORIENTATION',
            'type'       => 'select',
            'default'    => 'p',
            'options'    => [
                'p' => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__ORIENTATION_PORTRAIT',
                'l' => 'LC__MODULE__DOCUMENT__DOCUMENT_TEMPLATE__ORIENTATION_LANDSCAPE'
            ],
            'p_strClass' => 'input-mini'
        ],
        'pdf.background.index'   => [
            'type'   => 'text',
            'hidden' => true
        ],
        'pdf.background.toc'     => [
            'type'   => 'text',
            'hidden' => true
        ],
        'pdf.background.content' => [
            'type'   => 'text',
            'hidden' => true
        ]
    ];

    /**
     * PDF default font.
     *
     * @var string
     */
    protected $m_defaultFont = 'helvetica';

    /**
     * Default page unit.
     *
     * @var string
     */
    protected $m_defaultPageUnit = 'mm';

    /**
     * Retrieves the format option definition for TCPDF.
     *
     * @return array
     */
    public function getFormatOptionDefinition()
    {
        $l_fontList = [
            'helvetica' => 'helvetica'
        ];

        foreach ($this->getFontsList() as $l_font) {
            $l_fontList[$l_font] = $l_font;
        }

        ksort($l_fontList);

        $this->m_formatOptions['pdf.font.family']['options'] = $l_fontList;

        return [
            'LC__DOCUMENT__PDF_SETTINGS' => $this->m_formatOptions
        ];
    }

    /**
     * @param string $style
     *
     * @return isys_document_format_pdf
     */
    public function setStyle(string $style): isys_document_format_pdf
    {
        $this->style = $style;

        return $this;
    }

    /**
     * Process the PDF header.
     *
     * @return void
     */
    public function Header()
    {
        $image = null;
        $imagePath = isys_module_document::get_upload_dir() . '/';
        $optionKey = 'pdf.background.' . $this->pageType;

        if (isset($this->m_options[$optionKey]) && !empty($this->m_options[$optionKey])) {
            $image = $this->m_options[$optionKey];

            if (!file_exists($imagePath . $image)) {
                $image = null;
            }
        }

        // @see  DOKU-309  If we defined a background image (that also exists in the filesystem) we display it underneath the content.
        if ($image) {
            $width = 210;
            $height = 297;

            // Swap the width and height for landscape orientation.
            if (isset($this->m_options['pdf.orientation']) && $this->m_options['pdf.orientation'] === 'l') {
                $width = 297;
                $height = 210;
            }

            // Get the current page break margin and 'auto-page-break' mode.
            $margin = $this->getBreakMargin();
            $autoPageBreak = $this->AutoPageBreak;

            // Disable auto-page-break.
            $this->SetAutoPageBreak(false, 0);

            // Set background image.
            $this->Image($imagePath . $image, 0, 0, $width, $height, '', '', '', false, 300, '', false, false, 0);

            // Restore auto-page-break status and set the starting point for the page content.
            $this->SetAutoPageBreak($autoPageBreak, $margin);
            $this->setPageMark();
        }

        if ($this->pageType !== self::C__PAGE_TYPE__INDEX && isset($this->m_options['header'])) {
            // This needs to be done.
            $this->SetFont(
                $this->m_options['pdf.font.family'],
                '',
                isset($this->m_options['pdf.font.size']) ? ($this->m_options['pdf.font.size'] > 5 ? $this->m_options['pdf.font.size'] : 10) : 10
            );

            $topmargin = ((int)$this->m_options['pdf.margin-top']);

            if ($topmargin > 0) {
                $this->setY($topmargin);
            }

            $this->writeHTML($this->style . $this->replace_variables($this->m_options['header']), true, false, true, false, '');
        }
    }

    /**
     * We overwrite this method to correctly handle fonts, that are included from other directories.
     *
     * @param string    $family
     * @param string    $style
     * @param null      $size
     * @param string    $fontfile
     * @param string    $subset
     * @param bool|true $out
     *
     * @return mixed|void
     */
    public function SetFont($family, $style = '', $size = null, $fontfile = '', $subset = 'default', $out = true)
    {
        try {
            return parent::SetFont($family, $style, $size, '', $subset, $out);
        } catch (Exception $e) {
            isys_notify::warning($e->getMessage(), ['life' => 10]);

            return parent::SetFont('helvetica', $style, $size, '', $subset, $out);
        }
    }

    /**
     * Process the PDF footer.
     *
     * @return void
     */
    public function Footer()
    {
        if ($this->pageType !== self::C__PAGE_TYPE__INDEX && isset($this->m_options['footer'])) {
            // This needs to be done.
            $this->SetFont(
                $this->m_options['pdf.font.family'],
                '',
                isset($this->m_options['pdf.font.size']) ? ($this->m_options['pdf.font.size'] > 5 ? $this->m_options['pdf.font.size'] : 10) : 10
            );

            // @see  DOKU-158
            $footer = str_replace('<p>&nbsp;</p>', '', $this->m_options['footer']);

            $this->setY('-' . $this->footerHeight);
            $this->SetAutoPageBreak(false);
            $this->writeHTML($this->style . $this->replace_variables($footer));
            $this->SetAutoPageBreak(true, $this->footerHeight);
        }
    }

    /**
     * Initialize formatter
     *
     * @param array $p_options
     *
     * @return $this
     */
    public function initialize($p_options)
    {
        $this->m_options = $p_options;

        $l_orientation = '';

        if (isset($this->m_options['pdf.orientation'])) {
            $l_orientation = $this->m_options['pdf.orientation'];
        }

        // Initialize configured font
        $this->initFont();

        // @See DOKU-158
        $footerLineCount = 1;
        $footer = str_replace('<p>&nbsp;</p>', '', $p_options['footer']);

        if (!empty($footer)) {
            $domObj = new DOMDocument();
            $domObj->loadHTML($footer);

            if ($domObj->getElementsByTagName('table')->length > 0) {
                $footerLineCount = $domObj->getElementsByTagName('tr')->length;
            } else {
                $footerLineCount = $domObj->getElementsByTagName('p')->length;
            }
        }

        // @See DOKU-158 Calculate footer height which considers the height of the footer content
        if (isset($p_options['footer'])){
            $this->footerHeight = (($this->getFontSize() * $this->getScaleFactor() * $footerLineCount) / self::C__FOOTER__PDF__SCALER) + $p_options['pdf.margin-bottom'];
        }

        // Page orientation
        $this->setPageOrientation($l_orientation, '', $this->footerHeight);

        // Set default page unit
        $this->setPageUnit($this->m_defaultPageUnit);

        if (isset($this->m_options['header'])) {
            $p_options['pdf.margin-top'] *= 2;
        }

        // Default margins
        $this->SetMargins($p_options['pdf.margin-left'], $p_options['pdf.margin-top'], $p_options['pdf.margin-right']);

        // Set PDF title
        if (isset($p_options['pdf.title'])) {
            $this->SetTitle($p_options['pdf.title']);
        }

        // Set PDF subject
        if (isset($p_options['pdf.subject'])) {
            $this->SetSubject($p_options['pdf.subject']);
        }

        // Set PDF keywords
        if (isset($p_options['pdf.keywords'])) {
            $this->SetSubject($p_options['pdf.keywords']);
        }

        $this->AddPage();

        return $this;
    }

    /**
     * Fill the list of available fonts ($this->fontlist).
     *
     * @return array
     */
    public function getFontsList()
    {
        parent::getFontsList();

        $l_font_upload_dir = self::getFontUploadDir();

        // @see ID-2265 / DOKU-60
        if (file_exists($l_font_upload_dir) && is_readable($l_font_upload_dir)) {
            $l_fonts = array_map('basename', glob($l_font_upload_dir . '/*.php'));

            if (count($l_fonts)) {
                foreach ($l_fonts as $l_font) {
                    $this->fontlist[] = substr($l_font, 0, -4);
                }
            }
        }

        return $this->fontlist;
    }

    /**
     * Initialize configured font from options
     *
     * @return $this
     */
    protected function initFont()
    {
        if (!isset($this->m_options['pdf.font.family'])) {
            $this->m_options['pdf.font.family'] = $this->m_defaultFont;
        }

        // Set font family.
        $this->SetFont($this->m_options['pdf.font.family'], '',
            isset($this->m_options['pdf.font.size']) ? ($this->m_options['pdf.font.size'] > 5 ? $this->m_options['pdf.font.size'] : 10) : 10);

        return $this;
    }

    /**
     * Write formatted content to $p_filename
     *
     * @param $p_filename
     *
     * @return $this
     */
    public function save($p_filename)
    {
        $this->lastPage();
        $this->output($p_filename, 'F');

        return $this;
    }

    /**
     * Send formatted content to browser.
     *
     * @param   string  $p_filename
     * @param   boolean $p_inline
     *
     * @return  $this
     */
    public function send($p_filename, $p_inline = false)
    {
        $this->lastPage();
        $this->output($p_filename, $p_inline ? 'I' : 'D');

        return $this;
    }

    /**
     * Add Chapter content.
     *
     * @param   string  $p_text
     *
     * @return  $this
     */
    public function addContent($p_text)
    {
        $this->writeHTML($this->style . $this->replace_variables($p_text), true, false, true, false, '');

        return $this;
    }

    /**
     * Add Chapter heading.
     *
     * @param   string  $title
     * @param   string  $p_pos
     * @param   integer $p_level
     *
     * @return  $this
     */
    public function addHeadline($title, $p_pos, $p_level = 0)
    {
        $position = $this->m_options['general.hx.show_numbers'] ? $p_pos : '';

        if ($position || $position === 0) {
            $title = $position . ' ' . $title;
        }

        $this->Bookmark($title, ($p_level - 1), 0, '', 'B', [0, 0, 0]);

        $fontColor = '#000000';
        $fontColorKey = 'general.h' . $p_level . '.color';

        if (isset($this->m_options[$fontColorKey])) {
            $fontColor = $this->m_options[$fontColorKey];
        }

        // @see  DOKU-308  Pass the defined headline-sizes.
        $fontSize = '';
        $fontSizeKey = 'general.h' . $p_level . '.size';

        if (isset($this->m_options[$fontSizeKey]) && is_numeric($this->m_options[$fontSizeKey])) {
            $fontSize = 'font-size:' . $this->m_options[$fontSizeKey] . 'pt;';
        }

        $this->writeHTML('<h' . $p_level . ' style="color:' . $fontColor . ';' . $fontSize . '">' . $title . '</h' . $p_level . '>', true, false, true);

        return $this;
    }

    /**
     * @return  $this
     */
    public function afterExport()
    {
        $tocPageNumber = 1;

        $this->endPage();

        if (isset($this->m_options['general.index-page']) && $this->m_options['general.index-page']) {
            $this->pageType = self::C__PAGE_TYPE__INDEX;

            $tocPageNumber = 2;

            // Adding the "index" page and moving it to the beginning.
            $this
                ->addNewPage()
                ->addContent($this->m_options['index'], null, null)
                ->movePage($this->getPage(), 1);

            $this->pageType = self::C__PAGE_TYPE__CONTENT;
        }

        if (isset($this->m_options['general.toc']) && $this->m_options['general.toc']) {
            $this->pageType = self::C__PAGE_TYPE__TOC;

            // Add a new page for TOC.
            $this->addTOCPage('', '', false);

            // Initialize configured font.
            $this->initFont();

            // DOKU-122
            $this->setStartingPageNumber(1);
            $this->addTOC($tocPageNumber, $this->m_options['pdf.font.family'], '.');

            // End of TOC page.
            $this->endTOCPage();

            $this->pageType = self::C__PAGE_TYPE__CONTENT;
        }

        return $this;
    }

    /**
     * Adds a new page to the document
     */
    public function addNewPage()
    {
        $this->AddPage();

        return $this;
    }

    /**
     * @return  $this
     */
    public function beforeExport()
    {
        return $this;
    }

    /**
     * Method for replacing some variables.
     *
     * @param   string $p_text
     *
     * @return  string
     */
    protected function replace_variables($p_text)
    {
        $p_replacements = [
            '%%CURRENT_PAGE%%'     => $this->getAliasNumPage(),
            '%%TOTAL_PAGES%%'      => $this->getAliasNbPages(),
            '%%REVISION_ID%%'      => $this->m_options['revisionId'],
            '%%REVISION_COMMENT%%' => $this->m_options['revisionComment'],
        ];

        // On the "table of contents" page, the current page is "FALSE".
        if ($this->pageType !== self::C__PAGE_TYPE__CONTENT) {
            $p_replacements['%%CURRENT_PAGE%%'] = '';
            $p_replacements['%%TOTAL_PAGES%%'] = '';
        }

        return strtr($p_text, $p_replacements);
    }

    /**
     * Method for getting the i-doit font upload directory.
     *
     * @static
     * @return  string
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public static function getFontUploadDir()
    {
        return isys_application::instance()->app_path . '/upload/fonts';
    }

    /**
     * @inheritdoc
     */
    public function getContent()
    {
        return '';
    }
}
