<?php

use idoit\Module\Document\CustomFields\Repository\CustomFieldsRepository;

/**
 * i-doit
 *
 * Document DAO.
 *
 * @package     i-doit
 * @subpackage  Modules
 * @author      Selcuk Kekec <skekec@i-doit.com>
 * @version     1.0.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.3.0
 */
class isys_document_dao_revision extends isys_component_dao
{
    /**
     * Method for retrieving documents.
     *
     * @param   int|array $p_revision_id
     * @param   string    $p_type
     * @param   string    $p_condition
     *
     * @return  isys_component_dao_result
     * @author  Selcuk Kekec <skekec@i-doit.com>
     */
    public function get_data($p_revision_id = null, $p_type = null, $p_condition = null)
    {
        $l_sql = 'SELECT * FROM isys_document_revision
            INNER JOIN isys_document ON isys_document_revision__isys_document__id = isys_document__id
            LEFT JOIN isys_document_type ON isys_document__isys_document_type__id = isys_document_type__id
            LEFT JOIN isys_catg_document_list ON isys_catg_document_list__isys_document__id = isys_document__id
            LEFT JOIN isys_obj ON isys_catg_document_list__isys_obj__id = isys_obj__id
            LEFT JOIN isys_cats_person_list user ON user.isys_cats_person_list__isys_obj__id = isys_document_revision__isys_obj__id
            LEFT JOIN isys_document_template ON isys_document__isys_document_template__id = isys_document_template__id
            WHERE TRUE ';

        if ($p_revision_id !== null) {
            if (is_array($p_revision_id)) {
                $l_sql .= ' AND isys_document_revision__id IN (' . implode(', ', array_map('intval', $p_revision_id)) . ')';
            } else {
                $l_sql .= ' AND isys_document_revision__id = ' . $this->convert_sql_id($p_revision_id);
            }
        }

        if ($p_type !== null) {
            $l_sql .= ' AND isys_document__isys_document_type__id = ' . $this->convert_sql_text($p_type);
        }

        return $this->retrieve($l_sql . ' ' . $p_condition . ';');
    }

    /**
     * Method for creating/saving a document.
     *
     * @param   integer $p_id
     * @param   array   $p_values
     * @param   integer $p_objectID
     *
     * @return  integer
     * @author  Selcuk Kekec <skekec@i-doit.com>
     */
    public function saveRevision($p_id = null, array $p_values = [], $p_objectID = null)
    {
        $l_data = [];

        if ($p_id === null || empty($p_id)) {
            $l_sql = 'INSERT INTO isys_document_revision SET %s;';
        } else {
            $l_sql = 'UPDATE isys_document_revision SET %s WHERE isys_document_revision__id = ' . $this->convert_sql_id($p_id) . ';';
        }

        if (count($p_values) > 0) {
            foreach ($p_values as $l_key => $l_value) {
                $l_data[] = 'isys_document_revision__' . $l_key . ' = ' . $this->convert_sql_text($l_value);
            }

            if ($this->update(str_replace('%s', implode(', ', $l_data), $l_sql)) && $this->apply_update()) {
                $p_id = $p_id ?: $this->get_last_insert_id();
            }
        }

        return $p_id;
    }

    /**
     * Remove a document.
     *
     * @param   mixed $p_id
     *
     * @return  boolean
     * @author  Selcuk Kekec <skekec@i-doit.com>
     */
    public function deleteRevision($p_id)
    {
        if (!is_array($p_id)) {
            $p_id = [$p_id];
        }

        return ($this->update('DELETE FROM isys_document_revision WHERE isys_document_revision__id IN (' . implode(',', array_map('intval', $p_id)) . ');') &&
            $this->apply_update());
    }

    /**
     *
     * Insert new chapter.
     *
     * @param   integer                               $p_documentID
     * @param   isys_document_compiler_model_document $p_model_document
     *
     * @return  boolean
     */
    public function insert_chapters($p_documentID, isys_document_compiler_model_document $p_model_document)
    {
        $l_chapter_models = $p_model_document->get_chapters();

        if (count($l_chapter_models)) {
            $l_data = [];
            foreach ($l_chapter_models as $l_chapter_model) {
                $l_options = isys_format_json::encode(['new_page' => (int)$l_chapter_model->get_new_page()]);

                $l_data[] = '(' .
                    $this->convert_sql_id($p_documentID) . ',' .
                    $this->convert_sql_text($l_chapter_model->get_compiled_text()) . ',' .
                    $this->convert_sql_int($l_chapter_model->get_level()) . ',' .
                    $this->convert_sql_text($l_chapter_model->get_chapter_num()) . ',' .
                    $this->convert_sql_text($l_chapter_model->get_title()) . ',' .
                    $this->convert_sql_text($l_options) .
                    ')';
            }

            if (count($l_data)) {
                $l_sql = 'INSERT INTO isys_document_text (isys_document_text__isys_document_revision__id, isys_document_text__text, isys_document_text__level, isys_document_text__pos, isys_document_text__title, isys_document_text__options) VALUES ';

                return ($this->update($l_sql . implode(",", $l_data) . ';') && $this->apply_update());
            }

            return true;
        }

        return false;
    }

    /**
     * Creates a new revision
     *
     * @param integer                               $p_document_id
     * @param isys_document_compiler_model_document $p_model_document
     * @param integer                               $p_creator
     * @param string                                $comment
     */
    public static function create_revision($p_document_id, isys_document_compiler_model_document $p_model_document, $p_creator, $comment = null)
    {
        global $g_comp_database;

        $l_instance = new self($g_comp_database);

        $customFields = [];
        $customFieldValues = (new CustomFieldsRepository())->getCustomFieldsForDocument($p_document_id);

        foreach ($customFieldValues as $customFieldValue) {
            $customFields[$customFieldValue->getName()] = $customFieldValue->getValue()->getFormattedValue() ?: $customFieldValue->getDefaultValue()->getFormattedValue();
        }

        // Create revision.
        $l_revision_id = $l_instance->saveRevision(null, [
            'isys_document__id' => $p_document_id,
            'created'           => date('Y-m-d H:i:s'),
            'index'             => $p_model_document->get_index(),
            'header'            => $p_model_document->get_header(),
            'footer'            => $p_model_document->get_footer(),
            'isys_obj__id'      => $p_creator,
            'comment'           => $comment,
            'custom_fields'     => isys_format_json::encode($customFields)
        ]);

        if ($l_revision_id) {
            $l_instance->insert_chapters($l_revision_id, $p_model_document);
        }
    }

    /**
     * Replace all chapters of a given revision.
     *
     * @param  int                                    $p_documentID
     * @param  isys_document_compiler_model_document  $p_model_document
     *
     * @return bool
     */
    public function replace_chapters($p_documentID, isys_document_compiler_model_document $p_model_document)
    {
        $sql = 'DELETE FROM isys_document_text WHERE isys_document_text__isys_document_revision__id = ' . $this->convert_sql_id($p_documentID) . ';';

        if ($this->update($sql) && $this->apply_update()) {
            return $this->insert_chapters($p_documentID, $p_model_document);
        }

        return false;
    }

    /**
     * Get id of last revision for document.
     *
     * @param   integer $p_document_id
     *
     * @return  int|bool
     */
    public function get_last_revision($p_document_id)
    {
        $l_sql = 'SELECT isys_document_revision__id FROM isys_document_revision
            INNER JOIN isys_document ON isys_document__id = isys_document_revision__isys_document__id
            WHERE TRUE AND isys_document__id = ' . $this->convert_sql_id($p_document_id) . '
            ORDER BY isys_document_revision__id DESC LIMIT 1;';

        $l_res = $this->retrieve($l_sql);

        if (count($l_res)) {
            $l_row = $l_res->get_row();

            return $l_row['isys_document_revision__id'];
        }

        return false;
    }

    /**
     * Get all revisions of a document.
     *
     * @param   integer $p_document_id
     *
     * @return  isys_component_dao_result
     */
    public function get_revisions_of_document($p_document_id)
    {
        $condition = ' AND isys_document_revision__isys_document__id = ' . $this->convert_sql_id($p_document_id) . ' ORDER BY isys_document_revision__created DESC';

        return $this->get_data(null, null, $condition);
    }

    /**
     * Load models of document as array.
     *
     * @param   integer $p_document_id
     * @param   integer $p_revision_id
     *
     * @return  array  Array of isys_document_export_model.
     */
    public function load_models_for_revision($p_document_id, $p_revision_id = null)
    {
        if (empty($p_revision_id)) {
            $p_revision_id = $this->get_last_revision($p_document_id);
        }

        $l_models = [];

        // Select models.
        $l_document_text_res = $this->retrieve('SELECT *
            FROM isys_document_text
            WHERE isys_document_text__isys_document_revision__id = ' . $this->convert_sql_id($p_revision_id) . ';');

        if (count($l_document_text_res)) {
            $l_models = [];

            // Build model array.
            while ($l_row = $l_document_text_res->get_row()) {
                $l_tmp = new isys_document_export_model();

                $l_options = isys_format_json::decode($l_row['isys_document_text__options']);

                $l_tmp->set_level((int)$l_row['isys_document_text__level'])
                    ->set_pos($l_row['isys_document_text__pos'])
                    ->set_text($l_row['isys_document_text__text'])
                    ->set_title($l_row['isys_document_text__title']);

                if (is_numeric($l_options['new_page'])) {
                    $l_tmp->set_new_page((bool)$l_options['new_page']);
                }

                $l_models[$l_row['isys_document_text__pos']] = $l_tmp;
                unset($l_tmp);
            }
        }

        // This is necessary for sorting the chapters correctly.
        uksort($l_models, function ($a, $b) {
            return strnatcmp($a, $b);
        });

        return array_values($l_models);
    }
}
