<?php

use Symfony\Component\Filesystem\Filesystem;

/**
 * i-doit
 *
 * Exports a document to JSON.
 *
 * @package     i-doit
 * @subpackage  Modules
 * @version     1.0.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.4.5
 */
class isys_document_export_template_json
{
    /**
     * Database instance.
     *
     * @var isys_component_database
     */
    protected $m_db = null;

    /**
     * The template ID.
     *
     * @var integer
     */
    private $m_template_id = null;

    /**
     * This variable holds the temporary template export dir.
     *
     * @var string
     */
    private $m_export_dir = null;

    /**
     * This variable holds the temporary template image export dir.
     *
     * @var string
     */
    private $m_export_dir_images = null;

    /**
     * This variable will hold our LOG instance.
     *
     * @var isys_log
     */
    private $m_log = null;

    /**
     * This array will hold the complete export data.
     *
     * @var array
     */
    private $m_export_data = [
        'document' => [],
        'chapter'  => [],
        'reports'  => []
    ];

    /**
     * This array will hold all images, used in the document.
     *
     * @var array
     */
    private $m_export_images = [];

    /**
     * This array will hold all reports, used in the document.
     *
     * @var array
     */
    private $m_used_reports = [];

    /**
     * Constructor.
     *
     * @param int $p_template_id
     */
    public function __construct($p_template_id)
    {
        $this->m_db = isys_application::instance()->container->get('database');
        $this->m_template_id = (int)$p_template_id;
        $this->m_export_dir = isys_module_document::get_export_dir() . '/template-temp';
        $this->m_export_dir_images = $this->m_export_dir . '/images';

        $this->m_log = isys_log::get_instance('document-template-export');
    }

    /**
     * Method for returning the log instance.
     *
     * @return  isys_log
     */
    public function get_log()
    {
        return $this->m_log;
    }

    /**
     * Export method for preparing and writing the template data to a JSON file.
     *
     * @param bool $p_as_zip
     *
     * @return array
     * @throws \idoit\Exception\JsonException
     * @throws isys_exception_database
     * @throws isys_exception_filesystem
     */
    public function export(bool $p_as_zip = false): array
    {
        $this->m_log->info('Starting the export!');

        // At first we create an empty directory to put our export in. Or we empty the existing one.
        if (is_dir($this->m_export_dir)) {
            (new isys_component_filemanager())->remove_files($this->m_export_dir);
        }

        if (!is_writable(dirname($this->m_export_dir))) {
            throw new isys_exception_filesystem('The directory "' . dirname($this->m_export_dir) . '" is not writable!', '');
        }

        mkdir($this->m_export_dir);
        mkdir($this->m_export_dir_images);

        $this->prepare_template_data()
            ->prepare_chapter_data()
            ->exportImages()
            ->include_reports();

        // The JSON file shall always be called "data.json".
        $l_filename = 'data.' . $this->get_file_extension();

        file_put_contents($this->m_export_dir . '/' . $l_filename, isys_format_json::encode($this->m_export_data));

        if ($p_as_zip) {
            try {
                $this->m_log->info('Creating ZIP file.');

                // Now we ZIP the directory.
                $l_zip = new ZipArchive();

                // We give the ZIP a better name than "data".
                $l_zipfilename = 'document-template-' . $this->m_template_id . '.zip';

                if ($l_zip->open(isys_module_document::get_export_dir() . '/' . $l_zipfilename, ZipArchive::CREATE) === true) {
                    // Inside the ZIP file we want a unified name for the JSON.
                    $l_zip->addFile($this->m_export_dir . '/' . $l_filename, $l_filename);

                    if (count($this->m_export_images)) {
                        $l_zip->addEmptyDir('images');

                        foreach ($this->m_export_images as $l_img) {
                            $l_zip->addFile($this->m_export_dir_images . '/' . $l_img['new-filename'], 'images/' . $l_img['new-filename']);
                        }
                    }

                    $l_zip->close();
                }

                $this->m_log->info('...all done!');
                $l_filename = $l_zipfilename;
            } catch (Exception $e) {
                new isys_exception_general('While creating the ZIP archive, an error occured: ' . $e->getMessage());
            }

            // After zipping, we remove the template temp directory.
            (new isys_component_filemanager())->remove_files($this->m_export_dir);
        }

        return [
            'filename' => $l_filename,
            'filepath' => isys_module_document::get_export_dir() . '/' . $l_filename,
            'log'      => $this->m_log->get_log(true)
        ];
    }

    /**
     * Method for preparing the main template data.
     *
     * @return $this
     * @throws \idoit\Exception\JsonException
     * @throws isys_exception_database
     */
    protected function prepare_template_data(): self
    {
        $l_dao_template = new isys_document_dao_templates($this->m_db);

        $l_template_data = $l_dao_template->get_data($this->m_template_id)
            ->get_row();

        $this->m_export_data['document']['title'] = trim($l_template_data['isys_document_template__title']);
        $this->m_export_data['document']['category'] = trim($l_template_data['isys_document_type__title']);
        $this->m_export_data['document']['header'] = trim($this->handle_dynamic_content($l_template_data['isys_document_template__header']));
        $this->m_export_data['document']['footer'] = trim($this->handle_dynamic_content($l_template_data['isys_document_template__footer']));
        $this->m_export_data['document']['options'] = isys_format_json::decode($l_template_data['isys_document_template__options']);

        return $this;
    }

    /**
     * Method for preparing the chapter data.
     *
     * @return $this
     */
    protected function prepare_chapter_data(): self
    {
        $l_dao_chapters = new isys_document_dao_chapters($this->m_db);

        $l_chapter_res = $l_dao_chapters->get_data(null, null, 'AND isys_document_template__id = ' . $this->m_template_id);

        if (count($l_chapter_res)) {
            while ($l_chapter_row = $l_chapter_res->get_row()) {
                $l_parent = null;
                $this->m_log->info('Exporting chapter "' . trim($l_chapter_row['isys_document_chapter__title']) . '".');

                if ($l_chapter_row['isys_document_chapter__isys_document_chapter__id'] !== null) {
                    $l_parent = trim($l_dao_chapters->get_data($l_chapter_row['isys_document_chapter__isys_document_chapter__id'])
                        ->get_row_value('isys_document_chapter__title'));
                }

                $this->m_export_data['chapter'][] = [
                    'title'  => trim($l_chapter_row['isys_document_chapter__title']),
                    'parent' => $l_parent,
                    'html'   => trim($this->handle_dynamic_content($l_chapter_row['isys_document_component__text'] ?: $l_chapter_row['text']))
                ];
            }
        }

        return $this;
    }

    /**
     * Method for finding HTML images inside a given string.
     *
     * @param string $p_html
     *
     * @return string
     * @throws \idoit\Exception\JsonException
     */
    protected function handle_dynamic_content(string $p_html): string
    {
        // First we search for IMG tags and their SRC attribute.
        $l_found = !!preg_match_all('~\<img.*?src="(.*?)".*?\>~', $p_html, $l_matches);

        // We retrieve the SRC attribute, of the found IMG tag.
        if ($l_found) {
            foreach ($l_matches[1] as $l_img_path) {
                $this->m_log->info('Found image "' . $l_img_path . '".');

                $l_data = [
                    'path'      => $l_img_path,
                    'filename'  => substr(strrchr($l_img_path, '/'), 1),
                    'extension' => strrchr($l_img_path, '.')
                ];

                $l_data['new-filename'] = isys_helper_upload::prepare_filename($l_data['filename']);
                // The "new" path needs to be relative and only inherit normal slashes "/".
                $l_data['new-path'] = str_replace([BASE_DIR, '\\'], ['', '/'], isys_module_document::get_upload_dir()) . '/' . $l_data['new-filename'];

                // We add the image information to our internal array.
                $this->m_export_images[] = $l_data;

                // We relativize the IMGs SRC attribute ("http://example.com/my-dir/image.jpg" => "src/classes/modules/document/resources/upload/7d8a9ec5_image.jpg")
                $p_html = str_replace($l_img_path, $l_data['new-path'], $p_html);
            }
        }

        // Now we search for "report" placeholder.
        $l_found = !!preg_match_all('~\<span.*?data-json=\"(.*?)\".*?data-widget-type=\"report\".*?\>~', $p_html, $l_matches);

        // We retrieve the data attribute, of the found "report" placeholder.
        if ($l_found) {
            foreach ($l_matches[1] as $l_report_json) {
                $l_report_data = isys_format_json::decode(str_replace("'", '"', $l_report_json));

                $this->m_used_reports[$l_report_data['data']['id']] = $l_report_data['data']['name'];
            }
        }

        return $p_html;
    }

    /**
     * Method which tries to export all used images.
     *
     * @return self
     */
    private function exportImages(): self
    {
        foreach ($this->m_export_images as $l_filedata) {
            try {
                // Simply copy files from the filesystem.
                if (substr($l_filedata['path'], 0, 1) === '/' && file_exists($l_filedata['new-path'])) {
                    $this->copyImage($l_filedata['new-path'], $l_filedata['new-filename']);
                }

                // Try to download 'external' images.
                if (substr($l_filedata['path'], 0, 5) === 'http:' || substr($l_filedata['path'], 0, 6) === 'https:') {
                    $this->downloadImage($l_filedata['path'], $l_filedata['new-filename']);
                }
            } catch (Throwable $e) {
                isys_notify::warning($e->getMessage(), ['life' => 10]);
            }
        }

        return $this;
    }

    /**
     * @param string $imagePath
     * @param string $imageFileName
     *
     * @return void
     * @see DOKU-458 Either copy from filesystem or try to download.
     */
    private function copyImage(string $imagePath, string $imageFileName): void
    {
        $filesystem = new Filesystem();
        $filesystem->copy($imagePath, "{$this->m_export_dir_images}/{$imageFileName}");
    }

    /**
     * @param string $imageUrl
     * @param string $imageFileName
     *
     * @return void
     * @throws Exception
     * @see DOKU-458 Either copy from filesystem or try to download.
     */
    private function downloadImage(string $imageUrl, string $imageFileName): void
    {
        $downloader = isys_application::instance()->container->get('http_client');
        $language = isys_application::instance()->container->get('language');

        $allowedImageExtensions = ['bmp', 'gif', 'ico', 'jpeg', 'jpg', 'png', 'svg', 'tif', 'tiff', 'webp'];
        $fileResponse = $downloader->request($imageUrl);

        if ($fileResponse->hasHeader('Content-Type')) {
            $imageTypes = array_filter($fileResponse->getHeader('Content-Type'), fn ($type) => strpos($type, 'image/') === 0);

            if (!count($imageTypes)) {
                throw new Exception($language->get('LC__MODULE__DOCUMENT__ERROR__EXPORT_IMAGE_MIMETYPE', $imageFileName, $fileResponse->getHeader('Content-Type')[0]));
            }
        }

        if (strrpos($imageFileName, '.') === false) {
            throw new Exception($language->get('LC__MODULE__DOCUMENT__ERROR__EXPORT_IMAGE_EXTENSION', $imageFileName));
        } else {
            $extension = substr($imageFileName, strrpos($imageFileName, '.') + 1);

            if (!in_array($extension, $allowedImageExtensions)) {
                throw new Exception($language->get('LC__MODULE__DOCUMENT__ERROR__EXPORT_IMAGE_EXTENSION', $imageFileName));
            }
        }

        isys_file_put_contents("{$this->m_export_dir_images}/{$imageFileName}", $fileResponse->getBody()->getContents());
    }

    /**
     * Method for including report-data to the export.
     *
     * @return $this
     * @throws Exception
     */
    public function include_reports(): self
    {
        global $g_comp_database_system;

        $l_dao_report = isys_report_dao::instance($g_comp_database_system);

        foreach ($this->m_used_reports as $l_report_id => $l_report_name) {
            try {
                $l_report = $l_dao_report->get_report($l_report_id);

                $this->m_log->info('Including the report "' . $l_report_name . '".');
                $this->m_export_data['reports'][] = [
                    'name'        => $l_report_name,
                    'description' => $l_report['isys_report__description'],
                    'sql'         => $l_report['isys_report__query'],
                    'data'        => $l_report['isys_report__querybuilder_data']
                ];
            } catch (Exception $e) {
                $this->m_log->warning('You are not authorized to export the report "' . $l_report_name . '".');
            }
        }

        return $this;
    }

    /**
     * Mimetype of the exported file.
     *
     * @return string
     */
    public function get_mimetype(): string
    {
        return 'application/json';
    }

    /**
     * Get the export-file extension.
     *
     * @return string
     */
    public function get_file_extension(): string
    {
        return 'json';
    }
}
