<?php

/**
 * i-doit documents module
 *
 * Online-Repository View
 *
 * @package     i-doit
 * @subpackage  Modules
 * @author      Dennis Stücken <dstuecken@synetics.de>
 * @version     1.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.0
 */
class isys_document_view_online_repository
{
    /**
     * URL to template repository, fist %s occurence is replaced with language short (= de, en)
     *
     * @var string
     */
    private $m_repository_url = 'https://r.i-doit.com/documents/templates/%s';

    /**
     * @var isys_document_dao
     */
    private $m_dao;

    /**
     * @var isys_document_dao_templates
     */
    private $m_dao_templates;

    /**
     * @var isys_component_template
     */
    private $m_tpl;

    /**
     * Cached array of online templates
     *
     * @var array
     */
    private $m_online_templates = [];

    /**
     * @throws isys_exception_auth
     * @throws isys_exception_general
     *
     * @return bool
     */
    public function process()
    {
        // Auth check.
        if (isys_auth_document::instance()->check(isys_auth::VIEW, 'REPOSITORY')) {
            try {
                $l_cache = isys_cache::keyvalue();
                $language = isys_application::instance()->container->get('language');
                $currentLanguage = isys_application::instance()->container->get('session')->get_language() ?: 'en';

                // Load available online templates.
                $this->m_online_templates = $l_cache->get('documents.online-templates.' . $currentLanguage);

                if (!$this->m_online_templates) {
                    // Cache downloaded online templates for one day to not always retrieve them again and again.
                    $l_cache->set('documents.online-templates.' . $currentLanguage, ($this->m_online_templates = $this->retrieve()), isys_convert::DAY);
                }

                if (isset($_GET['template'])) {
                    // Download selected template (get.template).
                    $l_file = $this->download($_GET['template']);

                    if ($l_file && file_exists($l_file) && $this->m_dao_templates->import_by_zip($l_file)) {
                        // Download was successfull. Import template
                        $this->m_tpl->assign('message', $language->get('LC__MODULE__IMPORT__CABLING__SUCCEEDED'));
                        unlink($l_file);
                    }
                }

                // Show a list.
                $this->listing($this->m_online_templates);
            } catch (Exception $e) {
                $this->m_tpl->assign('error', $e->getMessage());
            }
        }

        return true;
    }

    /**
     * Show available templates in an html list
     *
     * @param array $p_online_templates
     */
    private function listing(array $p_online_templates)
    {
        $online = true;
        $language = isys_application::instance()->container->get('language');

        if (function_exists('internetAvailable')) {
            $online = internetAvailable();
        }

        $this->m_tpl
            ->smarty_tom_add_rule('tom.content.bottom.buttons.*.p_bInvisible=1')
            ->include_template('contentbottomcontent', isys_module_document::getPath() . 'templates/online_repository/list.tpl')
            ->assign('content_title', $language->get('LC__MODULE__DOCUMENT__ONLINE_REPOSITORY') . ' (' . $language->get('LC__MODULE__DOCUMENT__TEMPLATE_COUNT', count($p_online_templates)) . ')')
            ->assign('online', $online)
            ->assign('onlineTemplates', $p_online_templates)
            ->assign('downloadURL', isys_helper_link::create_url([
                    C__GET__MODULE_ID     => C__MODULE__DOCUMENT,
                    C__GET__SETTINGS_PAGE => isys_module_document::CL__ACTION__ONLINE_REPOSITORY,
                    'treeNode'            => $_GET['treeNode']
                ]));
    }

    /**
     * Download a template and return its local path for importing.
     *
     * @param int $index (index in $this->m_online_templates)
     *
     * @return bool
     */
    public function download($index)
    {
        try {
            if (isys_auth_document::instance()->check(isys_auth::VIEW, 'online-repository')) {
                if (!isset($this->m_online_templates[$index])) {
                    throw new isys_exception_general('Could not download. An unknown error occurred.');
                }

                if (!isset($this->m_online_templates[$index]['url'])) {
                    throw new isys_exception_general('Error: Download URL of your selected template was not found. Try another template.');
                }

                $l_downloads = isys_module_document::download_files([$this->m_online_templates[$index]['url']]);

                if (!isset($l_downloads[$this->m_online_templates[$index]['url']])) {
                    throw new isys_exception_general('Error: Download failed');
                }

                if (is_string($l_downloads[$this->m_online_templates[$index]['url']])) {
                    if (file_put_contents(isys_glob_get_temp_dir() . 'document_template_' . $index . '.zip', $l_downloads[$this->m_online_templates[$index]['url']])) {
                        return isys_glob_get_temp_dir() . 'document_template_' . $index . '.zip';
                    }

                    throw new isys_exception_general('Error writing downloaded file to temporary location ' . isys_glob_get_temp_dir());
                }

                if ($l_downloads[$this->m_online_templates[$index]['url']] instanceof ErrorException) {
                    throw new Exception($l_downloads[$this->m_online_templates[$index]['url']]->getMessage());
                }

                throw new isys_exception_general('Error: Download retrieved an unexpected file format.');
            }
        } catch (ErrorException $e) {
            global $g_error;

            $g_error = $e->getMessage();
        } catch (Exception $e) {
            global $g_error;

            $g_error = $e->getMessage();
            isys_notify::error($e->getMessage(), isys_notify::options(null, null, true));
        }

        return false;
    }

    /**
     * Retrieve online repository. The following json format is expected:
     *
     * [
     *    {
     *       "title": "Vorlage 1",
     *       "category": "Betriebshandbuch",
     *       "url": "https://r.i-doit.com/documents/templates/1.zip",
     *       "date": "2014-09-19",
     *       "author": "Synetics GmbH"
     *    },
     *    {
     *       "title": "Vorlage 2",
     *       "category": "Handbuch",
     *       "url": "https://r.i-doit.com/documents/templates/2.zip",
     *       "date": "2014-09-19",
     *       "author": "Synetics GmbH"
     *    }
     * ]
     *
     * @return array
     * @throws Exception
     */
    private function retrieve(): array
    {
        $language = isys_application::instance()->container->get('session')->get_language() ?: 'en';
        $repositoryUrl = sprintf($this->m_repository_url, $language);

        try {
            $jsonData = isys_module_document::download_files([$repositoryUrl]);

            if (isset($jsonData[$repositoryUrl]) && isys_format_json::is_json_array($jsonData[$repositoryUrl])) {
                return isys_format_json::decode($jsonData[$repositoryUrl]);
            }

            return [];
        } catch (Throwable $e) {

        }

        return [];
    }

    /**
     * @param isys_component_template     $p_template
     * @param isys_document_dao           $p_dao_document
     * @param isys_document_dao_templates $p_dao_document_templates
     */
    public function __construct(isys_component_template $p_template, isys_document_dao $p_dao_document, isys_document_dao_templates $p_dao_document_templates)
    {
        $this->m_tpl = $p_template;
        $this->m_dao = $p_dao_document;
        $this->m_dao_templates = $p_dao_document_templates;
    }
}
