<?php

namespace idoit\Module\Events\Controller;

use idoit\Controller\Base;
use idoit\Controller\NavbarHandable;
use idoit\Module\Events\Model\Dao;
use idoit\Module\Events\Model\Subscription;
use idoit\Module\Events\View\Event;
use idoit\Module\Events\View\EventList;
use idoit\Tree\Node;
use isys_application as Application;
use isys_auth as Auth;
use isys_auth_events as AuthEvents;
use isys_component_template_navbar as ComponentNavbar;
use isys_controller as Controllable;
use isys_module as Module;
use isys_module_events as ModuleEvents;
use isys_register as Register;

/**
 * Events Main controller
 *
 * @package     Modules
 * @subpackage  Events
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class Main extends Base implements Controllable, NavbarHandable
{
    /**
     * @var Module
     */
    protected $module;

    /**
     * Default request handler, gets called in every /events request.
     *
     * @param  Register    $request
     * @param  Application $application
     *
     * @return \idoit\View\Renderable|void
     */
    public function handle(Register $request, Application $application)
    {
        $onClick = "document.isys_form.sort.value=''; document.isys_form.navMode.value=this.getAttribute('data-navmode'); $('isys_form').submit();";

        // Reset form_submit url since form_submit is not capable of the new url-rewrite paths..
        ComponentNavbar::getInstance()
            ->set_js_onclick($onClick, C__NAVBAR_BUTTON__SAVE)
            ->set_js_onclick($onClick, C__NAVBAR_BUTTON__EDIT)
            ->set_js_onclick($onClick, C__NAVBAR_BUTTON__CANCEL)
            ->set_js_onclick($onClick, C__NAVBAR_BUTTON__DELETE);
    }

    /**
     * @param  Application $application
     *
     * @return \idoit\Model\Dao\Base|Dao
     * @throws \Exception
     */
    public function dao(Application $application)
    {
        return Dao::instance($application->container->get('database'));
    }

    /**
     * @param Register             $request
     * @param Application          $application
     * @param \isys_component_tree $tree
     *
     * @return Node|\isys_tree_node
     * @throws \Exception
     */
    public function tree(Register $request, Application $application, \isys_component_tree $tree)
    {
        // Disable node sorting.
        $tree->set_tree_sort(false);

        return (new Node('Events', '', $request->get('BASE') . 'images/icons/silk/lightbulb.png'))
            ->add(new Node(
                'Hooks',
                $request->get('BASE') . 'events',
                $request->get('BASE') . 'images/icons/silk/layers.png',
                null,
                null,
                null,
                AuthEvents::instance()->is_allowed_to(Auth::VIEW, 'hooks')
            ))
            ->add(new Node(
                Application::instance()->container->get('language')->get('LC__MODULE__EVENTS__HISTORY'),
                $request->get('BASE') . 'events/history',
                $request->get('BASE') . 'images/icons/silk/report.png',
                null,
                null,
                null,
                AuthEvents::instance()->is_allowed_to(Auth::VIEW, 'history')
            ));
    }

    /**
     * @param  Register    $request
     * @param  Application $application
     *
     * @return Event|EventList|\idoit\View\Renderable
     * @throws \isys_exception_database
     */
    public function onDefault(Register $request, Application $application)
    {
        if ($request->get('id') > 0) {
            ComponentNavbar::getInstance()->set_active(false, C__NAVBAR_BUTTON__SAVE);

            return $this->onEdit($request, $application);
        }

        return new EventList($request);
    }

    /**
     * @param  Register    $request
     * @param  Application $application
     *
     * @return \idoit\View\Renderable
     * @throws \isys_exception_database
     */
    public function onDuplicate(Register $request, Application $application)
    {
        return $this->onDefault($request, $application);
    }

    /**
     * @param  Register    $request
     * @param  Application $application
     *
     * @return \idoit\View\Renderable
     * @throws \isys_exception_database
     */
    public function onCancel(Register $request, Application $application)
    {
        return $this->onDefault($request, $application);
    }

    /**
     * @param  Register    $request
     * @param  Application $application
     *
     * @return Event|\idoit\View\Renderable
     */
    public function onNew(Register $request, Application $application)
    {
        ComponentNavbar::getInstance()
            ->set_active(true, C__NAVBAR_BUTTON__SAVE)
            ->set_active(true, C__NAVBAR_BUTTON__CANCEL);

        return new Event($request);
    }

    /**
     * @param  Register    $request
     * @param  Application $application
     *
     * @return Event|\idoit\View\Renderable
     * @throws \isys_exception_auth
     * @throws \isys_exception_dao
     */
    public function onSave(Register $request, Application $application)
    {
        // Check for edit right
        AuthEvents::instance()->hooks(Auth::EDIT);

        $id = null;
        $post = $request->get('POST');

        if (isset($post['eventSubscriptionID']) && $post['eventSubscriptionID'] > 0) {
            $id = (int) $post['eventSubscriptionID'];
        }

        $id = Subscription::instance($this->getDi()->get('database'))
            ->saveSubscription($id, [
                'eventId'         => $post['event_id'],                // Deprecated for now.
                'title'           => $post['title'],
                'queued'          => 0,                                // Deprecated for now.
                'type'            => ModuleEvents::TYPE_SHELL_COMMAND, // Deprecated for now.
                'command'         => $post['command'],
                'options'         => $post['parameters'],
                'source'          => $post['source'],
                'action'          => $post['action'],
                'context'         => $post['context'],
                'reference'       => $post['reference'],
                'referenceOption' => $post['reference_option']
            ]);

        // Return the view.
        return (new Event($request))->setID($id);
    }

    /**
     * @param  Register    $request
     * @param  Application $application
     *
     * @return Event|\idoit\View\Renderable
     * @throws \Exception
     */
    public function onEdit(Register $request, Application $application)
    {
        if (isset($_POST['id']) && !$request->get('id')) {
            if (is_array($_POST['id'])) {
                $id = reset($_POST['id']);
            } else {
                $id = $_POST['id'];
            }

            // Always work with GET requests.
            \isys_core::send_header('location', $application->www_path . 'events/main/' . (int)$id);
            die;
        }

        // Return the view.
        $view = new Event($request);

        // Retrieve id from URL "events/main/123" (default way).
        if ($request->get('id')) {
            $view->setID($request->get('id'));
        }

        return $view;
    }

    /**
     * @param  Register    $request
     * @param  Application $application
     *
     * @return Event|EventList|\idoit\View\Renderable
     * @throws \isys_exception_database
     */
    public function onPrint(Register $request, Application $application)
    {
        return $this->onDefault($request, $application);
    }

    /**
     * @param  Register    $request
     * @param  Application $application
     *
     * @return \idoit\View\Renderable
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     */
    public function onPurge(Register $request, Application $application)
    {
        return $this->onDelete($request, $application);
    }

    /**
     * @param  Register    $request
     * @param  Application $application
     *
     * @return \idoit\View\Renderable
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     */
    public function onArchive(Register $request, Application $application)
    {
        return $this->onDelete($request, $application);
    }

    /**
     * @param  Register    $request
     * @param  Application $application
     *
     * @return Event|EventList|\idoit\View\Renderable
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     */
    public function onDelete(Register $request, Application $application)
    {
        $post = $request->get('POST');

        if (isset($post['id']) && is_array($post['id'])) {
            Subscription::instance($this->getDi()->get('database'))->deleteSubscription($post['id']);
        }

        return $this->onDefault($request, $application);
    }

    /**
     * @param  Register    $request
     * @param  Application $application
     *
     * @return \idoit\View\Renderable
     * @throws \isys_exception_database
     */
    public function onQuickPurge(Register $request, Application $application)
    {
        return $this->onDefault($request, $application);
    }

    /**
     * @param  Register    $request
     * @param  Application $application
     *
     * @return \idoit\View\Renderable
     * @throws \isys_exception_database
     */
    public function onRecycle(Register $request, Application $application)
    {
        return $this->onDefault($request, $application);
    }

    /**
     * @param  Register    $request
     * @param  Application $application
     *
     * @return Event|EventList|\idoit\View\Renderable
     * @throws \isys_exception_database
     */
    public function onReset(Register $request, Application $application)
    {
        return $this->onDefault($request, $application);
    }

    /**
     * @param  Register    $request
     * @param  Application $application
     *
     * @return Event|EventList|\idoit\View\Renderable
     * @throws \isys_exception_database
     */
    public function onUp(Register $request, Application $application)
    {
        return $this->onDefault($request, $application);
    }

    /**
     * @param Module $p_module
     */
    public function __construct(Module $p_module)
    {
        $this->module = $p_module;
    }
}
