<?php

namespace idoit\Module\Events\Model;

use idoit\Model\Dao\Base;
use isys_application as Application;
use isys_cmdb_dao as CmdbDao;
use isys_format_json as JSON;
use isys_module_events as ModuleEvents;

/**
 * Class Subscription
 *
 * @package   idoit\Module\Events\Model
 * @author    Leonard Fischer <lfischer@i-doit.com>
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Subscription extends Base
{
    const FIELDS = [
        'isys_event_subscription__id'                => 'id',
        'isys_event_subscription__isys_event__id'    => 'eventId', // Deprecated.
        'isys_event_subscription__isys_obj__id'      => 'objectId',
        'isys_event_subscription__title'             => 'title',
        'isys_event_subscription__queued'            => 'queued',  // Deprecated.
        'isys_event_subscription__type'              => 'type',    // Deprecated.
        'isys_event_subscription__command'           => 'command',
        'isys_event_subscription__options'           => 'options',
        'isys_event_subscription__date_created'      => 'dateCreated',
        'isys_event_subscription__source'            => 'source',
        'isys_event_subscription__action'            => 'action',
        'isys_event_subscription__context'           => 'context',
        'isys_event_subscription__reference'         => 'reference',
        'isys_event_subscription__reference_option'  => 'referenceOption',
    ];

    /**
     * @param int|array $id
     * @param string    $condition
     *
     * @return \isys_component_dao_result
     * @throws \isys_exception_database
     */
    public function getSubscriptions($id = null, $condition = null)
    {
        $sql = 'SELECT ' . $this->selectImplode(self::FIELDS) . ' 
            FROM isys_event_subscription 
            LEFT JOIN isys_obj ON isys_event_subscription__isys_obj__id = isys_obj__id
            WHERE TRUE ';

        if ($id !== null) {
            if (!is_array($id)) {
                $id = [$id];
            }

            $sql .= ' AND isys_event_subscription__id ' . $this->prepare_in_condition($id);
        }

        return $this->retrieve($sql . ' ' . $condition . ';');
    }


    /**
     * Save an event subscription.
     *
     * @param int   $id
     * @param array $data
     *
     * @return int
     * @throws \isys_exception_dao
     */
    public function saveSubscription($id, $data)
    {
        $values = [];

        // Merge given data with default values.
        $data = array_merge($data, [
            'objectId'    => Application::instance()->container->get('session')->get_user_id(),
            'queued'      => 0,
            'type'        => ModuleEvents::TYPE_SHELL_COMMAND,
            'dateCreated' => 'NOW()',
        ]);

        if (isset($data['objectId'])) {
            $values[] = 'isys_event_subscription__isys_obj__id = ' . $this->convert_sql_id($data['objectId']);
        }

        if (isset($data['title'])) {
            $values[] = 'isys_event_subscription__title = ' . $this->convert_sql_text($data['title']);
        }

        if (isset($data['queued'])) {
            $values[] = 'isys_event_subscription__queued = ' . $this->convert_sql_boolean($data['queued']);
        }

        if (isset($data['type'])) {
            $values[] = 'isys_event_subscription__type = ' . $this->convert_sql_int($data['type']);
        }

        if (isset($data['command'])) {
            $values[] = 'isys_event_subscription__command = ' . $this->convert_sql_text($data['command']);
        }

        if (isset($data['options'])) {
            $values[] = 'isys_event_subscription__options = ' . $this->convert_sql_text($data['options']);
        }

        if (isset($data['dateCreated'])) {
            $values[] = 'isys_event_subscription__date_created = ' . $this->convert_sql_datetime($data['dateCreated']);
        }

        // @see  EVENTS-2  New fields since Events 1.1.
        if (isset($data['source'])) {
            $values[] = 'isys_event_subscription__source = ' . $this->convert_sql_text(implode(',', (array)$data['source']));
        }

        if (isset($data['action'])) {
            $values[] = 'isys_event_subscription__action = ' . $this->convert_sql_text(implode(',', (array)$data['action']));
        }

        if (isset($data['context'])) {
            $values[] = 'isys_event_subscription__context = ' . $this->convert_sql_text(implode(',', (array)$data['context']));
        }

        if (isset($data['reference'])) {
            $values[] = 'isys_event_subscription__reference = ' . $this->convert_sql_text($data['reference']);
        }

        if (isset($data['referenceOption'])) {
            $values[] = 'isys_event_subscription__reference_option = ' . $this->convert_sql_text(implode(',', (array)$data['referenceOption']));
        }


        if (!count($values)) {
            return 0;
        }

        if ($id === null) {
            $sql = 'INSERT INTO isys_event_subscription SET ' . implode(',', $values) . ';';
        } else {
            $sql = 'UPDATE isys_event_subscription SET ' . implode(',', $values) . ' WHERE isys_event_subscription__id = ' . $this->convert_sql_id($id) . ';';
        }

        $this->update($sql) && $this->apply_update();

        if ($id === null) {
            return (int)CmdbDao::instance($this->m_db)->get_last_id_from_table('isys_event_subscription');
        }

        return (int)$id;
    }

    /**
     * @param int|array $id
     *
     * @return bool
     * @throws \isys_exception_dao
     */
    public function deleteSubscription($id)
    {
        if (!is_array($id)) {
            $id = [$id];
        }

        $sql = 'DELETE FROM isys_event_subscription WHERE isys_event_subscription__id ' . $this->prepare_in_condition($id) . ';';

        return $this->update($sql) && $this->apply_update();
    }
}
