<?php

namespace idoit\Module\Events\View;

use idoit\Model\Dao\Base as DaoBase;
use idoit\Module\Events\Component\EventParameter;
use idoit\Module\Events\Model\ReferenceOption;
use idoit\Module\Events\Model\Subscription;
use idoit\View\Renderable;
use isys_application;
use isys_auth as Auth;
use isys_auth_events as AuthEvents;
use isys_component_template as ComponentTemplate;
use isys_component_template_navbar as ComponentNavbar;
use isys_module as ModuleBase;

/**
 * i-doit cmdb controller
 *
 * @package     i-doit
 * @subpackage  Core
 * @author      Dennis Stücken <dstuecken@synetics.de>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class Event extends EventList
{
    /**
     * @param  ModuleBase        $module
     * @param  ComponentTemplate $template
     * @param  DaoBase           $model
     *
     * @return $this|Renderable
     */
    public function process(ModuleBase $module, ComponentTemplate $template, DaoBase $model)
    {
        $language = isys_application::instance()->container->get('language');
        $database = isys_application::instance()->container->get('database');

        $isAllowedToEdit = AuthEvents::instance()->is_allowed_to(Auth::EDIT, 'hooks');
        $editMode = isys_glob_is_edit_mode();
        $referenceOptionData = [];

        ComponentNavbar::getInstance()
            ->set_active(!$editMode && $isAllowedToEdit, C__NAVBAR_BUTTON__EDIT)
            ->set_visible(!$editMode, C__NAVBAR_BUTTON__EDIT)
            ->set_active($editMode && $isAllowedToEdit, C__NAVBAR_BUTTON__SAVE)
            ->set_visible($editMode, C__NAVBAR_BUTTON__SAVE);

        if ($this->id) {
            $eventData = Subscription::instance($database)
                ->getSubscriptions($this->id)
                ->get_row();
        } else {
            // Defaults
            $eventData = [
                'title'           => '',
                'command'         => '',
                'options'         => '',
                'source'          => '',
                'action'          => '',
                'context'         => '',
                'reference'       => 'object_type',
                'referenceOption' => ''
            ];
        }

        if ($eventData['reference'] === EventParameter::REFERENCE_CATEGORY) {
            $referenceOptionData = ReferenceOption::instance($database)->getCategories();

            if (!isys_glob_is_edit_mode()) {
                // We can NOT use "array_merge(...$referenceOptionData);" because of the string keys.
                $referenceOptionData = call_user_func_array('array_merge', $referenceOptionData);
            }
        } elseif ($eventData['reference'] === EventParameter::REFERENCE_OBJECT) {
            $referenceOptionData = ReferenceOption::instance($database)->getObjectTypes();
        }

        $l_rules = [
            'title'              => [
                'p_strValue' => $eventData['title']
            ],
            'source[]'           => [
                'p_arData'        => EventParameter::getSources(),
                'p_strClass'      => 'input-small',
                'p_bDbFieldNN'    => true,
                'p_multiple'      => true,
                'chosen'          => true,
                'p_strSelectedID' => $eventData['source']
            ],
            'action[]'           => [
                'p_arData'        => EventParameter::getActions(),
                'p_strClass'      => 'input-small',
                'p_bDbFieldNN'    => true,
                'p_multiple'      => true,
                'chosen'          => true,
                'p_strSelectedID' => $eventData['action']
            ],
            'context[]'          => [
                'p_arData'          => EventParameter::getContexts(),
                'p_bDbFieldNN'      => true,
                'p_multiple'        => true,
                'chosen'            => true,
                'p_strSelectedID'   => $eventData['context'],
                'disableInputGroup' => true,
                'p_bInfoIconSpacer' => 0
            ],
            'reference'          => [
                'p_arData'        => EventParameter::getReferences(),
                'p_strClass'      => 'input-small',
                'p_bDbFieldNN'    => true,
                'p_strSelectedID' => $eventData['reference']
            ],
            'reference_option[]' => [
                'p_arData'        => $referenceOptionData,
                'p_bDbFieldNN'    => true,
                'p_multiple'      => true,
                'chosen'          => true,
                'p_strSelectedID' => $eventData['referenceOption']
            ],
            'command'            => [
                'p_strValue' => $eventData['command']
            ],
            'parameters'         => [
                'p_strValue' => $eventData['options']
            ]
        ];

        // Assign template variables.
        $template
            ->assign('baseUrl', isys_application::instance()->www_path)
            ->assign('eventSubscriptionID', $this->id)
            ->smarty_tom_add_rules('tom.content.bottom.content', $l_rules)
            ->smarty_tom_add_rule('tom.content.bottom.buttons.*.p_bInvisible=1')
            ->smarty_tom_add_rule('tom.content.navbar.cRecStatus.p_bInvisible=1');

        // Set paths to templates.
        $this->paths['contentbottomcontent'] = $module::getPath() . 'templates/edit.tpl';

        return $this;
    }
}
