<?php

use idoit\AddOn\ActivatableInterface;
use idoit\AddOn\AuthableInterface;
use idoit\AddOn\InstallableInterface;
use idoit\Module\Events\Component\EventParameter;
use idoit\Module\Events\Handler\Get;
use idoit\Module\Events\Handler\Post;
use idoit\Module\Events\Handler\Shell;
use idoit\Module\Events\Model\Log;

/**
 * i-doit
 *
 * Events module class
 *
 * @package     modules
 * @subpackage  events
 * @author      Dennis Stücken <dstuecken@synetics.de>
 * @version     1.5
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.5
 */
class isys_module_events extends isys_module implements AuthableInterface, InstallableInterface, ActivatableInterface
{
    // Define, if this module shall be displayed in the named menus.
    const DISPLAY_IN_MAIN_MENU   = false;
    const DISPLAY_IN_SYSTEM_MENU = true;
    const MAIN_MENU_REWRITE_LINK = true;

    // Event type constants.
    const TYPE_SHELL_COMMAND     = 1;
    const TYPE_HTTP_GET          = 2;
    const TYPE_HTTP_POST         = 3;

    /**
     * @var bool
     */
    protected static $m_licenced = true;

    /**
     * Variable which the module request class.
     *
     * @var  isys_module_request
     */
    protected $m_modreq;

    /**
     * Return event types
     *
     * @return array
     */
    public static function event_types()
    {
        return [
            self::TYPE_SHELL_COMMAND => 'SHELL COMMAND',
            self::TYPE_HTTP_GET      => 'HTTP GET',
            self::TYPE_HTTP_POST     => 'HTTP POST',
        ];
    }

    /**
     * @param $eventRow
     * @param $args
     */
    public static function delegate($eventRow, $args)
    {
        $log = new Log(isys_application::instance()->container->get('database'));

        // @see  EVENTS-16  Add information about the current user to the event.
        $session = isys_application::instance()->container->get('session');

        if ($session) {
            $args['_user'] = [
                'id'       => $session->get_user_id(),
                'username' => $session->get_current_username()
            ];
        }

        try {
            switch ($eventRow['type']) {
                case self::TYPE_SHELL_COMMAND:
                    $eventHandler = new Shell();
                    break;
                case self::TYPE_HTTP_GET:
                    $eventHandler = new Get();
                    break;
                default:
                case self::TYPE_HTTP_POST:
                    $eventHandler = new Post();
                    break;
            }

            if ($eventRow['queued']) {
                $response = $eventHandler->handleQueued($eventRow, $args);
            } else {
                $response = $eventHandler->handleLive($eventRow, $args);
            }

            $log->log($eventRow['id'], $eventRow['title'], $response->output, $response->success ? 1 : 0, $eventRow['command'], $response->returnCode);
        } catch (Exception $e) {
            $log->log($eventRow['id'], $eventRow['title'], $e->getMessage(), 0, $eventRow['command']);
        }
    }

    /**
     * Initializes the module.
     *
     * @param   isys_module_request & $p_req
     *
     * @return  $this
     */
    public function init(isys_module_request $p_req)
    {
        $this->m_modreq = $p_req;

        return $this;
    }

    /**
     * Build breadcrumb navifation
     *
     * @param &$p_gets
     *
     * @return array
     */
    public function breadcrumb_get(&$p_gets)
    {
        return [
            [
                isys_application::instance()->container->get('language')->get('LC__CONFIGURATION') => [
                    C__GET__MODULE_ID => C__MODULE__EVENTS
                ]
            ]
        ];
    }

    /**
     * This method builds the tree for the system menu.
     *
     * @param   isys_component_tree $tree
     * @param   boolean             $systemContext
     * @param   integer             $parentNodeId
     *
     * @see     isys_module_cmdb->build_tree();
     */
    public function build_tree(isys_component_tree $tree, $systemContext = true, $parentNodeId = null)
    {
        global $g_dirs;

        // Check if parent node is delivered.
        if ($parentNodeId !== null && is_int($parentNodeId) && $systemContext) {
            $l_mod_gets[C__GET__MODULE_SUB_ID] = C__MODULE__EVENTS;

            // Check for PRO module
            if (defined('C__MODULE__PRO')) {
                global $g_config;

                // Add node
                $tree->add_node(
                    C__MODULE__EVENTS . '01338',
                    $parentNodeId,
                    isys_application::instance()->container->get('language')->get('LC__MODULE__EVENTS'),
                    rtrim(isys_application::instance()->www_path, '/') . '/events',
                    null,
                    $g_dirs['images'] . '/axialis/basic/light-on.svg'
                );
            }
        }
    }

    /**
     * Checks if a add-on is installed.
     *
     * @return int|bool
     */
    public static function isInstalled()
    {
        return isys_module_manager::instance()->is_installed('events');
    }

    /**
     * Basic installation process for all mandators.
     *
     * @param  \isys_component_database $tenantDatabase
     * @param  \isys_component_database $systemDatabase
     * @param  integer                  $moduleId
     * @param  string                   $type
     * @param  integer                  $tenantId
     *
     * @since  i-doit 1.12
     * @return boolean
     */
    public static function install($tenantDatabase, $systemDatabase, $moduleId, $type, $tenantId)
    {
        if ($type === 'update') {
            // Check if the migration has already been done.
            $dao = isys_cmdb_dao::instance($tenantDatabase);

            $countSql = 'SELECT COUNT(isys_event_subscription__id) AS cnt FROM isys_event_subscription WHERE isys_event_subscription__source IS NULL;';

            if ($dao->retrieve($countSql)->get_row_value('cnt')) {
                $mapping = [
                    'mod.cmdb.objectCreated'            => [
                        'action'    => EventParameter::ACTION_NEW,
                        'reference' => EventParameter::REFERENCE_OBJECT,
                    ],
                    'mod.cmdb.objectDeleted'            => [
                        'action'    => EventParameter::ACTION_PURGE,
                        'reference' => EventParameter::REFERENCE_OBJECT,
                    ],
                    'mod.cmdb.afterCreateCategoryEntry' => [
                        'action'    => EventParameter::ACTION_NEW,
                        'reference' => EventParameter::REFERENCE_CATEGORY,
                    ],
                    'mod.cmdb.afterCategoryEntrySave'   => [
                        'action'    => EventParameter::ACTION_SAVE,
                        'reference' => EventParameter::REFERENCE_CATEGORY,
                    ],
                    'mod.cmdb.beforeRankRecord'         => [
                        'action'    => EventParameter::ACTION_RANK . ',' . EventParameter::ACTION_PURGE,
                        'reference' => EventParameter::REFERENCE_CATEGORY,
                    ],
                    'mod.cmdb.afterObjectTypeSave'      => [
                        'action'    => EventParameter::ACTION_NEW . ',' . EventParameter::ACTION_SAVE,
                        'reference' => EventParameter::REFERENCE_OBJECT_TYPE
                    ],
                    'mod.cmdb.afterObjectTypePurge'     => [
                        'action'    => EventParameter::ACTION_PURGE,
                        'reference' => EventParameter::REFERENCE_OBJECT_TYPE
                    ],
                ];

                $source = $dao->convert_sql_text(EventParameter::SOURCE_GUI . ',' . EventParameter::SOURCE_API . ',' . EventParameter::SOURCE_CONSOLE);

                foreach ($mapping as $event => $map) {
                    $updateSql = 'UPDATE isys_event_subscription 
                        INNER JOIN isys_event ON isys_event__id = isys_event_subscription__isys_event__id
                        SET 
                        isys_event_subscription__source = ' . $source . ',
                        isys_event_subscription__action = ' . $dao->convert_sql_text($map['action']) . ',
                        isys_event_subscription__context = NULL,
                        isys_event_subscription__reference = ' . $dao->convert_sql_text($map['reference']) . ',
                        isys_event_subscription__reference_option = NULL
                        WHERE isys_event__identifier = ' . $dao->convert_sql_text($event) . ';';

                    $dao->update($updateSql) && $dao->apply_update();
                }
            }
        }

        return true;
    }

    /**
     * Uninstall add-on for all mandators.
     *
     * @param \isys_component_database $tenantDatabase
     *
     * @return boolean
     */
    public static function uninstall($tenantDatabase)
    {
        return true;
    }

    /**
     * Checks if a add-on is active.
     *
     * @return integer|bool
     */
    public static function isActive()
    {
        return isys_module_manager::instance()->is_installed('events', true);
    }

    /**
     * Method that is called after clicking "activate" in admin center for specific mandator.
     *
     * @param \isys_component_database $tenantDatabase
     *
     * @return boolean
     */
    public static function activate($tenantDatabase)
    {
        return true;
    }

    /**
     * Method that is called after clicking "deactivate" in admin center for specific mandator.
     *
     * @param \isys_component_database $tenantDatabase
     *
     * @return boolean
     */
    public static function deactivate($tenantDatabase)
    {
        return true;
    }

    /**
     * @return isys_auth_events
     */
    public static function getAuth()
    {
        return isys_auth_events::instance();
    }
}
