<?php

namespace idoit\Module\Events\Component;

use idoit\Context\Context;
use idoit\Module\Events\Handler\Get;
use idoit\Module\Events\Handler\Post;
use idoit\Module\Events\Handler\Shell;
use idoit\Module\Events\Model\Log;
use idoit\Module\Events\Model\Subscription;
use isys_application as Application;
use isys_cmdb_dao;
use isys_cmdb_dao_category;
use isys_component_database as ComponentDatabase;
use isys_component_session as ComponentSession;
use isys_component_template_language_manager as ComponentLanguage;

/**
 * Class EventParameter
 *
 * @package   idoit\Module\Events\Component
 * @author    Leonard Fischer <lfischer@i-doit.com>
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Delegator
{
    /**
     * @var ComponentDatabase
     */
    private $database;

    /**
     * @var ComponentLanguage
     */
    private $language;

    /**
     * @var ComponentSession
     */
    private $session;

    /**
     * @var Context
     */
    private $context;

    /**
     * @var string
     */
    private $action;

    /**
     * @var string
     */
    private $eventContext;

    /**
     * @var string
     */
    private $reference;

    /**
     * @var string
     */
    private $referenceOption;

    /**
     * Delegator constructor.
     *
     * @param ComponentDatabase $database
     * @param ComponentLanguage $language
     * @param ComponentSession $session
     *
     * @throws \isys_exception_database
     */
    public function __construct(ComponentDatabase $database, ComponentLanguage $language, ComponentSession $session)
    {
        $this->database = $database;
        $this->language = $language;
        $this->session = $session;
        $this->context = Context::instance();
    }

    /**
     * @param string $eventHandler
     * @param array  $args
     *
     * @throws \isys_exception_dao
     */
    private function trigger($eventHandler, $args)
    {
        $log = new Log($this->database);

        // @see  EVENTS-16  Add information about the current user to the event.
        $args['_user'] = [
            'id'       => $this->session->get_user_id(),
            'username' => $this->session->get_current_username()
        ];

        foreach ($this->getHooks() as $hook) {
            try {
                switch ($hook['type']) {
                    case \isys_module_events::TYPE_SHELL_COMMAND:
                        $handler = new Shell();
                        break;
                    case \isys_module_events::TYPE_HTTP_GET:
                        $handler = new Get();
                        break;
                    default:
                    case \isys_module_events::TYPE_HTTP_POST:
                        $handler = new Post();
                        break;
                }

                if ($hook['queued']) {
                    $response = $handler->handleQueued($hook, $args);
                } else {
                    $response = $handler->handleLive($hook, $args);
                }

                $log->log($hook['id'], $hook['title'], $response->output, $response->success ? 1 : 0, $hook['command'], $response->returnCode);
            } catch (\Exception $e) {
                $log->log($hook['id'], $hook['title'], $e->getMessage(), 0, $hook['command']);
            }
        }
    }

    /**
     * This method will be used to set the current event context (manual, template or import).
     */
    private function setCurrentEventContext()
    {
        $eventContext = EventParameter::CONTEXT_MANUAL;

        if ($this->context->getGroup() === Context::CONTEXT_GROUP_IMPORT) {
            $eventContext = EventParameter::CONTEXT_IMPORT;
        }

        if (in_array($this->context->getContextCustomer(), [Context::CONTEXT_MASS_CHANGE, Context::CONTEXT_TEMPLATE], true)) {
            $eventContext = EventParameter::CONTEXT_TEMPLATE;
        }

        if (in_array($this->context->getContextCustomer(), [Context::CONTEXT_DUPLICATE, Context::CONTEXT_MULTIEDIT], true)) {
            $eventContext = EventParameter::CONTEXT_MANUAL;
        }

        $this->eventContext = $eventContext;
    }

    /**
     * Retrieve the affected hooks.
     *
     * @return array
     */
    private function getHooks()
    {
        $hooks = [];

        $this->setCurrentEventContext();

        $model = Subscription::instance($this->database);
        $condition = ' AND isys_event_subscription__source LIKE ' . $model->convert_sql_text('%' . $this->context->getOrigin() . '%') . '
            AND isys_event_subscription__action LIKE ' . $model->convert_sql_text('%' . $this->action . '%') . ' 
            AND isys_event_subscription__reference = ' . $model->convert_sql_text($this->reference) . '
            AND (isys_event_subscription__context LIKE ' . $model->convert_sql_text('%' . $this->eventContext . '%') . ' OR isys_event_subscription__context IS NULL )';

        if ($this->referenceOption !== null) {
            $condition .= 'AND (isys_event_subscription__reference_option LIKE ' . $model->convert_sql_text('%' . $this->referenceOption . '%') . ' OR isys_event_subscription__reference_option IS NULL)';
        }

        // Only load events, that listen to the current origin.
        $result = $model->getSubscriptions(null, $condition);

        while ($row = $result->get_row()) {
            $row['source'] = explode(',', $row['source']);
            $row['action'] = explode(',', $row['action']);
            $row['referenceOption'] = explode(',', $row['referenceOption']);

            $hooks[] = $row;
        }

        return $hooks;
    }

    /**
     * @param int    $objectId
     * @param int    $sysId
     * @param int    $objectTypeId
     * @param string $objectTitle
     * @param int    $cmdbStatus
     * @param string $username
     *
     * @throws \isys_exception_database
     */
    public function onObjectCreated($objectId, $sysId, $objectTypeId, $objectTitle, $cmdbStatus, $username)
    {
        $dao = isys_cmdb_dao::instance($this->database);
        $sql = 'SELECT isys_obj_type__title AS title, isys_obj_type__const AS const 
            FROM isys_obj_type
            INNER JOIN isys_obj ON isys_obj__isys_obj_type__id = isys_obj_type__id
            WHERE isys_obj__id = ' . $dao->convert_sql_id($objectId) . ' LIMIT 1;';
        $objectType = $dao->retrieve($sql)->get_row();

        $this->action = EventParameter::ACTION_NEW;
        $this->reference = EventParameter::REFERENCE_OBJECT;
        $this->referenceOption = $objectType['const'];

        $this->trigger(__FUNCTION__, [
            'id'              => $objectId,
            'title'           => $objectTitle,
            'cmdbStatusID'    => $cmdbStatus,
            'cmdbStatus'      => $this->language->get($dao->get_object($objectId)->get_row_value('isys_cmdb_status__title')),
            'objectTypeID'    => $objectTypeId,
            'objectTypeConst' => $objectType['const'],
            'objectType'      => $this->language->get($objectType['title']),
            'sysID'           => $sysId,
            'username'        => $username
        ]);
    }

    /**
     * @param int $objectId
     *
     * @throws \isys_exception_database
     */
    public function onObjectDeleted($objectId)
    {
        $cmdbDao = isys_cmdb_dao::factory($this->database);

        $sql = 'SELECT isys_obj__title AS objTitle, isys_obj_type__title AS objTypeTitle, isys_obj_type__const AS const 
            FROM isys_obj_type
            INNER JOIN isys_obj ON isys_obj__isys_obj_type__id = isys_obj_type__id
            WHERE isys_obj__id = ' . $cmdbDao->convert_sql_id($objectId) . ' LIMIT 1;';

        $objectData = isys_cmdb_dao::factory($this->database)->retrieve($sql)->get_row();

        $this->action = EventParameter::ACTION_PURGE;
        $this->reference = EventParameter::REFERENCE_OBJECT;
        $this->referenceOption = $objectData['const'];

        $this->trigger(__FUNCTION__, [
            'id'    => $objectId,
            'title' => $objectData['objTitle'],
            'type'  => $this->language->get($objectData['objTypeTitle'])
        ]);
    }

    /**
     * @param  isys_cmdb_dao $dao
     * @param  integer       $objectId
     * @param  integer       $categoryEntryId
     * @param  string        $title
     * @param  array         $row
     * @param  string        $table
     * @param  integer       $currentStatus
     * @param  integer       $newStatus
     * @param  integer       $categoryType
     * @param  integer       $direction
     *
     * @throws \Exception
     */
    public function onBeforeRankRecord(\isys_cmdb_dao $dao, $objectId, $categoryEntryId, $title, $row, $table, $currentStatus, $newStatus, $categoryType, $direction)
    {
        $data = [];
        $sourceTable = 'isys_obj';

        if ($categoryEntryId !== null && $categoryEntryId > 0) {
            $data = [];

            if (is_a($dao, 'isys_cmdb_dao_category')) {
                if (!$row || !is_array($row)) {
                    /**
                     * @var isys_cmdb_dao_category $dao
                     */
                    $row = $dao->get_data_by_id($categoryEntryId)
                        ->get_row();
                }

                $sourceTable = $dao->get_source_table();
            }

        }

        $categoryConstant = method_exists($dao, 'get_category_const') ? $dao->get_category_const() : null;

        if ($dao instanceof \isys_cmdb_dao_category_g_custom_fields) {
            $categoryConstantInfo = $dao->get_category_info($dao->get_catg_custom_id());
            $categoryConstant = $categoryConstantInfo['isysgui_catg_custom__const'];

            if (is_array($row)) {
                foreach ($row as $key => $customFieldRow) {
                    if (isset($customFieldRow['isys_catg_custom_fields_list__field_key'], $customFieldRow['isys_catg_custom_fields_list__field_content']) && is_scalar($customFieldRow['isys_catg_custom_fields_list__field_key'])) {
                        $data[$customFieldRow['isys_catg_custom_fields_list__field_key']] = $this->language->get($customFieldRow['isys_catg_custom_fields_list__field_content']);
                    }
                }
            }
        } elseif (is_array($row)) {
            foreach ($row as $key => $value) {
                if ($key === 'isys_obj__title') {
                    $newKey = 'object';
                } elseif ($key === 'isys_obj_type__title') {
                    $newKey = 'objectType';
                    $value = $this->language->get($value);
                } elseif ($key === 'isys_obj_type__const') {
                    $newKey = 'objectTypeConst';
                } else {
                    $newKey = str_replace('_', '', str_replace('_list', '', str_replace($sourceTable, '', $key)));
                }

                if (!strstr($newKey, '__') && !strstr($newKey, 'isys')) {
                    $data[$newKey] = $value;
                }
            }
        }

        $this->action = EventParameter::ACTION_RANK;
        $this->referenceOption = null;

        if ($this->context->getContextTechnical() === Context::CONTEXT_RANK_CATEGORY) {
            $this->reference = EventParameter::REFERENCE_CATEGORY;
            $this->referenceOption = $categoryConstant;
        } else {
            $this->reference = EventParameter::REFERENCE_OBJECT;

            if (isset($row['isys_obj__isys_obj_type__id']) && $row['isys_obj__isys_obj_type__id'] > 0) {
                $this->referenceOption = $dao->get_objtype($row['isys_obj__isys_obj_type__id'])->get_row_value('isys_obj_type__const');
            }
        }

        $this->trigger(__FUNCTION__, [
            'title'              => $title,
            C__CMDB__GET__OBJECT => $objectId,
            'categoryID'         => method_exists($dao, 'get_category_id') ? $dao->get_category_id() : null,
            'categoryDataID'     => $categoryEntryId,
            'categoryConst'      => $categoryConstant,
            'currentStatus'      => $currentStatus,
            'newStatus'          => $newStatus,
            'data'               => $data,
            'direction'          => $direction
        ]);
    }

    /**
     * @param isys_cmdb_dao_category $dao
     * @param integer                $categoryId
     * @param                        $saveSuccess
     * @param integer                $objectId
     * @param array                  $posts
     * @param array                  $changes
     *
     * @throws \isys_exception_database
     */
    public function onAfterCategoryEntrySave(isys_cmdb_dao_category $dao, $categoryId, $saveSuccess, $objectId, $posts, $changes)
    {
        if (!$categoryId && !$dao->is_multivalued()) {
            $sourceTable = substr($dao->get_source_table(), -5) === '_list' ?
                $dao->get_source_table() :
                $dao->get_source_table() . '_list';

            $categoryId = $dao->get_data_by_object($objectId)->get_row_value($sourceTable . '__id');
        }

        $categoryConstant = method_exists($dao, 'get_category_const') ? $dao->get_category_const() : null;
        $categoryType = $dao->get_category_type_const();

        // @see EVENTS-9 With custom categories we need to retrieve the constant a bit differently.
        if ($dao instanceof \isys_cmdb_dao_category_g_custom_fields) {
            $categoryConstantInfo = $dao->get_category_info($dao->get_catg_custom_id());
            $categoryConstant = $categoryConstantInfo['isysgui_catg_custom__const'];

            // Set category type statically to 'custom'
            $categoryType = 'C__CMDB__CATEGORY__TYPE_CUSTOM';
        }

        $dataPath = $categoryConstant ?: 'C__CATG__GLOBAL';

        $this->action = EventParameter::ACTION_SAVE;
        $this->reference = EventParameter::REFERENCE_CATEGORY;
        $this->referenceOption = $categoryConstant;

        $this->trigger(__FUNCTION__, [
            'success'        => (int) ($saveSuccess === null),
            'objectID'       => $objectId,
            'categoryID'     => $dao->get_category_id(),
            'categoryConst'  => $categoryConstant,
            'categoryType'   => $categoryType,
            'categoryDataID' => $categoryId,
            'multivalue'     => $dao->is_multivalued(),
            'changes'        => $changes,
            'postData'       => $posts,
            'data'           => \isys_cmdb_dao_category_data::initialize($objectId)
                ->path($dataPath)
                ->data()
                ->toArray()
        ]);
    }

    /**
     * @param  integer                $categoryID
     * @param  integer                $categoryEntryId
     * @param  boolean                $result
     * @param  integer                $objectId
     * @param  isys_cmdb_dao_category $dao
     *
     * @throws \isys_exception_database
     */
    public function onAfterCategoryEntryCreate($categoryID, $categoryEntryId, $result, $objectId, isys_cmdb_dao_category $dao)
    {
        $categoryConstant = method_exists($dao, 'get_category_const') ? $dao->get_category_const() : null;
        $categoryType = $dao->get_category_type_const();

        // @see EVENTS-9 With custom categories we need to retrieve the constant a bit differently.
        if ($dao instanceof \isys_cmdb_dao_category_g_custom_fields) {
            $categoryConstantInfo = $dao->get_category_info($dao->get_catg_custom_id());
            $categoryConstant = $categoryConstantInfo['isysgui_catg_custom__const'];

            // Set category type statically to 'custom'
            $categoryType = 'C__CMDB__CATEGORY__TYPE_CUSTOM';
        }

        $this->action = EventParameter::ACTION_NEW;
        $this->reference = EventParameter::REFERENCE_CATEGORY;
        $this->referenceOption = $categoryConstant;

        $this->trigger(__FUNCTION__, [
            'success'        => (int) ($result === null),
            'objectID'       => $objectId,
            'categoryID'     => $dao->get_category_id(),
            'categoryDataID' => $categoryEntryId,
            'categoryConst'  => $categoryConstant,
            'categoryType'   => $categoryType,
            'multivalue'     => $dao->is_multivalued()
        ]);
    }

    /**
     * @param  integer $typeId
     * @param  array   $posts
     * @param          $success
     *
     * @throws \isys_exception_database
     */
    public function onAfterObjectTypeSave($typeId, $posts, $success)
    {
        $objectTypeData = isys_cmdb_dao::instance($this->database)
            ->get_objtype($typeId)
            ->get_row();

        $this->action = EventParameter::ACTION_SAVE;
        $this->reference = EventParameter::REFERENCE_OBJECT_TYPE;
        $this->referenceOption = $objectTypeData['isys_obj_type__const'];

        $this->trigger(__FUNCTION__, [
            'success'      => $success,
            'typeID'       => $typeId,
            'postData'     => $posts,
            'title'        => $this->language->get($objectTypeData['isys_obj_type__title']),
            'description'  => $objectTypeData['isys_obj_type__description'],
            'const'        => $objectTypeData['isys_obj_type__const'],
            'status'       => $objectTypeData['isys_obj_type__status'],
            'visible'      => $objectTypeData['isys_obj_type__show_in_tree'],
            'locationType' => $objectTypeData['isys_obj_type__show_in_rack'],
            'color'        => $objectTypeData['isys_obj_type__color'],
            'sysidPrefix'  => $objectTypeData['isys_obj_type__sysid_prefix']
        ]);
    }

    /**
     * @param  $typeId
     * @param  $title
     * @param  $success
     * @param  $data
     *
     * @throws \Exception
     */
    public function onAfterObjectTypePurge($typeId, $title, $success, $data)
    {
        $this->action = EventParameter::ACTION_PURGE;
        $this->reference = EventParameter::REFERENCE_OBJECT_TYPE;
        $this->referenceOption = $data['isys_obj_type__const'];

        $this->trigger(__FUNCTION__, [
            'success'      => $success,
            'typeID'       => $typeId,
            'title'        => $this->language->get($title),
            'description'  => $data['isys_obj_type__description'],
            'const'        => $data['isys_obj_type__const'],
            'status'       => $data['isys_obj_type__status'],
            'visible'      => $data['isys_obj_type__show_in_tree'],
            'locationType' => $data['isys_obj_type__show_in_rack'],
            'color'        => $data['isys_obj_type__color'],
            'sysidPrefix'  => $data['isys_obj_type__sysid_prefix']
        ]);
    }
}
