/**
 * i-doit floorplan location tree class.
 *
 * @author  Leonard Fischer <lfischer@i-doit.com>
 */
window.FloorplanTree = Class.create(window.BaseTree, {
    positionedObjects: [],
    counterCache:      null,
    loading:           {},

    /**
     * Constructor method.
     *
     * @param $container
     * @param options
     * @returns {Window.FloorplanTree}
     */
    initialize: function ($super, $container, options) {
        var tmpCache     = this.cache,
            tmpOpenNodes = this.openNodes,
            tmpOptions   = {
                rootNodeId:          1,
                activeRoomplanId:    1,
                mode:                null,
                editMode:            false,
                onSelect:            Prototype.emptyFunction,
                onOpenFloorplan:     Prototype.emptyFunction,
                onAdd:               Prototype.emptyFunction,
                onAddAllChildren:    Prototype.emptyFunction,
                onRemove:            Prototype.emptyFunction,
                onRemoveAllChildren: Prototype.emptyFunction
            };

        $super($container, Object.extend(tmpOptions, options || {}));

        this.cache = tmpCache;
        this.openNodes = tmpOpenNodes;
        this.counterCache = null;
        this.refreshCounter = true;

        return this;
    },

    /**
     *
     * @param objId
     * @returns {Window.FloorplanTree}
     */
    setActiveFloorplan: function (objId) {
        this.options.activeRoomplanId = parseInt(objId);

        this.refreshCounter = true;

        return this;
    },

    /**
     *
     * @param editMode
     * @returns {Window.FloorplanTree}
     */
    setEditMode: function (editMode) {
        this.options.editMode = !!editMode;

        this.process();

        return this;
    },

    /**
     * Method for adding all necessary observers.
     */
    addObserver: function ($super) {
        var that = this;

        // The default "toggle" observer will be set by parent.
        $super();

        this.$container.on('click', 'span.mouse-pointer', function (ev) {
            that.options.onSelect(ev.findElement('li').readAttribute('data-id'));
        });

        this.$container.on('click', '[data-action]', function (ev) {
            var $button  = ev.findElement('button'),
                $li      = $button.up('li'),
                objectId = parseInt($li.readAttribute('data-id'));

            switch ($button.readAttribute('data-action')) {
                case 'add':
                    that.options.onAdd(objectId, that.cache['n' + objectId]);
                    that.process();

                    break;

                case 'remove':
                    that.options.onRemove(objectId, that.cache['n' + objectId]);
                    that.process();

                    break;

                case 'add-all-children':
                    if (that.cache['n' + objectId].children.length) {
                        that.options.onAddAllChildren(objectId, that.cache['n' + objectId]);
                        that.process();
                    } else {
                        that.loadChildrenNodes(objectId, function (objectId, data) {
                            that.options.onAddAllChildren(objectId, data);
                            that.process();
                        });
                    }

                    break;

                case 'remove-all-children':
                    if (that.cache['n' + objectId].children.length) {
                        that.options.onRemoveAllChildren(objectId, that.cache['n' + objectId]);
                        that.process();
                    } else {
                        that.loadChildrenNodes(objectId, function (objectId, data) {
                            that.options.onRemoveAllChildren(objectId, that.cache['n' + objectId]);
                            that.process();
                        });
                    }

                    break;

                case 'open-floorplan':
                    that.options.onOpenFloorplan(objectId, false);
                    break;

                case 'create-floorplan':
                    that.options.onOpenFloorplan(objectId, true);
                    break;
            }
        });
    },

    /**
     * Method for setting the tree "mode" (will only affect the data-loading).
     *
     * @param mode
     * @returns {Window.FloorplanTree}
     */
    setMode: function (mode) {
        this.options.mode = mode;

        this.cache = {};

        return this;
    },

    /**
     * Method for setting the root object id.
     *
     * @param id
     * @returns {Window.FloorplanTree}
     */
    setRootNodeId: function (id) {
        this.options.rootNodeId = parseInt(id);

        return this;
    },

    /**
     * Method for retrieving all positioned objects.
     *
     * @returns {number[]}
     */
    getPositionedObjects: function () {
        return this.positionedObjects.map(function (d) {
            return parseInt(d.substr(1));
        });
    },

    /**
     * Method for setting the currently located objects.
     *
     * @param objects
     * @returns {Window.FloorplanTree}
     */
    setPositionedObjects: function (objects) {
        this.positionedObjects = objects.map(function (d) {
            return 'n' + d;
        });

        this.refreshCounter = true;

        return this;
    },

    /**
     * Add a single positioned object.
     *
     * @param objectId
     * @returns {Window.FloorplanTree}
     */
    addPositionedObject: function (objectId) {
        this.positionedObjects.push('n' + objectId);

        this.refreshCounter = true;

        return this;
    },

    /**
     * Remove a single positioned object.
     *
     * @param objectId
     * @returns {Window.FloorplanTree}
     */
    removePositionedObject: function (objectId) {
        this.positionedObjects = this.positionedObjects.filter(function (d) {
            return d !== 'n' + objectId;
        });

        this.refreshCounter = true;

        return this;
    },

    /**
     * Method for finding out if the given object is located in the floorplan.
     *
     * @param objectId
     * @returns {boolean}
     */
    isPositionedObject: function (objectId) {
        return this.positionedObjects.indexOf('n' + objectId) !== -1;
    },

    /**
     * Method for rendering a node.
     *
     * @param data
     * @param root
     * @returns {*}
     */
    renderNode: function (data) {
        var $node      = new Element('li', {'data-id': data.objectId}),
            open       = this.isOpenNode(data.objectId),
            positioned = this.isPositionedObject(data.objectId),
            counter    = null,
            $toggleButton, $addAllButton, $removeAllButton, $openFloorplan;

        if (this.counterCache) {
            counter = this.counterCache.hasOwnProperty('n' + data.objectId) ? this.counterCache['n' + data.objectId] : null;
        }

        if (this.options.editMode) {
            $toggleButton = new Element('button', {
                type: 'button',
                className: 'fr btn btn-small ' + (positioned ? 'opacity-100' : ''),
                'data-action': (positioned ? 'remove' : 'add'),
                title: idoit.Translate.get('LC__MODULE__FLOORPLAN__VISUALIZATION__' + (positioned ? 'UN' : '') + 'POSITION')
            })
                .update(new Element('img', { src: window.dir_images + 'axialis/basic/symbol-' + (positioned ? 'remove' : 'add') + '.svg' }));

            if (data.hasChildren) {
                $addAllButton = new Element('button', {
                    type: 'button',
                    className: 'fr btn btn-small',
                    'data-action': 'add-all-children',
                    title: idoit.Translate.get('LC__MODULE__FLOORPLAN__VISUALIZATION__POSITION_ALL')
                })
                    .update(new Element('img', { src: window.floorplanData.wwwPath + 'assets/images/icons/text_list_bullets_add.png' }));

                $removeAllButton = new Element('button', {
                    type: 'button',
                    className: 'fr btn btn-small',
                    'data-action': 'remove-all-children',
                    title: idoit.Translate.get('LC__MODULE__FLOORPLAN__VISUALIZATION__UNPOSITION_ALL')
                })
                    .update(new Element('img', { src: window.floorplanData.wwwPath + 'assets/images/icons/text_list_bullets_delete.png' }));
            }
        } else if (data.isContainer) {
            $openFloorplan = new Element('button', {
                type: 'button',
                className: 'fr btn ' + (data.hasOwnFloorplan ? 'opacity-100' : ''),
                'data-action': (data.hasOwnFloorplan ? 'open-floorplan' : 'create-floorplan'),
                title: idoit.Translate.get('LC__MODULE__FLOORPLAN__' + (data.hasOwnFloorplan ? 'TAB__LOCATION__FUNCTION_OPEN_FLOORPLAN' : 'CATEGORY__CREATE_FLOORPLAN'))
            })
                .update(new Element('img', { src: (data.hasOwnFloorplan ? window.floorplanWwwPath + 'assets/add-on-icon.svg' : window.dir_images + 'axialis/basic/symbol-add.svg') }));
        }

        return $node
            .update(new Element('img', { src: this.getToggleImage(open), className: 'child-toggle ' + (data.hasChildren ? '' : 'hide') }))
            .insert(new Element('div', { className: 'tree-inner' })
                .update(new Element('img', { src: data.objectTypeIcon, className: 'mr5', title: data.objectTypeTitle }))
                .insert(new Element('span', { title: data.objectTitle, className: 'mr-auto ' + (positioned || this.options.activeRoomplanId === data.objectId ? 'mouse-pointer ' : 'text-neutral-400 ') + (this.options.activeRoomplanId === data.objectId ? 'text-bold' : '') })
                    .update(data.objectTitle + (counter !== null ? ' <em class="opacity-50">(' + counter + ')</em>' : '')))
                .insert($addAllButton)
                .insert($removeAllButton)
                .insert($openFloorplan)
                .insert($toggleButton))
            .insert(new Element('ul', { className: 'css-tree ' + (open ? '' : 'hide') }));
    },

    /**
     * Overwrite "process" method for some specific logic.
     */
    process: function ($super) {
        $super();

        // Remove function buttons from the root node.
        this.$container.select('.root button').invoke('remove');

        // In this mode we want to display numbers underneath each node, that has positioned objects.
        if (this.options.mode === 'locations' && this.refreshCounter) {
            this.refreshCounter = false;

            new Ajax.Request(window.www_dir + 'floorplan/tree/getPositionedObjectTreeHierarchy', {
                parameters: {
                    positionedObjects: JSON.stringify(this.positionedObjects.map(function (nodeId) {
                        return nodeId.substr(1);
                    }).uniq())
                },
                onComplete: function (xhr) {
                    try {
                        var json = xhr.responseJSON;

                        this.counterCache = json.data;

                        this.process();
                    } catch (e) {
                        this.counterCache = null;
                    }
                }.bind(this)
            });
        }
    },

    /**
     * Method for loading children nodes via ajax.
     *
     * @param objectId
     * @param callback
     */
    loadChildrenNodes: function (objectId, callback) {
        if (this.loading.hasOwnProperty('o' + objectId)) {
            // Do nothing, because loading is already in progess...
            return;
        }

        this.loading['o' + objectId] = true;

        new Ajax.Request(window.www_dir + 'floorplan/tree/getChildLocations/' + objectId, {
            parameters: {
                mode: this.options.mode
            },
            onComplete: function (xhr) {
                var json = xhr.responseJSON, i;

                if (!json.success) {
                    idoit.Notify.error(json.message || xhr.responseText, {sticky: true});
                }

                this.cache['n' + objectId] = json.data;

                for (i in json.data.children) {
                    // Don't overwrite the node, if it contained itself.
                    if (!json.data.children.hasOwnProperty(i) || json.data.children[i].objectId == objectId) {
                        continue;
                    }

                    this.cache['n' + json.data.children[i].objectId] = json.data.children[i];
                }

                if (Object.isFunction(callback)) {
                    callback(objectId, this.cache['n' + objectId]);
                }

                delete this.loading['o' + objectId];
            }.bind(this)
        });
    }
});
