/**
 * i-doit "circle" javascript class.
 *
 * @author  Leonard Fischer <lfischer@i-doit.com>
 */
window.FloorplanVisualizationTypeFloorplan = Class.create(window.FloorplanVisualizationTypeMaster, {
    update: function (options) {
        if (!this.ready) {
            return;
        }

        const that = this;

        if (this.options.layout) {
            // Define the extent size according to the layout
            this.extentWidth = d3.extent(this.options.layout, function (d) { return d.x; });
            this.extentHeight = d3.extent(this.options.layout, function (d) { return d.y; });
        } else if (this.options.backgroundImage) {
            this.extentWidth = [0, this.options.backgroundImage[0]];
            this.extentHeight = [0, this.options.backgroundImage[1]];
        }

        Object.extend(this.options, options || {transition: false});

        this.$container
            .style('opacity', function (d) { return d.visibility; })
            .classed('hide', function (d) { return d.visibility === 0; })
            .interrupt()
            .transition()
            .duration(that.options.transition ? 400 : 0)
            .attr('transform', function (d) {
                return 'translate(' + d.x + ',' + d.y + ')rotate(' + d.angle + ')';
            });

        if (this.extentWidth && this.extentHeight) {
            this.$container.select('.inner-content')
                .interrupt()
                .transition()
                .duration(that.options.transition ? 400 : 0)
                .attr('transform', function (d) {
                    const scaleX = d.width / d3.sum(that.extentWidth, function (x) {return Math.abs(x)});
                    const scaleY = d.height / d3.sum(that.extentHeight, function (y) {return Math.abs(y)});

                    // We translate, in case a layout is not perfectly centered.
                    const left = d3.mean(that.extentWidth.map(function (x) {return x * scaleX})) * -1;
                    const top = d3.mean(that.extentHeight.map(function (y) {return y * scaleY})) * -1;

                    return 'translate(' + left + ',' + top + ')scale(' + scaleX + ',' + scaleY + ')';
                });
        }

        this.$container.selectAll('.rotate-image,.toggle-floorplan').dispatch('update:position');

        // Scale the objects according to the origin scale.
        this.$container.select('.vis-objects').attr('transform', 'scale(' + (1 / (that.options.originScale * 10)) + ')');

        // Re-position the scale circles and drag-element.
        this.renderDragArea();
    },

    render: function () {
        const that = this;
        const floorplan = new window.Floorplan(this.$container.node());

        new Ajax.Request(window.www_dir + 'floorplan/ajax/getFloorplanData', {
            parameters: {
                objectId: this.data.objId
            },
            onComplete: function (response) {
                var json = response.responseJSON, i;

                if (!json) {
                    idoit.Notify.error(response.responseText, {sticky: true});
                    return;
                }

                if (json && !json.success) {
                    idoit.Notify.error(json.message, {sticky: true});
                    return;
                }

                try {
                    var selectedProfile = null;

                    // Add a virtual "root" layer that lies beneath everything else.
                    json.data.layerData.push({
                        id:   null,
                        sort: 9999
                    });

                    // Assign the layout and origin scale. This will be used for scaling the object later.
                    that.options.originScale = json.data.floorplanOptions.scale;
                    that.options.layout = null;
                    that.options.backgroundImage = null;

                    if (json.data.layout) {
                        that.options.layout = json.data.layout.data;
                    }

                    if (Array.isArray(json.data.floorplanOptions.backgroundImageSize)) {
                        that.options.backgroundImage = json.data.floorplanOptions.backgroundImageSize;
                    }

                    // @see  FP-84  Cleaning up some (by now) unused options.
                    delete json.data.floorplanOptions.displayScale;
                    delete json.data.floorplanOptions.measurementValue;
                    delete json.data.floorplanOptions.measurementUnit;
                    delete json.data.floorplanOptions.scaleFactor;
                    delete json.data.floorplanOptions.scaleFactorMM;

                    // Remove some additional options that are not necessary for this visualization.
                    json.data.floorplanOptions.scale = 0.1;
                    json.data.floorplanOptions.gridDisplay = false;
                    json.data.floorplanOptions.scaleShow = false;
                    json.data.floorplanOptions.snapToGrid = false;
                    json.data.floorplanOptions.includeHandlers = false;
                    json.data.floorplanOptions.floorplanLevel = that.options.floorplanLevel + 1;

                    // Add the "onObjSelect" handler for nested objects.
                    json.data.floorplanOptions.onObjSelect = that.options.onObjSelect;

                    floorplan
                        .preProcess()
                        .setData(json.data.objectData)
                        .setLayers(json.data.layerData)
                        .extendOptions(json.data.floorplanOptions)
                        .resetBackgroundImage()
                        .setBackground(json.data.layout, json.data.forms)
                        .process();

                    floorplan.responsive();

                    // Clear the container HTML and add the floorplan visualization.
                    that.$container.html('').append(function () {
                        // Add the 'inner-content' class and pass the node to the container.
                        return floorplan.$container.classed('inner-content', true).node();
                    });

                    // Add the 'pointer' to the floorplan backgrounds.
                    that.$container.selectAll('.vis-background-image image,.vis-background polygon,.vis-background .forms .object')
                        .classed('mouse-pointer', true);

                    that.$container.on('click', function () {
                        that.$container.selectAll('.clicked').classed('clicked', false);

                        if (Element.up(d3.event.target, '.vis-background,.vis-background-image')) {
                            floorplan.unselectObject();

                            that.options.onObjUnselect(true);
                        }
                    });

                    // Add 'click' handler via custom logic, since there appears to be a problem when handling this floorplan-logic internally.
                    that.$container.selectAll('.floorplan-object.level-' + floorplan.uuid + '[data-id]')
                        .each(function (d, i, selection) {
                            // Using a "foreach" approach, since the D3 way mixes up data.
                            d3.select(selection[i]).on('click', function () {
                                d3.event.stopPropagation();

                                floorplan.unselectObject();
                                that.options.onObjSelect(d, true);

                                d3.select(this).classed('clicked', true);

                                // Open the object information tab (the callback will check the profile).
                                that.options.triggerObjectInformationTab();
                            });
                        });

                    that.ready = true;
                    that.update();
                } catch (e) {
                    idoit.Notify.error(e, {sticky: true});
                }
            }
        });
    }
});
