<?php

namespace idoit\Module\Floorplan\Model;

use idoit\Model\Dao\Base;
use isys_component_dao_result;
use isys_format_json;

/**
 * i-doit Floorplan Layer Model
 *
 * @package     Modules
 * @subpackage  Floorplan
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class Layer extends Base
{
    /**
     * This array holds all available table fields of "isys_floorplan_layers".
     *
     * @var  array
     */
    public const LAYER_FIELDS = [
        'isys_floorplan_layers__id' => 'id',
        'isys_floorplan_layers__title' => 'title',
        'isys_floorplan_layers__color' => 'color',
        'isys_floorplan_layers__isys_floorplan__id' => 'floorplanId',
        'isys_floorplan_layers__sort' => 'sort',
        'isys_floorplan_layers__configuration' => 'configuration',
        'isys_floorplan_layers__status' => 'status'
    ];

    /**
     * Method for retrieving all layers.
     *
     * @param string $condition
     *
     * @return  isys_component_dao_result
     * @throws  \Exception
     * @throws  \isys_exception_database
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function getData($condition = null): isys_component_dao_result
    {
        return $this->retrieve('SELECT ' . $this->selectImplode(self::LAYER_FIELDS) . ' FROM isys_floorplan_layers WHERE TRUE ' . ($condition ?: '') . ';');
    }

    /**
     * Method for saving a layer and its data.
     *
     * @param integer $layerId
     * @param array   $layerData
     *
     * @return  integer
     * @throws  \isys_exception_dao
     * @throws  \isys_exception_database
     */
    public function saveLayer($layerId, array $layerData = []): int
    {
        if (count($this->getData('AND isys_floorplan_layers__id = ' . $this->convert_sql_id($layerId)))) {
            $sql = 'UPDATE isys_floorplan_layers SET %s WHERE isys_floorplan_layers__id = ' . $this->convert_sql_id($layerId) . ';';
        } else {
            $sql = 'INSERT INTO isys_floorplan_layers SET %s;';
        }

        foreach ($layerData as $field => &$value) {
            if (!isset(self::LAYER_FIELDS[$field])) {
                unset($layerData[$field]);
            }

            switch ($field) {
                case 'isys_floorplan_layers__id':
                case 'isys_floorplan_layers__isys_floorplan__id':
                    $value = $field . ' = ' . $this->convert_sql_id($value);
                    break;
                case 'isys_floorplan_layers__title':
                case 'isys_floorplan_layers__color':
                    $value = $field . ' = ' . $this->convert_sql_text($value);
                    break;
                case 'isys_floorplan_layers__configuration':
                    $value = $field . ' = ' . $this->convert_sql_text(isys_format_json::encode($value));
                    break;
                case 'isys_floorplan_layers__status':
                case 'isys_floorplan_layers__sort':
                    $value = $field . ' = ' . $this->convert_sql_int($value);
                    break;
            }
        }

        if (\count($layerData) && $this->update(str_replace('%s', implode(',', $layerData), $sql)) && $this->apply_update()) {
            return $this->get_last_insert_id();
        }

        return 0;
    }

    /**
     * Method for removing an layer.
     *
     * @param integer $layerId
     *
     * @return  boolean
     * @throws  \isys_exception_dao
     */
    public function deleteLayer($layerId): bool
    {
        $sql = 'DELETE FROM isys_floorplan_layers 
            WHERE isys_floorplan_layers__id = ' . $this->convert_sql_id($layerId) . ';';

        return $this->update($sql) && $this->apply_update();
    }
}
