<?php

namespace idoit\Module\Floorplan\Model;

use idoit\Model\Dao\Base;

/**
 * i-doit Floorplan Layout Model
 *
 * @package     Modules
 * @subpackage  Floorplan
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class LayoutDao extends Base
{
    /**
     * This array holds all available table fields of "isys_floorplan_layouts".
     *
     * @var array
     */
    const LAYOUT_FIELDS = [
        'isys_floorplan_layouts__id',
        'isys_floorplan_layouts__title',
        'isys_floorplan_layouts__layout',
        'isys_floorplan_layouts__forms',
        'isys_floorplan_layouts__deletable',
        'isys_floorplan_layouts__status'
    ];

    /**
     * Method for retrieving all layouts.
     *
     * @param string $condition
     *
     * @return \isys_component_dao_result
     * @throws \Exception
     * @throws \isys_exception_database
     */
    public function getData(?string $condition = null): \isys_component_dao_result
    {
        $selectAs = [];

        foreach (self::LAYOUT_FIELDS as $field) {
            $fields = explode('__', $field);
            $selectAs[] = $field . ' AS "' . end($fields) . '"';
        }

        return $this->retrieve('SELECT ' . implode(',', $selectAs) . ' FROM isys_floorplan_layouts WHERE TRUE ' . ($condition ?: '') . ';');
    }

    /**
     * Method for saving a layout and its data.
     *
     * @param int   $layoutId
     * @param array $layoutData
     *
     * @return int
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     */
    public function saveLayout(?int $layoutId = null, array $layoutData = []): int
    {
        if (count($this->getData('AND isys_floorplan_layouts__id = ' . $this->convert_sql_id($layoutId)))) {
            $sql = 'UPDATE isys_floorplan_layouts SET %s WHERE isys_floorplan_layouts__id = ' . $this->convert_sql_id($layoutId) . ';';
        } else {
            $sql = 'INSERT INTO isys_floorplan_layouts SET %s;';
        }

        foreach ($layoutData as $field => &$value) {
            if (!in_array($field, self::LAYOUT_FIELDS, true)) {
                unset($layoutData[$field]);
            }

            switch ($field) {
                case 'isys_floorplan_layouts__id':
                    $value = $field . ' = ' . $this->convert_sql_id($value);
                    break;
                case 'isys_floorplan_layouts__title':
                case 'isys_floorplan_layouts__layout':
                case 'isys_floorplan_layouts__forms':
                    $value = $field . ' = ' . $this->convert_sql_text($value);
                    break;
                case 'isys_floorplan_layouts__deletable':
                case 'isys_floorplan_layouts__status':
                    $value = $field . ' = ' . $this->convert_sql_int($value);
                    break;
            }
        }

        if (count($layoutData) && $this->update(str_replace('%s', implode(',', $layoutData), $sql)) && $this->apply_update()) {
            return (int)($layoutId ?: $this->get_last_insert_id());
        }

        return 0;
    }

    /**
     * Returns true/false wether a layout is deletable or not.
     *
     * @param int $layoutId
     *
     * @return bool
     * @throws \isys_exception_database
     */
    public function isDeletable(int $layoutId): bool
    {
        return (bool)$this
            ->getData('AND isys_floorplan_layouts__id = ' . $this->convert_sql_id($layoutId))
            ->get_row_value('deletable');
    }

    /**
     * Method for removing an layout.
     *
     * @param int $layoutId
     *
     * @return bool
     * @throws \isys_exception_dao
     */
    public function deleteLayout(int $layoutId): bool
    {
        $sql = 'DELETE FROM isys_floorplan_layouts WHERE isys_floorplan_layouts__id = ' . $this->convert_sql_id($layoutId) . ' AND isys_floorplan_layouts__deletable = 1;';

        return $this->update($sql) && $this->apply_update();
    }
}
