/**
 * i-doit layout sidebar javascript class.
 *
 * @author  Leonard Fischer <lfischer@i-doit.com>
 */
window.LayoutSidebar = Class.create({
    options: {},
    
    initialize: function ($el, options) {
        const that = this;
        
        this.$element = $el;
        this.options = {
            $canvas: null,
            currentSelectionType: null,
            currentForm: null
        };
        
        Object.extend(this.options, options || {});
    
        this.$tabContainer = $('floorplan-layout-popup-sidebar-tabs');
    
        // Grid options.
        this.$gridSnapToGrip = $('grid-snap-to-grid');
        
        // Design options.
        this.$designTab = $('floorplan-layout-popup-sidebar-design');
        this.$designSelectionX = $('design-selection-x');
        this.$designSelectionY = $('design-selection-y');
        this.$designSelectionRotation = $('design-selection-rotation');
        this.$designFillOpacity = $('design-background-transparency');
        this.$designFill = $('design-background-color');
        this.$designStrokeWidth = $('design-border-size');
        this.$designStroke = $('design-border-color');
        
        // Form options.
        this.$formList = $('floorplan-layout-popup-sidebar-forms');
        
        new Tabs(this.$tabContainer, {
            wrapperClass: 'browser-tabs',
            contentClass: 'browser-tab-content',
            tabClass: 'text-shadow mouse-pointer'
        });
        
        this.$formList.select('.form-preview').each(function($preview) {
            const $svg = d3.create('svg')
                .attr('width', '24')
                .attr('height', '24')
                .attr('viewBox', '0 0 24 24')
                .node();
            
            const $form = createForm($preview.readAttribute('data-type'))
                .setStyle({
                    fill: '#ffffff',
                    fillOpacity: 1,
                    stroke: '#000000',
                    strokeWidth: 1
                })
                .setOptions(JSON.parse($preview.readAttribute('data-options')))
                .setTransform(JSON.parse($preview.readAttribute('data-transform')))
                .render()
                .node();
            
            $svg.append($form);
            
            $preview.update($svg.outerHTML);
        })
        
        this.setObserver();
        
        return this;
    },
    
    setObserver: function () {
        const that = this;
        
        this.$element.on('layout:selected', function(ev) {
            that.options.currentSelectionType = 'layout';
            that.currentForm = null;
            that.formSelected.call(that, ev);
        });
        
        this.$element.on('form:selected', function(ev) {
            that.options.currentSelectionType = 'form';
            that.currentForm = ev.memo.form;
            that.formSelected.call(that, ev);
        });
        
        this.$element.on('point:selected', this.pointSelected.bind(this));
        this.$element.on('point:unselect', this.pointUnselected.bind(this));
        this.$element.on('form:unselected', this.formUnselected.bind(this));
    
        this.$gridSnapToGrip.on('change', this.setSnapToGrid.bind(this));
        this.$designTab.select('.js-color').each(function ($el) {
            $el.store('color', new jscolor.color($el, {hash: true}));
        });
    
        this.$designTab.on('change', 'input,select', this.formDesignChanged.bind(this));
        this.$designSelectionX.on('change', this.formPositionChanged.bind(this));
        this.$designSelectionY.on('change', this.formPositionChanged.bind(this));
        this.$designSelectionRotation.on('change', this.formPositionChanged.bind(this));
        
        this.$formList.on('click', 'button', this.addFormToEditor.bind(this));
    },
    
    process: function () {
        this.zoomTo(this.width / 2, this.height / 2, 1);
        
        return this;
    },
    
    setSnapToGrid: function (ev) {
        this.options.$canvas.fire('grid:toggle-snap-to-grid', {
            snapToGrid: this.$gridSnapToGrip.checked
        })
    },
    
    formSelected: function (ev) {
        const style = ev.memo.form.style;

        this.$designFillOpacity.setValue(style.fillOpacity * 100).enable();
        this.$designFill.setValue(style.fill).enable();
        this.$designFill.retrieve('color').fromString(style.fill.replace(/#/, ''));
        
        this.$designStrokeWidth.setValue(style.strokeWidth).enable();
        this.$designStroke.setValue(style.stroke).enable();
        this.$designStroke.retrieve('color').fromString(style.stroke.replace(/#/, ''));
        
        if (this.options.currentSelectionType === 'form') {
            this.$designSelectionX.setValue(this.currentForm.transform.x).enable();
            this.$designSelectionY.setValue(this.currentForm.transform.y).enable();
            this.$designSelectionRotation.setValue(this.currentForm.transform.r).enable();
            
            this.prepareDesignInputs();
        } else {
            this.$designSelectionX.setValue(0).disable();
            this.$designSelectionY.setValue(0).disable();
            this.$designSelectionRotation.setValue(0).disable();
        }

        this.$tabContainer.down('li a', 1).simulate('click');
    },
    
    prepareDesignInputs: function () {
        this.removeDesignInputs();
    
        for (const i in this.currentForm.options) {
            if (! this.currentForm.options.hasOwnProperty(i)) {
                continue;
            }
    
            this.createSpecificField(i, this.currentForm.options[i]);
        }
    },
    
    removeDesignInputs: function () {
        this.$designTab.down('table').select('.specific').invoke('remove');
    },
    
    createSpecificField: function (type, value) {
        const id = 'design-selection-' + type;
        const label = idoit.Translate.get('LC__MODULE__FLOORPLAN__POPUP_LAYOUT__TAB_DESIGN__OPTION__' + type.toUpperCase());
        const $table = this.$designTab.down('table');
        
        $table
            .insert(new Element('tr', {className: 'specific'})
                .update(new Element('td', {className: 'key pr20'})
                    .update(new Element('label', {htmlFor: id})
                        .update(label + ' (cm)')))
                .insert(new Element('td')
                    .update(new Element('input', {className: 'input input-size-block', 'data-identifier': type, id: id, name: id, size: 65, type: 'text'}).setValue(value))));
    },
    
    pointSelected: function (ev) {
        this.currentPoint = ev.memo;

        this.$tabContainer.down('li a', 1).simulate('click');
    
        this.options.currentSelectionType = 'point';
        this.$designSelectionX.setValue(this.currentPoint.data.x).enable();
        this.$designSelectionY.setValue(this.currentPoint.data.y).enable();
    },
    
    formPositionChanged: function () {
        const valueX = window.FloorplanHelper.processNumber(this.$designSelectionX.getValue());
        const valueY = window.FloorplanHelper.processNumber(this.$designSelectionY.getValue());
        const valueRotation = window.FloorplanHelper.processNumber(this.$designSelectionRotation.getValue());
        
        this.$designSelectionX.setValue(valueX);
        this.$designSelectionY.setValue(valueY);
        this.$designSelectionRotation.setValue(valueRotation);
        
        if (this.options.currentSelectionType === 'form') {
            this.options.$canvas.fire('form:design-updated', {
                x: valueX,
                y: valueY,
                r: valueRotation
            });
        } else {
            this.options.$canvas.fire(this.options.currentSelectionType + ':position-updated', {
                pointIndex: this.currentPoint.index,
                x: valueX,
                y: valueY
            });
        }
    },
    
    formDesignChanged: function (ev) {
        const $fields = this.$designTab.select('[data-identifier]');
        const data = {};
        
        for (const i in $fields) {
            if (! $fields.hasOwnProperty(i)) {
                continue;
            }
    
            data[$fields[i].readAttribute('data-identifier')] = $fields[i].getValue();
        }
        
        this.options.$canvas.fire(this.options.currentSelectionType + ':design-updated', data);
    },
    
    pointUnselected: function () {
        this.currentForm = null;
        
        this.removeDesignInputs();
        
        // Set default values and disable the fields.
        this.$designSelectionX.setValue('0').disable();
        this.$designSelectionY.setValue('0').disable();
        this.$designSelectionRotation.setValue('0').disable();
    },
    
    formUnselected: function () {
        this.currentForm = null;
        
        this.removeDesignInputs();
    
        // Set default values and disable the fields.
        this.$designSelectionX.setValue('0').disable();
        this.$designSelectionY.setValue('0').disable();
        this.$designSelectionRotation.setValue('0').disable();
        this.$designFillOpacity.setValue('0').disable();
        this.$designFill.setValue('#FFFFFF').disable();
        this.$designStrokeWidth.setValue('m').disable();
        this.$designStroke.setValue('#000000').disable();
    },
    
    addFormToEditor: function (ev) {
        const $li = ev.findElement('li');
        
        this.options.$canvas.fire('form:add', {
            type: $li.down('[data-type]').readAttribute('data-type')
        });
    }
});