/**
 * @todo  Write the current Polygon of the selected object in a formfield and input this into the polygonEditor, when it's loaded!
 */

/**
 * i-doit polygon editor javascript base class.
 *
 * @author  Leonard Fischer <lfischer@i-doit.com>
 */
window.PolygonEditor = Class.create({
    svg:     null,
    vis:     null,
    options: {},
    
    getOption: function (option) {
        return this.options[option];
    },
    
    setOption: function (option, value) {
        this.options[option] = value;
        
        return this;
    },
    
    initialize: function ($el, options) {
        var that = this, x = 100, y = 100;
        
        this.$element = $el;
        this.options = {
            objectId:              0,
            objectTitle:           '',
            objectTypeTitle:       '',
            objectTypeIcon:        '',
            objectTypeColor:       '',
            onPointAfterDrag:      Prototype.emptyFunction, // Event for when a point gets dragged.
            onPointAfterAdd:       Prototype.emptyFunction, // Event for when a point gets added.
            onPointAfterRemove:    Prototype.emptyFunction, // Event for when a point gets removed.
            minPoints:             3,                       // The number of points is not allowed to drop underneath this limit.
            maxPoints:             20,                      // The number of points is not allowed to exceed above this limit.
            maxPointsNotification: '',                      // This notification will be displayed, when the "maxPoints" have been reached.
            polygons:              {},                      // This object will hold all pre- and user-defined polygons.
            objectPolygon:         false                    // If a polygon is given, it will be used as the default polygon (on startup).
        };
        
        this.width = this.$element.getWidth();
        this.height = this.$element.getHeight();
        
        Object.extend(this.options, options || {});
        
        // Set the default "point"-drag handler.
        this.drag = d3.drag()
            .on('start', function () {
                that.$adder.classed('hide', true);
            })
            .on('drag', function (d) {
                d.x = Math.round(d3.event.x / 10) * 10;
                d.y = Math.round(d3.event.y / 10) * 10;
                
                if (d.x < -330) {
                    d.x = -330;
                } else if (d.x > 330) {
                    d.x = 330;
                }
                
                if (d.y < -180) {
                    d.y = -180;
                } else if (d.y > 180) {
                    d.y = 180;
                }
                
                // After the coordinates have been set, refresh the polygon.
                that.updatePolygon();
            })
            .on('end', function () {
                that.options.onPointAfterDrag();
            });
        
        // Initialize the SVG element.
        this.svg = d3.select($el).append('svg')
            .attr('width', this.width)
            .attr('height', this.height)
            .on('mousemove', function () {
                var m = d3.mouse(this),
                    p;
                
                m[0] -= that.width / 2;
                m[1] -= that.height / 2;
                
                p = FloorplanHelper.closestPointOnPath(that.$path.node(), m);
                
                that.$adder
                    .classed('hide', p.distance > 40)
                    .attr('cx', p.x)
                    .attr('cy', p.y);
            })
            .append('g')
            .attr('transform', 'translate(' + (this.width / 2) + ',' + (this.height / 2) + ')');
        
        this.vis = this.svg.append('g');
        
        // Initialize the "borders" which indicate the visual boundaries.
        this.vis.append('rect')
            .attr('x', -100)
            .attr('y', -100)
            .attr('width', 200)
            .attr('height', 200)
            .classed('boundaries', true);
        
        this.$polygon = this.vis.append('polygon');
        this.$path = this.vis.append('path');
        this.$adder = this.vis.append('circle')
            .attr('class', 'adder')
            .attr('r', 5)
            .on('dblclick', this.addPoint.bind(this));
        
        this.line =  d3.line()
            .x(function (d) { return d.x; })
            .y(function (d) { return d.y; })
            .curve(d3.curveLinearClosed);
        
        // Write the object type and object title in the center, so the user knows where it will be displayed.
        this.vis.append('text')
            .text(this.options.objectTypeTitle)
            .attr('x', function () {
                return -(this.getComputedTextLength() / 2);
            })
            .attr('y', -5);
        
        this.vis.append('image')
            .attr('xlink:href', this.options.objectTypeIcon)
            .attr('width', 16)
            .attr('height', 16)
            .attr('x', function () {
                return that.vis.select('text').attr('x') - 22;
            })
            .attr('y', -17);
        
        this.vis.append('text')
            .text(this.options.objectTitle)
            .attr('x', function () {
                return -(this.getComputedTextLength() / 2);
            })
            .attr('y', 15);
        
        // We display a rectangle by default.
        if (this.options.objectPolygon) {
            this.data = this.options.objectPolygon;
        } else {
            this.data = [{x: -x, y: -y}, {x: x, y: -y}, {x: x, y: y}, {x: -x, y: y}];
        }
        
        return this;
    },
    
    setData: function (data) {
        this.data = data;
        
        return this;
    },
    
    getData: function () {
        return this.data;
    },
    
    process: function () {
        var $circles;
        
        this.$polygon
            .datum(this.data)
            .attr('fill', this.options.objectTypeColor);
        
        $circles = this.vis.selectAll('circle.dragger')
            .data(this.data);
        
        $circles.enter()
            .append('circle')
            .attr('class', 'dragger')
            .attr('r', 8)
            .call(this.drag)
            .on('dblclick', this.removePoint.bind(this));
        
        $circles.exit()
            .remove();
        
        this.updatePolygon();
        
        return this;
    },
    
    updatePolygon: function () {
        var $circles = this.vis.selectAll('circle.dragger');
        
        this.$path
            .datum(this.data)
            .attr('d', this.line);
        
        this.$polygon
            .attr('points', function (d) {
                return d.map(function (d) {
                    return d.x + ',' + d.y
                }).join(' ');
            });
        
        $circles
            .attr('cx', function (d) { return d.x; })
            .attr('cy', function (d) { return d.y; });
        
        return this;
    },
    
    addPoint: function () {
        var newPoint = [this.$adder.attr('cx'), this.$adder.attr('cy')],
            smallestDistance = Infinity,
            after = 0,
            index, distance, i, indexPoint, nextPoint;
    
        if (this.data.length >= this.options.maxPoints) {
            idoit.Notify.info(this.options.maxPointsNotification.replace('%s', '' + this.options.maxPoints), {life: 5});
            return;
        }
        
        for (i in this.data) {
            if (!this.data.hasOwnProperty(i)) {
                continue;
            }
    
            distance = FloorplanHelper.distanceBetweenPoints([this.data[i].x, this.data[i].y], newPoint);
            
            if (distance < smallestDistance){
                smallestDistance = distance;
                index = parseInt(i);
            }
        }
    
        // Get the next point to evaluate if we need to insert the new point before or after the found index.
        indexPoint = this.data[index];
        nextPoint = this.data[(index === this.data.length - 1 ? 0 : index+1)];
        
        newPoint = {
            x: Math.round(newPoint[0] / 10) * 10,
            y: Math.round(newPoint[1] / 10) * 10
        };
        
        if (FloorplanHelper.checkPointInRectangle(nextPoint.x, nextPoint.y,
            indexPoint.x, nextPoint.y,
            indexPoint.x, indexPoint.y,
            nextPoint.x, indexPoint.y,
            newPoint.x, newPoint.y)) {
            after = 1;
        }
    
        this.data.splice(index + after, 0, newPoint);
        this.options.onPointAfterAdd();
        this.process();
    },
    
    removePoint: function (d, i) {
        if (this.data.length <= this.options.minPoints) {
            return;
        }
        
        this.data.splice(i, 1);
        
        this.options.onPointAfterRemove();
        this.process();
    },
    
    loadPolygon: function (polygon) {
        if (this.options.polygons.hasOwnProperty(polygon)) {
            this.setData(this.options.polygons[polygon].points);
            this.process();
        }
    }
});