/**
 * i-doit Profile javascript class.
 *
 * @author  Leonard Fischer <lfischer@i-doit.com>
 */
window.FloorplanProfile = Class.create({
    options: {},
    
    initialize: function ($el, options, data) {
        this.counter = 1;
        this.currentProfile = 0;
        this.$element = $el;
        this.data = data;
        this.deletions = [];
        this.options = {
            $newProfileButton:                 null,
            $profileList:                      null,
            $profileConfiguration:             null,
            $profileName:                      null,
            $profileHighlightColor:            null,
            $profileDisplayRadius:             null,
            $profileChangeTabOnObjectSelect:   null,
            $profileDblClickToOpenObject:      null,
            $profileDefaultObjectTypeFilter:   null,
            $profilePropertySelectorContainer: null,
            $profileOverwriteGlobalDefault:    null,
            $profileOverwriteUsers:            null,
            $profileOverwriteUserDefaults:     null,
            $saveButton:                       null,
            $cancelButton:                     null,
            primaryProfile:                    0,
            propertySelectorProvides:          0,
            ajaxSmartyUrl:                     '',
            createRight:                       false,
            editRight:                         false,
            deleteRight:                       false,
            supervisorRight:                   false
        };
        
        Object.extend(this.options, options || {});
        
        this.addObserver();
        
        this.process();
        
        return this;
    },
    
    getProfile: function (id) {
        var profile = this.data.filter(function (d) {
            return d.id == id;
        });
        
        if (profile.length !== 1) {
            // throw "Got " + profile.length + " profiles, that match the given ID (" + id + ").";
            return {
                id:              null,
                title:           '',
                translatedTitle: '',
                selfdefined:     true,
                config:          {
                    highlightColor:          '#ffffff',
                    objectTypes:             [],
                    displayRadius:           0,
                    changeTabOnObjectSelect: 0,
                    dblClickToOpenObject:    0
                }
            };
        }
        
        return profile[0];
    },
    
    addObserver: function () {
        var self = this;
        
        this.$element.on('keydown', 'input', function (ev) {
            if (ev.keyCode == Event.KEY_RETURN) {
                ev.preventDefault();
            }
        });
        
        this.options.$newProfileButton.on('click', function () {
            if (!self.options.createRight) {
                idoit.Notify.warning('You are not allowed to create new profiles');
                
                return;
            }
            
            self.createNewProfile();
        });
        
        this.options.$profileList.on('click', 'button[data-action="make-default"]', function (ev) {
            self.makeDefault(ev.findElement('li').readAttribute('data-profile-id'));
        });
        
        this.options.$profileList.on('click', 'button[data-action="edit"]', function (ev) {
            if (!self.options.editRight) {
                idoit.Notify.warning('You are not allowed to edit profiles');
                return;
            }
            
            self.editProfile(ev.findElement('li').readAttribute('data-profile-id'));
        });
        
        this.options.$profileList.on('click', 'button[data-action="duplicate"]', function (ev) {
            if (!self.options.editRight) {
                idoit.Notify.warning('You are not allowed to edit profiles');
                return;
            }
            
            self.duplicateProfile(ev.findElement('li').readAttribute('data-profile-id'));
        });
        
        this.options.$profileList.on('click', 'button[data-action="delete"]', function (ev) {
            if (!self.options.deleteRight) {
                idoit.Notify.warning('You are not allowed to delete profiles');
                return;
            }
            
            self.deleteProfile(ev.findElement('li').readAttribute('data-profile-id'));
        });
        
        this.options.$profileOverwriteGlobalDefault.on('click', function () {
            if (!self.options.supervisorRight) {
                idoit.Notify.warning('You are not allowed to administrate profiles');
                return;
            }
            
            self.makeGlobalDefault(self.currentProfile);
        });
        
        this.options.$profileOverwriteUserDefaults.on('click', function () {
            if (!self.options.supervisorRight) {
                idoit.Notify.warning('You are not allowed to administrate profiles');
                return;
            }
            
            self.makeUserDefault(self.currentProfile, self.options.$profileOverwriteUsers.getValue());
        });
        
        this.options.$saveButton.on('click', function () {
            if (self.options.$profileConfiguration.hasClassName('hide')) {
                self.saveProfiles();
            } else {
                self.saveCurrentProfile();
            }
        });
        
        this.options.$cancelButton.on('click', function () {
            if (self.options.$profileConfiguration.hasClassName('hide')) {
                // Close the popup.
                popup_close('popup_commentary');
            } else {
                // The form is being displayed, simply hide it.
                self.hideForm();
            }
        });
    },
    
    saveProfiles: function () {
        // Deactivate the button and display "loading" state.
        this.options.$saveButton.disable()
            .down('img').writeAttribute('src', window.dir_images + 'ajax-loading.gif');
        
        // Save the profiles.
        new Ajax.Request(window.www_dir + 'floorplan/profile/save', {
            parameters: {
                profiles:        JSON.stringify(this.data),
                deletedProfiles: JSON.stringify(this.deletions)
            },
            onComplete: function (response) {
                var json = response.responseJSON;
                
                if (!json) {
                    idoit.Notify.error(response.responseText, {sticky: true});
                    return;
                }
                
                if (json && !json.success) {
                    idoit.Notify.error(json.message, {sticky: true});
                    return;
                }
                
                // Set the profiles to the global floorplan context.
                window.floorplanData.profiles = json.data;
    
                $('floorplanProfile').fire('update:selection');
                
                // Close the popup.
                popup_close('popup_commentary');
            }
        });
    },
    
    saveCurrentProfile: function () {
        var profile = this.getProfile(this.currentProfile),
            profileObjectInfo = $F('C__VISUALIZATION_PROFILES__OBJ_INFO_CONFIG__COMPLETE');
        
        if (Object.isArray(profile.config)) {
            profile.config = {};
        }
        
        profile.title = this.options.$profileName.getValue();
        profile.config.highlightColor = this.options.$profileHighlightColor.getValue();
        profile.config.displayRadius = this.options.$profileDisplayRadius.getValue();
        profile.config.changeTabOnObjectSelect = this.options.$profileChangeTabOnObjectSelect.getValue();
        profile.config.dblClickToOpenObject = this.options.$profileDblClickToOpenObject.getValue();
        profile.config.objectTypes = this.options.$profileDefaultObjectTypeFilter.getValue();
        profile.config.properties = {};
        
        if (!profileObjectInfo.blank()) {
            try {
                profile.config.properties = JSON.parse(profileObjectInfo);
            } catch (e) {
                idoit.Notify.warning(idoit.Translate.get('LC__FLOORPLAN_PROFILE__OBJECT_INFO_PARSE_ERROR'), {sticky: true});
            }
        }
        
        if (profile.translatedTitle.blank() || profile.title.substr(0, 3) !== 'LC_') {
            profile.translatedTitle = profile.title;
        }
        
        if (profile.id === null) {
            profile.id = this.currentProfile;
            
            this.data.push(profile);
        }
        
        this.currentProfile = 0;
        
        this.process();
        this.hideForm();
    },
    
    process: function () {
        var i, $li, $editButton, $duplicateButton, $deleteButton, primary;
        
        // Build the profile list, according to the data.
        this.options.$profileList.update();
        
        for (i in this.data) {
            if (!this.data.hasOwnProperty(i)) {
                continue;
            }
            
            primary = (this.data[i].id == this.options.primaryProfile);
            $editButton = '';
            $duplicateButton = '';
            $deleteButton = '';
            
            if (this.options.editRight) {
                $editButton = new Element('button', {
                    type: 'button',
                    className: 'btn mr5',
                    title: idoit.Translate.get('LC__FLOORPLAN_PROFILE__EDIT_PROFILE'),
                    'data-tooltip': 1,
                    'data-action': 'edit',
                    disabled: this.data[i].selfdefined ? null : 'disabled'
                })
                    .update(new Element('img', { src: window.dir_images + 'axialis/basic/tool-pencil.svg', alt: '' }));
                
                $duplicateButton = new Element('button', {
                    type: 'button',
                    className: 'btn mr5',
                    title: idoit.Translate.get('LC__FLOORPLAN_PROFILE__DUPLICATE_PROFILE'),
                    'data-tooltip': 1,
                    'data-action': 'duplicate',
                })
                    .update(new Element('img', { src: window.dir_images + 'axialis/basic/pages-stack.svg', alt: ''}));
            }
            
            if (this.data[i].selfdefined && this.options.deleteRight) {
                $deleteButton = new Element('button', {
                    type: 'button',
                    className: 'btn ml-auto',
                    title: idoit.Translate.get('LC__FLOORPLAN_PROFILE__DELETE_PROFILE'),
                    'data-tooltip': 1,
                    'data-action': 'delete',
                })
                    .update(new Element('img', { src: window.dir_images + 'axialis/basic/symbol-cancel.svg', alt: ''}));
            }
            
            $li = new Element('li')
                .writeAttribute('data-profile-id', this.data[i].id)
                .insert(new Element('button', {
                    type: 'button',
                    className: 'btn mr5',
                    title: idoit.Translate.get('LC__FLOORPLAN_PROFILE__SET_DEFAULT_PROFILE' + (this.data[i].id > 0 ? '' : '_FIRST_SAVE')),
                    'data-tooltip': 1,
                    'data-action': 'make-default',
                    disabled: this.data[i].id > 0 ? null : 'disabled'
                })
                    .update(new Element('div', { className: 'bullet-marker', style: 'background-color:' + (primary ? '#0ba04a' : '#e42d2c') + ';' })))
                .insert($editButton)
                .insert($duplicateButton)
                .insert(new Element('div')
                    .writeAttribute('class', 'cmdb-marker')
                    .writeAttribute('style', 'background-color:' + (this.data[i].config.highlightColor || '#ffffff') + ';'))
                .insert(new Element('strong')
                    .writeAttribute('class', 'cb')
                    .update(this.data[i].translatedTitle))
                .insert($deleteButton);
            
            this.options.$profileList.insert($li);
        }
    
        $('body').fire('update:tooltips');
    },
    
    setForm: function (data) {
        data = Object.extend({
            id:              null,
            title:           '',
            translatedTitle: '',
            selfdefined:     true,
            config:          {
                highlightColor:          '#ffffff',
                objectTypes:             [],
                displayRadius:           0,
                changeTabOnObjectSelect: 0,
                dblClickToOpenObject:    0
            }
        }, data || {});
        
        // This will happen, when we click "new".
        if (data.id === null) {
            data.id = -(this.counter++);
        }
        
        // Set the current profile (necessary for saving).
        this.currentProfile = data.id;
        
        // Set the profile name.
        this.options.$profileName.setValue(data.title);
        
        // Reset the highlight color.
        this.options.$profileHighlightColor.setValue(data.config.highlightColor).simulate('change');
        new jscolor.color(this.options.$profileHighlightColor, {hash: true});
        
        // Set other options.
        this.options.$profileDisplayRadius.setValue(data.config.displayRadius);
        this.options.$profileChangeTabOnObjectSelect.setValue(data.config.changeTabOnObjectSelect);
        this.options.$profileDblClickToOpenObject.setValue(data.config.dblClickToOpenObject);
        
        // Reset the object type filter + chosen.
        this.options.$profileDefaultObjectTypeFilter.setValue(data.config.objectTypes);
        this.options.$profileDefaultObjectTypeFilter.simulate('change');
        this.options.$profileDefaultObjectTypeFilter.fire('chosen:updated');
    },
    
    showForm: function (profile) {
        // Reset the form.
        this.setForm(profile);
        
        // Show the form and hide the list and "new" button.
        this.options.$profileConfiguration.removeClassName('hide');
        this.options.$profileList.addClassName('hide');
        this.options.$newProfileButton.addClassName('hide');
        
        // Set focus to the name field.
        this.options.$profileName.focus();
        
        this.options.$cancelButton.down('span').update(idoit.Translate.get('LC_UNIVERSAL__ABORT'));
    },
    
    hideForm: function () {
        // Hide the form and show the list and "new" button.
        this.options.$profileConfiguration.addClassName('hide');
        this.options.$profileList.removeClassName('hide');
        this.options.$newProfileButton.removeClassName('hide');
        
        this.options.$cancelButton.down('span').update(idoit.Translate.get('LC__VISUALIZATION_PROFILES__CLOSE_POPUP'));
    },
    
    createNewProfile: function () {
        this.showForm();
    
        this.reloadPropertySelector({});
    },
    
    makeDefault: function (id) {
        var self    = this,
            $button = this.options.$profileList.down('[data-profile-id="' + id + '"] [data-action="make-default"]');
        
        if (id <= 0) {
            idoit.Notify.warning('You need to save the profiles, before defining them as default!');
            return;
        }
        
        if ($button) {
            $button.disable()
                .update(new Element('img', { src: window.dir_images + 'axialis/user-interface/loading.svg', className: 'animation-rotate' }));
        }
        
        // Save the profiles.
        new Ajax.Request(window.www_dir + 'floorplan/profile/makeDefault', {
            parameters: {
                profileId: id
            },
            onComplete: function (response) {
                var json = response.responseJSON;
                
                if (!json) {
                    idoit.Notify.error(response.responseText, {sticky: true});
                    return;
                }
                
                if (json && !json.success) {
                    idoit.Notify.error(json.message, {sticky: true});
                    return;
                }
                
                self.options.$profileList.select('[data-action="make-default"]')
                    .invoke('enable')
                    .invoke('update', new Element('div', { className: 'bullet-marker', style: 'background-color: #e42d2c' }).outerHTML);
                
                if ($button) {
                    $button.update(new Element('div', { className: 'bullet-marker', style: 'background-color: #0ba04a' }));
                }
            }
        });
    },
    
    makeGlobalDefault: function (id) {
        var self = this;
    
        if (id <= 0) {
            idoit.Notify.warning('You need to save the profiles, before defining them as default!');
            return;
        }
        
        this.options.$profileOverwriteGlobalDefault
            .disable()
            .down('img')
            .addClassName('animation-rotate')
            .writeAttribute('src', window.dir_images + 'axialis/user-interface/loading.svg');
    
        // Save the profiles.
        new Ajax.Request(window.www_dir + 'floorplan/profile/makeGlobalDefault', {
            parameters: {
                profileId: id
            },
            onComplete: function (response) {
                var json = response.responseJSON;
            
                if (!json) {
                    idoit.Notify.error(response.responseText, {sticky: true});
                    return;
                }
            
                if (json && !json.success) {
                    idoit.Notify.error(json.message, {sticky: true});
                    return;
                }
            
                self.options.$profileOverwriteGlobalDefault
                    .enable()
                    .down('img')
                    .removeClassName('animation-rotate')
                    .writeAttribute('src', window.dir_images + 'axialis/basic/star.svg');
            }
        });
    },
    
    makeUserDefault: function (id, users) {
        var self = this;
    
        if (id <= 0) {
            idoit.Notify.warning('You need to save the profiles, before defining them as default!');
            return;
        }
        
        this.options.$profileOverwriteUserDefaults
            .disable()
            .down('img')
            .writeAttribute('src', window.dir_images + 'ajax-loading.gif');
        
        // Save the profiles.
        new Ajax.Request(window.www_dir + 'floorplan/profile/makeUserDefault', {
            parameters: {
                profileId: id,
                users: users
            },
            onComplete: function (response) {
                var json = response.responseJSON;
                
                if (!json) {
                    idoit.Notify.error(response.responseText, {sticky: true});
                    return;
                }
                
                if (json && !json.success) {
                    idoit.Notify.error(json.message, {sticky: true});
                    return;
                }
                
                self.options.$profileOverwriteUserDefaults
                    .enable()
                    .down('img')
                    .writeAttribute('src', window.dir_images + 'icons/silk/table_add.png');
            }
        });
    },
    
    editProfile: function (id) {
        var self       = this,
            profile    = this.getProfile(id),
            properties = {};
        
        this.showForm(profile);
        
        if (profile.hasOwnProperty('config') && profile.config.hasOwnProperty('properties')) {
            properties = profile.config.properties;
        }
        
        this.reloadPropertySelector(properties || {});
    },
    
    reloadPropertySelector: function (properties) {
        var self = this,
            lvls = Object.clone(properties),
            parameters;
    
        // Delete the "root" node for the level preselection.
        delete lvls.root
        
        parameters = {
            name:              'C__VISUALIZATION_PROFILES__OBJ_INFO_CONFIG',
            preselection:      JSON.stringify(properties.root),
            preselection_lvls: JSON.stringify({1:lvls}),
            grouping:          false,
            sortable:          true,
            p_bInfoIconSpacer: 0,
            p_bEditMode:       true,
            p_bInfoIcon:       false,
            provide:           this.options.propertySelectorProvides,
            p_consider_rights: true,
            custom_fields:     true,
            report:            true,
            selector_size:     'small',
            dialog_width:      '280px'
        };
        
        new Ajax.Request(this.options.ajaxSmartyUrl + '&mode=edit', {
            parameters: {
                plugin_name: 'f_property_selector',
                parameters:  JSON.stringify(parameters)
            },
            onComplete: function (xhr) {
                var json = xhr.responseJSON;
                
                is_json_response(xhr, true);
                
                if (json.success) {
                    self.options.$profilePropertySelectorContainer
                        .update(json.data)
                        .select('select.chosen-select')
                        .each(function ($select) {
                            new Chosen($select, {
                                disable_search_threshold: 10,
                                width:                    '280px',
                                height:                   '20px',
                                search_contains:          true
                            });
                        });
                }
            }
        });
    },
    
    duplicateProfile: function (id) {
        // "Object.clone()" only creates a shallow copy - but we need a "deep" copy.
        var duplicate = JSON.parse(JSON.stringify(this.getProfile(id)));
        
        if (!confirm(idoit.Translate.get('LC__FLOORPLAN_PROFILE__DUPLICATE_PROFILE_CONFIRM').replace('%s', duplicate.translatedTitle))) {
            return;
        }
        
        // Set a minus counter, so when we create multiple profiles and delete one, only the specific one gets deleted.
        duplicate.id = -(this.counter++);
        duplicate.title = idoit.Translate.get('LC__FLOORPLAN_PROFILE__DUPLICATE') + ': ' + duplicate.translatedTitle;
        duplicate.translatedTitle = duplicate.title;
        duplicate.selfdefined = true;
        
        this.data.push(duplicate);
        this.process();
    },
    
    deleteProfile: function (id) {
        if (!confirm(idoit.Translate.get('LC__FLOORPLAN_PROFILE__DELETE_PROFILE_CONFIRM'))) {
            return;
        }
        
        this.deletions.push(id);
        this.data = this.data.filter(function (d) {
            return d.id != id;
        });
        
        this.process();
    }
});
