<?php

namespace idoit\Module\Floorplan\Controller;

use idoit\Exception\Exception;
use idoit\Module\Floorplan\Model\Floorplan as FloorplanModel;
use idoit\Module\Floorplan\Model\Layer;
use idoit\Module\Floorplan\Model\Layer as LayerModel;
use isys_application as Application;
use isys_convert as Convert;
use isys_format_json as JSON;
use isys_register as Register;

/**
 * i-doit cmdb controller
 *
 * @package     Modules
 * @subpackage  Floorplan
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class Floorplan extends Ajax
{
    /**
     * Method for deleting a floorplan and its (object-) data.
     *
     * @param Register $request
     */
    public function delete(Register $request)
    {
        try {
            \isys_module_floorplan::getAuth()->check(\isys_auth::DELETE, 'visualization');

            $dao = $this->dao(Application::instance());
            $post = $request->get('POST');

            $floorplanObjectId = (int)$post['floorplanObjectId'];

            if ($floorplanObjectId <= 0) {
                throw new Exception($this->getDi()->get('language')->get('LC__MODULE__FLOORPLAN__SELECT_FLOORPLAN_BEFORE_SAVE'));
            }

            $floorplanModel = new \idoit\Module\Floorplan\Model\Floorplan($this->getDi()->get('database'));

            $floorplanId = $floorplanModel
                ->getFloorplanByObjectId($floorplanObjectId)
                ->get_row_value('isys_floorplan__id');

            $floorplanModel->deleteFloorplanByObjectId($floorplanObjectId);
            $floorplanModel->deleteFloorplanObjects($floorplanId);

            // Return the (possibly formatted) options.
            //$this->response['data'] = $floorplanData;
            $this->response['success'] = true;
        } catch (\Exception $e) {
            $this->response['success'] = false;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * Method for saving a floorplan and its (object-) data.
     *
     * @param Register $request
     */
    public function save(Register $request)
    {
        try {
            $dao = $this->dao(Application::instance());
            $post = $request->get('POST');

            $floorplanData = JSON::decode($post['floorplanData']);
            $floorplanObjectId = (int)$post['floorplanObjectId'];
            $floorplanObjectData = JSON::decode($post['floorplanObjectData']);
            $floorplanLayerData = JSON::decode($post['floorplanLayerData']);

            if ($floorplanObjectId <= 0) {
                throw new Exception($this->getDi()->get('language')->get('LC__MODULE__FLOORPLAN__SELECT_FLOORPLAN_BEFORE_SAVE'));
            }

            // @see FP-84  Remove some outdated scaling props.

            // First step: save the floorplan data.
            $dao->saveFloorplan($floorplanObjectId, [
                'isys_floorplan__configuration' => JSON::encode($floorplanData),
                'isys_floorplan__layout' => $post['floorplanLayoutData'],
                'isys_floorplan__forms' => $post['floorplanFormsData']
            ]);

            $floorplanModel = new \idoit\Module\Floorplan\Model\Floorplan($this->getDi()->get('database'));
            $layerModel = new Layer($this->getDi()->get('database'));

            $floorplanId = $floorplanModel
                ->getFloorplanByObjectId($floorplanObjectId)
                ->get_row_value('isys_floorplan__id');

            $floorplanModel->deleteFloorplanObjects($floorplanId);

            // Second step: save the object data.
            if (\is_array($floorplanObjectData) && \count($floorplanObjectData)) {
                foreach ($floorplanObjectData as $index => $object) {
                    $floorplanModel->saveFloorplanObject(
                        (int)$floorplanId,
                        (int)$object['objId'],
                        $index,
                        (array)$object
                    );
                }
            }

            if (\is_array($floorplanLayerData) && \count($floorplanLayerData)) {
                foreach ($floorplanLayerData as $index => $layer) {
                    if (!$layer['id']) {
                        continue;
                    }

                    $layerModel->saveLayer($layer['id'], [
                        'isys_floorplan_layers__title'         => $layer['title'],
                        'isys_floorplan_layers__color'         => $layer['color'],
                        'isys_floorplan_layers__sort'          => $layer['sort'],
                        'isys_floorplan_layers__configuration' => $layer['configuration'],
                    ]);
                }
            }

            // Return the (possibly formatted) options.
            $this->response['data'] = $floorplanData;
        } catch (\Exception $e) {
            $this->response['success'] = false;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * Method for saving a floorplan snapshot (from base64 string)
     * @param Register $request
     */
    public function saveSnapshot(Register $request)
    {
        try {
            $post = $request->get('POST');

            $this->dao(Application::instance())->saveFloorplan($post['floorplanObjectId'], [
                'isys_floorplan__snapshot' => $post['floorplanSnapshot']
            ]);

            $this->response['data'] = true;
        } catch (\Exception $e) {
            $this->response['success'] = false;
            $this->response['message'] = $e->getMessage();
        }
    }
}
