<?php

namespace idoit\Module\Floorplan\Model;

use Exception;
use idoit\Model\Dao\Base;
use isys_application;
use isys_application as Application;
use isys_cmdb_dao_category_g_formfactor as DaoFormfactor;
use isys_cmdb_dao_category_g_image as DaoImages;
use isys_cmdb_dao_category_property as DaoProperty;
use isys_convert as Convert;
use isys_format_json as JSON;
use isys_helper_link as HelperLink;
use isys_notify as Notify;
use isys_tenantsettings as TenantSettings;

/**
 * i-doit Floorplan Model
 *
 * @package     Modules
 * @subpackage  Floorplan
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class Dao extends Base
{
    /**
     * This array holds all available table fields of "isys_floorplan".
     *
     * @var  array
     */
    const FLOORPLAN_FIELDS = [
        'isys_floorplan__id',
        'isys_floorplan__isys_obj__id',
        'isys_floorplan__configuration',
        'isys_floorplan__snapshot',
        'isys_floorplan__layout',
        'isys_floorplan__forms',
        'isys_floorplan__isys_floorplan_type__id',
        'isys_floorplan__status'
    ];

    /**
     * This array holds all available table fields of "isys_floorplan_objects".
     *
     * @var  array
     */
    const FLOORPLAN_OBJECTS_FIELDS = [
        'isys_floorplan_objects__id',
        'isys_floorplan_objects__isys_obj__id',
        'isys_floorplan_objects__isys_floorplan__id',
        'isys_floorplan_objects__configuration',
        'isys_floorplan_objects__status'
    ];

    /**
     * Method for retrieving all necessary object type data.
     *
     * @return  \isys_component_dao_result
     * @throws  Exception
     * @throws  \isys_exception_database
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function getObjectTypes()
    {
        $sql = 'SELECT isys_obj_type__id, isys_obj_type__title, isys_obj_type__icon, isys_obj_type__color
			FROM isys_obj_type
			WHERE isys_obj_type__status = ' . $this->convert_sql_int(C__RECORD_STATUS__NORMAL) . ';';

        return $this->retrieve($sql);
    }

    /**
     * Method for retrieving all objects of a given floorplan.
     *
     * @param   integer $floorplanObjectId
     *
     * @return  \isys_component_dao_result
     * @throws  Exception
     * @throws  \isys_exception_database
     */
    public function getObjectsByFloorplan($floorplanObjectId = null)
    {
        // @see  FP-23  Only display objects with status "normal".
        $sql = 'SELECT 
            isys_obj__id, 
            isys_obj__title, 
            isys_obj__isys_obj_type__id, 
            isys_floorplan_objects__configuration,
            isys_floorplan_objects__isys_floorplan_layers,
            isys_floorplan_objects__isys_floorplan_layers,
            isys_floorplan_objects__sort,
            ST_X(isys_floorplan_objects__gps) AS lat, 
            ST_Y(isys_floorplan_objects__gps) AS lng,
            fp.isys_floorplan__id AS hasOwnFloorplan,
            fp.isys_floorplan__layout AS hasOwnLayout,
            fp.isys_floorplan__configuration AS configuration
			FROM isys_floorplan_objects
			INNER JOIN isys_obj ON isys_obj__id = isys_floorplan_objects__isys_obj__id
			INNER JOIN isys_floorplan AS main ON main.isys_floorplan__id = isys_floorplan_objects__isys_floorplan__id
			LEFT JOIN isys_floorplan AS fp ON fp.isys_floorplan__isys_obj__id = isys_obj__id
			WHERE isys_obj__status = ' . $this->convert_sql_int(C__RECORD_STATUS__NORMAL);

        if ($floorplanObjectId !== null) {
            $sql .= ' AND main.isys_floorplan__isys_obj__id = ' . $this->convert_sql_id($floorplanObjectId);
        }

        $sql .= ' ORDER BY isys_floorplan_objects__sort ASC;';

        return $this->retrieve($sql);
    }

    /**
     * Method for retrieving floorplan data.
     *
     * @param   mixed   $floorplanObjectId May be an integer, an array of integers or null.
     * @param   integer $floorplanStatus
     *
     * @return  \isys_component_dao_result
     * @throws  Exception
     * @throws  \isys_exception_database
     */
    public function getFloorplan($floorplanObjectId = null, $floorplanStatus = C__RECORD_STATUS__NORMAL)
    {
        $sql = 'SELECT isys_floorplan__id, 
            isys_floorplan__configuration, 
            isys_floorplan__layout,
            isys_floorplan__forms,
            isys_obj__title
            FROM isys_floorplan
			INNER JOIN isys_obj ON isys_obj__id = isys_floorplan__isys_obj__id
			WHERE TRUE
			AND isys_floorplan__status = ' . $this->convert_sql_int($floorplanStatus);

        if (is_array($floorplanObjectId)) {
            $sql .= ' AND isys_floorplan__isys_obj__id ' . $this->prepare_in_condition($floorplanObjectId) . ';';
        } elseif (is_numeric($floorplanObjectId) && $floorplanObjectId > 0) {
            $sql .= ' AND isys_floorplan__isys_obj__id = ' . $this->convert_sql_id($floorplanObjectId) . ';';
        }

        return $this->retrieve($sql . ';');
    }

    /**
     * Method for saving a floorplan and its data.
     *
     * @param null  $floorplanObjectId
     * @param array $floorplanData
     *
     * @return int
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     */
    public function saveFloorplan($floorplanObjectId, array $floorplanData = [])
    {
        if (count($this->getFloorplan($floorplanObjectId))) {
            $sql = 'UPDATE isys_floorplan SET %s WHERE isys_floorplan__isys_obj__id = ' . $this->convert_sql_id($floorplanObjectId) . ';';
        } else {
            $sql = 'INSERT INTO isys_floorplan SET %s;';
        }

        $floorplanData['isys_floorplan__isys_obj__id'] = $floorplanObjectId;

        foreach ($floorplanData as $field => &$value) {
            if (!in_array($field, self::FLOORPLAN_FIELDS, true)) {
                unset($floorplanData[$field]);
            }

            switch ($field) {
                case 'isys_floorplan__status':
                    $value = $field . ' = ' . $this->convert_sql_int($value);
                    break;

                case 'isys_floorplan__id':
                case 'isys_floorplan__isys_obj__id':
                case 'isys_floorplan__isys_floorplan_type__id':
                    $value = $field . ' = ' . $this->convert_sql_id($value);
                    break;

                case 'isys_floorplan__configuration':
                case 'isys_floorplan__snapshot':
                case 'isys_floorplan__layout':
                case 'isys_floorplan__forms':
                    $value = $field . ' = ' . $this->convert_sql_text($value);
                    break;
            }
        }

        $floorplanData[] = 'isys_floorplan__updated = NOW()';

        if (count($floorplanData) && $this->update(str_replace('%s', implode(',', $floorplanData), $sql)) && $this->apply_update()) {
            return $this->get_last_insert_id();
        }

        return 0;
    }

    /**
     * Method for getting "object information" via a profile.
     *
     * @param   integer $objectId
     * @param   integer $profileId
     *
     * @return  array
     * @throws  Exception
     * @throws  \isys_exception_database
     */
    public function getObjectInformation($objectId, $profileId)
    {
        global $g_dirs;

        $information = [];
        $emptyValue = TenantSettings::get('gui.empty_value', '-');
        $language = Application::instance()->container->get('language');

        $sql = 'SELECT isys_obj__title, isys_obj_type__id, isys_obj_type__title, isys_obj_type__obj_img_name, isys_obj_type__color, isys_cmdb_status__title, isys_cmdb_status__color
			FROM isys_obj
			INNER JOIN isys_obj_type ON isys_obj_type__id = isys_obj__isys_obj_type__id
			INNER JOIN isys_cmdb_status ON isys_cmdb_status__id = isys_obj__isys_cmdb_status__id
			WHERE isys_obj__id = ' . $this->convert_sql_id($objectId) . ' LIMIT 1;';

        $object = $this->retrieve($sql)
            ->get_row();

        $objectImage = $g_dirs['images'] . 'objecttypes/' . C__OBJTYPE_IMAGE__DEFAULT;

        if (!empty($object['isys_obj_type__obj_img_name'])) {
            $objectImage = $g_dirs['images'] . 'objecttypes/' . $object['isys_obj_type__obj_img_name'];
        }

        $image = DaoImages::instance($this->m_db)
            ->get_data(null, $objectId)
            ->get_row_value('isys_catg_image_list__image_link');

        if (!empty($image)) {
            $objectImage = HelperLink::create_url([
                C__GET__MODULE_ID => C__MODULE__CMDB,
                'file_manager'    => 'image',
                'file'            => urlencode($image)
            ], true);
        }

        $formfactorData = (new DaoFormfactor($this->m_db))->get_data(null, $objectId, '', null, C__RECORD_STATUS__NORMAL)
            ->get_row();

        if ($formfactorData !== false && is_array($formfactorData)) {
            $width = Convert::measure(
                $formfactorData['isys_catg_formfactor_list__installation_width'] ?: 0,
                $formfactorData['isys_depth_unit__id'],
                C__CONVERT_DIRECTION__BACKWARD
            );

            $height = Convert::measure(
                $formfactorData['isys_catg_formfactor_list__installation_height'] ?: 0,
                $formfactorData['isys_depth_unit__id'],
                C__CONVERT_DIRECTION__BACKWARD
            );

            $depth = Convert::measure(
                $formfactorData['isys_catg_formfactor_list__installation_depth'] ?: 0,
                $formfactorData['isys_depth_unit__id'],
                C__CONVERT_DIRECTION__BACKWARD
            );

            $weight = Convert::weight(
                $formfactorData['isys_catg_formfactor_list__installation_weight'],
                $formfactorData['isys_weight_unit__id'],
                C__CONVERT_DIRECTION__BACKWARD
            );

            $information['formfactor'] = [
                'width'        => $width ?: $emptyValue,
                'height'       => $height ?: $emptyValue,
                'depth'        => $depth ?: $emptyValue,
                'depthUnit'    => $language->get($formfactorData['isys_depth_unit__title']) ?: $emptyValue,
                'depthUnitId'  => $formfactorData['isys_depth_unit__id'] ?: null,
                'depthFactor'  => (1000 / Convert::measure(1000, $formfactorData['isys_depth_unit__id'], C__CONVERT_DIRECTION__BACKWARD)) ?: 1,
                'weight'       => $weight ?: $emptyValue,
                'weightUnit'   => $language->get($formfactorData['isys_weight_unit__title']) ?: $emptyValue,
                'weightUnitId' => $formfactorData['isys_weight_unit__id'] ?: null
            ];
        }

        return [
            'object'      => [
                'objId'        => (int)$objectId,
                'objTitle'     => $language->get($object['isys_obj__title']),
                'objImage'     => $objectImage,
                'objTypeId'    => (int)$object['isys_obj_type__id'],
                'objTypeTitle' => $language->get($object['isys_obj_type__title']),
                'objTypeColor' => $object['isys_obj_type__color'],
                'cmdbStatus'   => $language->get($object['isys_cmdb_status__title']),
                'cmdbColor'    => $object['isys_cmdb_status__color']
            ],
            'information' => $information,
            'properties' => $this->getPropertyDataByProfile($objectId, $profileId)
        ];
    }

    /**
     * Loads property data (like a report).
     *
     * @param  integer $objectId
     * @param  integer $profileId
     *
     * @return array
     * @throws \idoit\Exception\JsonException
     * @throws \isys_exception_database
     */
    private function getPropertyDataByProfile($objectId, $profileId)
    {
        $configuration = Profile::instance($this->m_db)
            ->getProfile($profileId)
            ->get_row_value('config');

        if (!JSON::is_json_array($configuration)) {
            return [];
        }

        $configuration = JSON::decode($configuration);

        if (!isset($configuration['properties'])) {
            return [];
        }

        try {
            $query = DaoProperty::instance($this->m_db)
                ->set_query_as_report(true)
                ->create_property_query_for_report(
                    1,
                    $configuration['properties'],
                    true,
                    ' AND obj_main.isys_obj__id = ' . $this->convert_sql_id($objectId) . ' '
                );
        } catch (Exception $exception) {
            Notify::error($exception->getMessage(), ['sticky' => true]);

            return [];
        }

        $propertyDataResult = [];
        $propertyData = $this->retrieve($query)->get_row();
        $language = Application::instance()->container->get('language');

        foreach ($propertyData as $property => $propertyValue) {
            $property = trim($property);

            if (strpos($property, 'isys_cmdb_dao_category_') === 0) {
                $propertyPart = explode('::', $property);

                if (class_exists($propertyPart[0])) {
                    $categoryDao = call_user_func([$propertyPart[0], 'instance'], $this->m_db);

                    if ($propertyValue !== null) {
                        $categoryDaoCallback[$propertyPart[2]] = $propertyValue;

                        $propertyDataResult[] = [
                            $language->get($propertyPart[3]),
                            call_user_func([$categoryDao, $propertyPart[1]], $categoryDaoCallback)
                        ];
                    } else {
                        $propertyDataResult[] = [$language->get($propertyPart[3]), ''];
                    }
                }

                continue;
            }

            if (strpos($property, '__') === 0) {
                continue;
            }

            $propertyTranslation = strstr($property, '###', true);

            if ($propertyTranslation === false) {
                $propertyTranslation = $property;
            }

            if (strpos($propertyTranslation, '#') !== false) {
                $keyParts = array_map(function ($translation) use ($language) {
                    return $language->get($translation);
                }, explode('#', $propertyTranslation));

                $format = '%s';

                if (count($keyParts) === 3) {
                    $format = '%s (%s -> %s)';
                } elseif (count($keyParts) === 2) {
                    $format = '%s (%s)';
                }

                $propertyTranslation = vsprintf($format, $keyParts);
            } else {
                $propertyTranslation = $language->get($propertyTranslation);
            }

            $propertyTranslationCheckParts = explode('::', $propertyTranslation);
            if ($propertyTranslationCheckParts[0]==='locales') {
                $propertyTranslation = implode('::', array_slice($propertyTranslationCheckParts, 2));
            }

            $propertyDataResult[] = [
                $propertyTranslation,
                $language->get($propertyValue)
            ];
        }

        return $propertyDataResult;
    }
}
