<?php

namespace idoit\Module\Floorplan\Model;

use idoit\Model\Dao\Base;
use isys_format_json as JSON;

/**
 * i-doit Floorplan Model
 *
 * @package     Modules
 * @subpackage  Floorplan
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class Floorplan extends Base
{
    /**
     * Remove all objects of a given floorplan.
     *
     * @param  integer $floorplanId
     *
     * @return boolean
     * @throws \isys_exception_dao
     */
    public function deleteFloorplanObjects($floorplanId)
    {
        $sql = 'DELETE FROM isys_floorplan_objects WHERE isys_floorplan_objects__isys_floorplan__id = ' . $this->convert_sql_id($floorplanId) . ';';

        return $this->update($sql) && $this->apply_update();
    }

    /**
     * Get a floorplan row by object ID.
     *
     * @param  integer $objectId
     *
     * @return \isys_component_dao_result
     * @throws \isys_exception_database
     */
    public function deleteFloorplanByObjectId($objectId)
    {
        $sql = 'DELETE FROM isys_floorplan WHERE isys_floorplan__isys_obj__id = ' . $this->convert_sql_id($objectId) . ' LIMIT 1;';

        return $this->retrieve($sql);
    }

    /**
     * Get a floorplan row by object ID.
     *
     * @param  integer $objectId
     *
     * @return \isys_component_dao_result
     * @throws \isys_exception_database
     */
    public function getFloorplanByObjectId($objectId)
    {
        $sql = 'SELECT * FROM isys_floorplan WHERE isys_floorplan__isys_obj__id = ' . $this->convert_sql_id($objectId) . ' LIMIT 1;';

        return $this->retrieve($sql);
    }

    /**
     * Get floorplans, together with snapshots.
     *
     * @param int|array $objectId
     *
     * @return \isys_component_dao_result
     * @throws \isys_exception_database
     */
    public function getFloorplanWithSnapshot($objectId = null)
    {
        $sql = 'SELECT 
            isys_floorplan__id AS id,
            isys_floorplan__snapshot AS snapshot,
            isys_obj__id AS objectId,
            isys_obj__title AS objectTitle
            FROM isys_floorplan
            INNER JOIN isys_obj ON isys_obj__id = isys_floorplan__isys_obj__id';

        if (is_array($objectId)) {
            $sql .= ' WHERE isys_obj__id ' . $this->prepare_in_condition($objectId);
        } elseif (is_numeric($objectId)) {
            $sql .= ' WHERE isys_obj__id = ' . $this->convert_sql_id($objectId);
        }

        return $this->retrieve($sql . ';');
    }

    /**
     * Save a object to a given floorplan.
     *
     * @param int   $floorplanId
     * @param int   $objectId
     * @param int   $sort
     * @param array $data
     *
     * @return bool
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     */
    public function saveFloorplanObject(int $floorplanId, int $objectId, int $sort, array $data = [])
    {
        // At first prepare a valid "configuration" array.
        $sqlFields = [
            'isys_floorplan_objects__isys_obj__id = ' . $this->convert_sql_id($objectId),
            'isys_floorplan_objects__isys_floorplan__id = ' . $this->convert_sql_id($floorplanId),
            'isys_floorplan_objects__sort = ' . $this->convert_sql_int($sort),
            'isys_floorplan_objects__isys_floorplan_layers = ' . $this->convert_sql_id($data['layer']),
            'isys_floorplan_objects__configuration = ' . $this->convert_sql_text(JSON::encode([
                'width'       => $data['width'],
                'height'      => $data['height'],
                'orientation' => $data['orientation'] ?: null,
                'polygon'     => $data['polygon'] ?: false,
                'angle'       => (int)$data['angle'] ?: 0,
                'visibility'  => (float)($data['visibility'] ?? 1),
                'visualizeAsFloorplan'  => (bool)($data['visualizeAsFloorplan'] ?? false),
                'radius'      => [
                    'display' => $data['radius']['display'],
                    'radius'  => (int)$data['radius']['radius'],
                    'unit'    => $data['radius']['unit'],
                    'color'   => $data['radius']['color'],
                    'opacity' => (int)$data['radius']['opacity'],
                ]
            ])),
            'isys_floorplan_objects__gps = GeomFromText("POINT(' . round($data['x'], 5) . ' ' . round($data['y'], 5) . ')")'
        ];

        // First we check if the given object exists in the floorplan.
        $sql = 'SELECT isys_floorplan_objects__id
			FROM isys_floorplan_objects
			WHERE isys_floorplan_objects__isys_floorplan__id = ' . $this->convert_sql_id($floorplanId) . '
			AND isys_floorplan_objects__isys_obj__id = ' . $this->convert_sql_id($objectId) . ' LIMIT 1;';

        $floorplanObjectRowId = $this->retrieve($sql)
            ->get_row_value('isys_floorplan_objects__id');

        if (!$floorplanObjectRowId) {
            $sql = 'INSERT INTO isys_floorplan_objects SET %s;';
        } else {
            $sql = 'UPDATE isys_floorplan_objects SET %s WHERE isys_floorplan_objects__id = ' . $this->convert_sql_id($floorplanObjectRowId) . ';';
        }

        return $this->update(str_replace('%s', implode(', ', $sqlFields), $sql)) && $this->apply_update();
    }

    /**
     *
     *
     * @param  string  $orderBy
     * @param  integer $limit
     *
     * @return \isys_component_dao_result
     * @throws \isys_exception_database
     */
    public function getStartupFloorplans($orderBy, $limit = null) {
        $sql = 'SELECT isys_obj__id
            FROM isys_floorplan 
            INNER JOIN isys_obj ON isys_obj__id = isys_floorplan__isys_obj__id
            WHERE isys_floorplan__status = ' . $this->convert_sql_int(C__RECORD_STATUS__NORMAL) . '
            AND isys_obj__status = ' . $this->convert_sql_int(C__RECORD_STATUS__NORMAL) . '
            AND ' . $orderBy . ' <> ""
            ORDER BY ' . $orderBy . ' DESC
            ' . ($limit !== null ? 'LIMIT ' . $this->convert_sql_int($limit) : '') . ';';

        return $this->retrieve($sql);
    }
}
