<?php

namespace idoit\Module\Floorplan\Model;

use idoit\Model\Dao\Base;

/**
 * i-doit Floorplan Polygon Model
 *
 * @package     Modules
 * @subpackage  Floorplan
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class PolygonDao extends Base
{
    /**
     * This array holds all available table fields of "isys_floorplan_polygons".
     *
     * @var  array
     */
    const POLYGON_FIELDS = [
        'isys_floorplan_polygons__id',
        'isys_floorplan_polygons__title',
        'isys_floorplan_polygons__points',
        'isys_floorplan_polygons__deletable',
        'isys_floorplan_polygons__status'
    ];

    /**
     * Method for retrieving all polygons.
     *
     * @param   string $condition
     *
     * @return  \isys_component_dao_result
     * @throws  \Exception
     * @throws  \isys_exception_database
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function getData($condition = null)
    {
        $selectAs = [];

        foreach (self::POLYGON_FIELDS as $field) {
            $selectAs[] = $field . ' AS "' . end(explode('__', $field)) . '"';
        }

        return $this->retrieve('SELECT ' . implode(',', $selectAs) . ' FROM isys_floorplan_polygons WHERE TRUE ' . ($condition ?: '') . ';');
    }

    /**
     * Method for retrieving necessary object data.
     *
     * @param   integer $objectId
     *
     * @return  \isys_component_dao_result
     * @throws  \Exception
     * @throws  \isys_exception_database
     */
    public function getObjectData($objectId)
    {
        return $this->retrieve('SELECT 
            isys_obj__id AS objId, 
            isys_obj__title AS objTitle, 
            isys_obj_type__title AS objTypeTitle, 
            isys_obj_type__color AS objTypeColor, 
            isys_obj_type__icon AS objTypeIcon
			FROM isys_obj
			INNER JOIN isys_obj_type ON isys_obj_type__id = isys_obj__isys_obj_type__id
			WHERE isys_obj__id = ' . $this->convert_sql_id($objectId) . ';');
    }

    /**
     * Method for saving a polygon and its data.
     *
     * @param   integer $polygonId
     * @param   array   $polygonData
     *
     * @return  integer
     * @throws  \isys_exception_dao
     * @throws  \isys_exception_database
     */
    public function savePolygon($polygonId, array $polygonData = [])
    {
        if (count($this->getData('AND isys_floorplan_polygons__id = ' . $this->convert_sql_id($polygonId)))) {
            $sql = 'UPDATE isys_floorplan_polygons SET %s WHERE isys_floorplan_polygons__id = ' . $this->convert_sql_id($polygonId) . ';';
        } else {
            $sql = 'INSERT INTO isys_floorplan_polygons SET %s;';
        }

        foreach ($polygonData as $field => &$value) {
            if (!in_array($field, self::POLYGON_FIELDS, true)) {
                unset($polygonData[$field]);
            }

            switch ($field) {
                case 'isys_floorplan_polygons__id':
                    $value = $field . ' = ' . $this->convert_sql_id($value);
                    break;
                case 'isys_floorplan_polygons__title':
                case 'isys_floorplan_polygons__points':
                    $value = $field . ' = ' . $this->convert_sql_text($value);
                    break;
                case 'isys_floorplan_polygons__deletable':
                case 'isys_floorplan_polygons__status':
                    $value = $field . ' = ' . $this->convert_sql_int($value);
                    break;
            }
        }

        if (count($polygonData) && $this->update(str_replace('%s', implode(',', $polygonData), $sql)) && $this->apply_update()) {
            return $this->get_last_insert_id();
        }

        return 0;
    }

    /**
     * Returns true/false wether a polygon is deletable or not.
     *
     * @param  integer $polygonId
     *
     * @return boolean
     * @throws \isys_exception_database
     */
    public function isDeletable($polygonId)
    {
        return (bool) $this
            ->getData('AND isys_floorplan_polygons__id = ' . $this->convert_sql_id($polygonId))
            ->get_row_value('deletable');
    }

    /**
     * Method for removing an polygon.
     *
     * @param   integer $polygonId
     *
     * @return  boolean
     * @throws  \isys_exception_dao
     */
    public function deletePolygon($polygonId)
    {
        $sql = 'DELETE FROM isys_floorplan_polygons WHERE isys_floorplan_polygons__id = ' . $this->convert_sql_id($polygonId) . ' AND isys_floorplan_polygons__deletable = 1;';

        return $this->update($sql) && $this->apply_update();
    }
}