<?php

namespace idoit\Module\Floorplan\View;

use Exception;
use idoit\Model\Dao\Base as DaoBase;
use idoit\Module\Floorplan\Model\Floorplan;
use idoit\Module\Floorplan\Model\Format;
use idoit\Module\Floorplan\Model\Profile;
use idoit\View\Base as ViewBase;
use isys_application as App;
use isys_auth as Auth;
use isys_component_template as Template;
use isys_format_json as JSON;
use isys_module as Module;
use isys_module_floorplan as ModuleFloorplan;

/**
 * i-doit cmdb controller
 *
 * @package     Modules
 * @subpackage  Floorplan
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class Visualization extends ViewBase
{
    private const SCALE_FACTORS = [
        'mm' => 1,
        'cm' => 10,
        'in' => 25.4,
        'ft' => 304.8,
        'm'  => 1000
    ];

    /**
     * @param   Module   $module
     * @param   Template $template
     * @param   DaoBase  $model
     *
     * @return  $this
     * @throws  Exception
     */
    public function process(Module $module, Template $template, DaoBase $model)
    {
        $auth = ModuleFloorplan::getAuth();
        $language = App::instance()->container->get('language');

        // Check if the user is allowed to view the floorplan.
        if (!$auth->is_allowed_to(Auth::VIEW, 'visualization')) {
            $this->paths['contentbottomcontent'] = ModuleFloorplan::getPath() . 'templates/auth-error.tpl';

            $template->assign('errorMessage', $language->get('LC__MODULE__FLOORPLAN__AUTH__MISSING_VISUALIZATION_RIGHT', $language->get('LC__AUTH__RIGHT_VIEW')));

            return $this;
        }

        // First order: retrieve the current floorplan ID.
        $floorplanObjectId = $this->request->get('id', 0);

        // Set paths to templates.
        $this->paths['leftcontent'] = ModuleFloorplan::getPath() . 'templates/visualizationLeft.tpl';
        $this->paths['contenttop'] = ModuleFloorplan::getPath() . 'templates/visualizationTop.tpl';
        $this->paths['contentbottomcontent'] = ModuleFloorplan::getPath() . 'templates/visualization.tpl';

        $floorplanProfiles = [];
        $objectTypes = [];

        $objectTypeRes = $model->getObjectTypes();

        if (count($objectTypeRes)) {
            while ($row = $objectTypeRes->get_row()) {
                $objectTypes[$row['isys_obj_type__id']] = [
                    'icon'  => Format::objectTypeIconPath($row['isys_obj_type__icon']),
                    'title' => $language->get($row['isys_obj_type__title']),
                    'color' => '#' . $row['isys_obj_type__color']
                ];
            }

            // Sorting by title, keeping the keys.
            uasort($objectTypes, function ($a, $b) {
                return strcasecmp($a['title'], $b['title']);
            });
        }

        // Load all available profiles.
        $profileResult = Profile::instance($model->get_database_component())->getProfiles();

        while ($profileRow = $profileResult->get_row()) {
            $profileRow['id'] = (int)$profileRow['id'];
            $profileRow['config'] = JSON::decode($profileRow['config']);

            unset($profileRow['status'], $profileRow['const']);

            $floorplanProfiles[] = $profileRow;
        }

        // Prepare rules
        $leftSmartyRules = [
            'floorplan_object_formfactor__width'       => [
                'p_bInfoIconSpacer' => 0,
                'p_strClass'        => 'only-numeric no-enter',
                'p_strStyle'        => 'width:50%;',
                'p_strPlaceholder'  => 'LC__CMDB__CATG__FORMFACTOR_INSTALLATION_WIDTH',
                'disableInputGroup' => true,
            ],
            'floorplan_object_formfactor__height'      => [
                'p_bInfoIconSpacer' => 0,
                'p_strClass'        => 'only-numeric no-enter',
                'p_strStyle'        => 'width:50%;',
                'p_strPlaceholder'  => 'LC__CMDB__CATG__FORMFACTOR_INSTALLATION_HEIGHT',
                'disableInputGroup' => true
            ],
            'floorplan_object_formfactor__depth'       => [
                'p_bInfoIconSpacer' => 0,
                'p_strClass'        => 'only-numeric no-enter',
                'p_strStyle'        => 'width:50%;',
                'p_strPlaceholder'  => 'LC__CMDB__CATG__FORMFACTOR_INSTALLATION_DEPTH',
                'disableInputGroup' => true
            ],
            'floorplan_object_formfactor__unit'        => [
                'p_bInfoIconSpacer' => 0,
                'p_strStyle'        => 'width:50%;',
                'p_strTable'        => 'isys_depth_unit',
                'p_bDbFieldNN'      => true,
                'disableInputGroup' => true
            ],
            'floorplan_object_formfactor__weight'      => [
                'p_bInfoIconSpacer' => 0,
                'p_strClass'        => 'only-numeric no-enter',
                'p_strPlaceholder'  => 'LC__CMDB__CATG__FORMFACTOR_INSTALLATION_WEIGHT',
                'disableInputGroup' => true
            ],
            'floorplan_object_formfactor__weight_unit' => [
                'p_bInfoIconSpacer' => 0,
                'p_strTable'        => 'isys_weight_unit',
                'p_bDbFieldNN'      => true,
                'disableInputGroup' => true
            ],
            'floorplan_object_orientation'             => [
                'p_bInfoIconSpacer' => 0,
                'p_strClass'        => 'input-mini fr hide',
                'p_bDisabled'       => true,
                'p_arData'          => [
                    'n' => 'LC__MODULE__FLOORPLAN__UP',
                    'e' => 'LC__MODULE__FLOORPLAN__RIGHT',
                    's' => 'LC__MODULE__FLOORPLAN__DOWN',
                    'w' => 'LC__MODULE__FLOORPLAN__LEFT'
                ]
            ]
        ];

        $topSmartyRules = [
            'floorplanGrid'    => [
                'p_strClass'        => 'input-block hide',
                'disableInputGroup' => true,
                'p_bSort'           => false,
                'p_bInfoIconSpacer' => 0,
                'p_bDbFieldNN'      => 1,
                'p_strSelectedID'   => 'm',
                'p_arData'          => [
                    $language->get('LC__MODULE__FLOORPLAN__GRID') => [
                        'cm' => 'LC__DEPTH_UNIT__CM',
                        'in' => 'LC__DEPTH_UNIT__INCH',
                        'ft' => 'LC__DEPTH_UNIT__FOOT',
                        'm'  => 'LC__DEPTH_UNIT__METER',
                    ]
                ]
            ],
            'floorplanProfile' => [
                'p_strClass' => 'input-small'
            ],
            'floorplan-layout-editor' => [
                'disableInputGroup' => true
            ]
        ];

        $contentSmartyRules = [
            'floorplan_measurement_display' => [
                'p_strClass'        => 'input-block',
                'disableInputGroup' => true,
                'p_bInfoIconSpacer' => 0,
                'p_bDbFieldNN'      => 1,
                'p_strSelectedID'   => 'nm',
                'p_arData'          => [
                    'nw' => $language->get('LC__MODULE__FLOORPLAN__TOP_LEFT'),
                    'ne' => $language->get('LC__MODULE__FLOORPLAN__TOP_RIGHT'),
                    'sw' => $language->get('LC__MODULE__FLOORPLAN__BOTTOM_LEFT'),
                    'se' => $language->get('LC__MODULE__FLOORPLAN__BOTTOM_RIGHT'),
                ]
            ],
            'floorplan_measurement_count'   => [
                'p_strClass'        => 'no-enter only-numeric input-mini',
                'p_bInfoIconSpacer' => 0
            ],
            'floorplan_measurement_unit'    => [
                'disableInputGroup' => true,
                'p_strStyle'   => 'width:70px',
                'p_bDbFieldNN' => true,
                'p_bSort'      => false,
                'p_arData'     => [
                    (string) self::SCALE_FACTORS['mm'] => $language->get('LC__DEPTH_UNIT__MM'),
                    (string) self::SCALE_FACTORS['cm'] => $language->get('LC__DEPTH_UNIT__CM'),
                    (string) self::SCALE_FACTORS['in'] => $language->get('LC__DEPTH_UNIT__INCH'),
                    (string) self::SCALE_FACTORS['ft'] => $language->get('LC__DEPTH_UNIT__FOOT'),
                    (string) self::SCALE_FACTORS['m'] => $language->get('LC__DEPTH_UNIT__METER'),
                ]
            ]
        ];

        try {
            $floorplanVersion = JSON::decode(file_get_contents(ModuleFloorplan::getPath() . 'package.json'))['version'];
        } catch (Exception $e) {
            $floorplanVersion = '1.2';
        }

        $this->prepareLatestFloorplans();

        $template
            ->activate_editmode()
            ->assign('assets_dir', ModuleFloorplan::getPath() . 'assets/')
            ->assign('authDelete', (int)$auth->is_allowed_to(Auth::DELETE, 'visualization'))
            ->assign('authDeleteErrorMessage', $language->get('LC__MODULE__FLOORPLAN__AUTH__MISSING_VISUALIZATION_RIGHT', $language->get('LC__AUTH__RIGHT_DELETE')))
            ->assign('authEdit', (int)$auth->is_allowed_to(Auth::EDIT, 'visualization'))
            ->assign('authEditErrorMessage', $language->get('LC__MODULE__FLOORPLAN__AUTH__MISSING_VISUALIZATION_RIGHT', $language->get('LC__AUTH__RIGHT_EDIT')))
            ->assign('editRight', $auth->is_allowed_to(Auth::EDIT, 'visualization'))
            ->assign('encType', 'multipart/form-data')
            ->assign('floorplanVersion', $floorplanVersion)
            ->assign('objectId', $floorplanObjectId)
            ->assign('objectTypeData', $objectTypes)
            ->assign('profiles', $floorplanProfiles)
            ->assign('scaleFactors', self::SCALE_FACTORS)
            ->smarty_tom_add_rules('tom.left', $leftSmartyRules)
            ->smarty_tom_add_rules('tom.content.top', $topSmartyRules)
            ->smarty_tom_add_rules('tom.content.bottom.content', $contentSmartyRules);

        return $this;
    }

    /**
     * Method for preparing the latest floorplans list.
     *
     * @throws \isys_exception_database
     */
    private function prepareLatestFloorplans()
    {
        $template = App::instance()->container->get('template');
        $database = App::instance()->container->get('database');

        $latestCreated = [];
        $latestUpdated = [];

        $floorplan = new Floorplan($database);
        $format = new Format($database);

        $latestCreatedResult = $floorplan->getStartupFloorplans('isys_obj__id', 6);

        while ($latestCreatedRow = $latestCreatedResult->get_row()) {
            $latestCreated[] = $format->object((int)$latestCreatedRow['isys_obj__id'], true);
        }

        $latestUpdatedResult = $floorplan->getStartupFloorplans('isys_floorplan__updated', 6);

        while ($latestUpdatedRow = $latestUpdatedResult->get_row()) {
            $latestUpdated[] = $format->object((int)$latestUpdatedRow['isys_obj__id'], true);
        }

        $template
            ->assign('latestCreatedObjects', $latestCreated)
            ->assign('latestUpdatedObjects', $latestUpdated);
    }
}
