(function () {
    'use strict';
    
    // This is a convenience method to move an element to the (visual) top.
    d3.selection.prototype.moveToFront = function () {
        return this.each(function () {
            this.parentNode.appendChild(this);
        });
    };
    
    // @see  FP-29  Implement a 'UUID' generator
    window.fpUuid = function () {
        return String.fromCharCode(Math.floor(Math.random() * 26) + 97)
               + Math.random().toString(16).slice(2)
               + Date.now().toString(16).slice(4);
    };
    
    window.FloorplanHelper = {
        /**
         *
         * @param objectType
         * @returns {string}
         */
        getObjectType: function (objectType) {
            // Fixed bug RT#30250.
            if (window.floorplanData.objectTypeData.hasOwnProperty(objectType)) {
                return window.floorplanData.objectTypeData[objectType];
            }
    
            if (!window.floorplanData.undefinedObjectType.hasOwnProperty('ot' + objectType)) {
                window.floorplanData.undefinedObjectType['ot' + objectType] = true;
        
                // Return a default and notify the user.
                idoit.Notify.warning((idoit.Translate.get('LC__MODULE__FLOORPLAN__VISUALIZATION__MISSING_OBJECT_TYPE_DESCRIPTION').replace(':id', objectType)), {life: 7.5});
            }
    
            return {
                color: '#FFFFFF',
                icon:  window.dir_images + 'axialis/documents-folders/document-color-grey.svg',
                title: idoit.Translate.get('LC__MODULE__FLOORPLAN__VISUALIZATION__MISSING_OBJECT_TYPE') + ' (#' + objectType + ')'
            };
        },
        
        /**
         *
         * @param p1
         * @param p2
         * @returns {number}
         */
        angleBetweenPoints: function (p1, p2) {
            if (p1[0] == p2[0] && p1[1] == p2[1]) {
                return Math.PI / 2;
            } else {
                return Math.atan2(p2[1] - p1[1], p2[0] - p1[0]);
            }
        },
        
        /**
         *
         * @param rad
         * @returns {number}
         */
        toDegrees: function (rad) {
            var degrees = (rad * (180 / Math.PI)) % 360;
        
            if (degrees < 0) {
                degrees += 360;
            }
        
            return degrees;
        },
    
        /**
         *
         * @param degrees
         * @returns {number}
         */
        fromDegrees: function (degrees) {
            return degrees / (180 / Math.PI);
        },
        
        /**
         * Fast method for calculating if a point lies inside a rectangle.
         *
         * @see  https://www.geeksforgeeks.org/check-whether-given-point-lies-inside-rectangle-not/
         * @param x1
         * @param y1
         * @param x2
         * @param y2
         * @param x3
         * @param y3
         * @param x4
         * @param y4
         * @param x
         * @param y
         * @returns {boolean}
         */
        checkPointInRectangle: function (x1, y1, x2, y2, x3, y3, x4, y4, x, y) {
            // Calculate area of rectangle ABCD
            var A  = area(x1, y1, x2, y2, x3, y3) + area(x1, y1, x4, y4, x3, y3),
                A1 = area(x, y, x1, y1, x2, y2),
                A2 = area(x, y, x2, y2, x3, y3),
                A3 = area(x, y, x3, y3, x4, y4),
                A4 = area(x, y, x1, y1, x4, y4);
            
            // Check if sum of A1, A2, A3 and A4  is same as A
            return (A === A1 + A2 + A3 + A4);
            
            function area(x1, y1, x2, y2, x3, y3) {
                return Math.abs((x1 * (y2 - y3) + x2 * (y3 - y1) + x3 * (y1 - y2)) / 2.0);
            }
        },
    
        /**
         * Returns information about the closest point between the given path and point.
         *
         * @param pathNode {SVGPathElement}
         * @param point
         * @returns {{x: *, y: *, pathLength: *, distance: number}}
         */
        closestPointOnPath: function (pathNode, point) {
            var pathLength   = pathNode.getTotalLength(),
                precision    = 8,
                best,
                bestLength,
                bestDistance = Infinity;
            
            // linear scan for coarse approximation
            for (var scan, scanLength = 0, scanDistance; scanLength <= pathLength; scanLength += precision)
            {
                if ((scanDistance = distance2(scan = pathNode.getPointAtLength(scanLength))) < bestDistance)
                {
                    best = scan;
                    bestLength = scanLength;
                    bestDistance = scanDistance;
                }
            }
            
            // binary search for precise estimate
            precision /= 2;
            while (precision > 0.5)
            {
                var before,
                    after,
                    beforeLength,
                    afterLength,
                    beforeDistance,
                    afterDistance;
                if ((beforeLength = bestLength - precision) >= 0 && (beforeDistance = distance2(before = pathNode.getPointAtLength(beforeLength))) < bestDistance)
                {
                    best = before;
                    bestLength = beforeLength;
                    bestDistance = beforeDistance;
                }
                else if ((afterLength = bestLength + precision) <= pathLength && (afterDistance = distance2(after = pathNode.getPointAtLength(afterLength))) < bestDistance)
                {
                    best = after;
                    bestLength = afterLength;
                    bestDistance = afterDistance;
                }
                else
                {
                    precision /= 2;
                }
            }
            
            return {
                x:          best.x,
                y:          best.y,
                pathLength: bestLength,
                distance:   Math.sqrt(bestDistance)
            };
            
            function distance2(p) {
                var dx = p.x - point[0],
                    dy = p.y - point[1];
                return dx * dx + dy * dy;
            }
        },
    
        /**
         * Calculates the distance between to given points.
         *
         * @param p1
         * @param p2
         * @returns {number}
         */
        distanceBetweenPoints: function (p1, p2) {
            return Math.sqrt(Math.pow(p2[1] - p1[1], 2) + Math.pow(p2[0] - p1[0], 2));
        },
    
        /**
         * This function will create a element which contains a 1:1 copy of the canvas' SVG.
         * Only difference is, that this element will be sized to display its content in full
         * and update the `transform="translate(...)"` values accordingly.
         *
         * @returns {*}
         */
        prepareFullSvg: function ($canvas) {
            var $nodes = $canvas.select('g.exportable'),
                $printView  = new Element('div'),
                match,
                i,
                boundingBox = {width: 200, height: 200},
                minX        = 0,
                minY        = 0,
                maxX        = 0,
                maxY        = 0;

            for (i in $nodes) {
                if (!$nodes.hasOwnProperty(i)) {
                    continue;
                }
                
                if ($nodes[i]) {
                    try {
                        boundingBox = $nodes[i].getBBox();
                    } catch (e) {
                        try {
                            // Using `getBoundingClientRect` as fallback (for example for IE 11).
                            boundingBox = $nodes[i].getBoundingClientRect();
                        } catch (e) {
                            // Do nothing, use the defaults.
                        }
                    }
                }

                try {
                    
                    // Get minimum X and Y coordinates.
                    minX = Math.min(minX, boundingBox.x);
                    minY = Math.min(minY, boundingBox.y);
                    
                    // Get maximum X and Y coordinates.
                    maxX = Math.max(maxX, (boundingBox.x+boundingBox.width));
                    maxY = Math.max(maxY, (boundingBox.y+boundingBox.height));
                } catch (e) {
                    // Do nothing
                }
            }
            
            // Add the node width and height to the dimensions (will also create a nice margin).
            minX = Math.abs(minX) * 1.01;
            minY = Math.abs(minY) * 1.01;
            maxX = Math.abs(maxX) * 1.01;
            maxY = Math.abs(maxY) * 1.01;
            
            // Pass the element to D3 since it can handle SVG elements.
            $printView = d3.select($printView.update($canvas.down('svg').outerHTML));
            $printView.select('#grid').remove();
            $printView.select('#floorplan-scale').remove();
            
            // Select the SVG, alter some attributes (+ some attributes of the direct <g> descendants) and return the node.
            return $printView.select('svg')
                .attr('viewBox', '0 0 ' + (minX + maxX) + ' ' + (minY + maxY))
                .attr('width', (minX + maxX))
                .attr('height', (minY + maxY))
                .call(function (svg) {
                    svg.selectAll('svg > g').attr('transform', 'translate(' + minX + ',' + minY + ') scale(1)');
                })
                .node();
        },
        
        processNumber(str) {
            // Ensure we got a string.
            str = ("" + str).trim();

            if (str.blank()) {
                return 0;
            }
            
            const isNegative = (str.substr(0,1) === '-');
            
            str = str.replace(/[^0-9\.,]/ig, '');
            
            // We found >1 period (as in "1.000.000") - remove them.
            if((str.match(/\./g) || []).length > 1) {
                str = str.replace(/\./g, '');
            }
    
            // We found >1 comma (as in "1,000,000") - remove them.
            if((str.match(/,/g) || []).length > 1) {
                str = str.replace(/,/g, '');
            }
            
            // We could have one period or comma, we only keep the one that is to the right (since this should be the decimal separator).
            if(str.indexOf('.') > str.indexOf(',')) {
                str = str.replace(/,/g, '');
            } else {
                str = str.replace(/\./g, '').replace(/,/g, '.');
            }
            
            return parseFloat(isNegative ? -str : str);
        },
    
        /**
         *
         * @see https://stackoverflow.com/a/18473154
         * @param centerX
         * @param centerY
         * @param radius
         * @param angleInDegrees
         * @returns {{x: *, y: *}}
         */
        polarToCartesian: function(centerX, centerY, radius, angleInDegrees) {
            const angleInRadians = (angleInDegrees-90) * Math.PI / 180.0;
            
            return {
                x: centerX + (radius * Math.cos(angleInRadians)),
                y: centerY + (radius * Math.sin(angleInRadians))
            };
        },
    
        /**
         *
         * @see https://stackoverflow.com/a/18473154
         * @param x
         * @param y
         * @param radius
         * @param startAngle
         * @param endAngle
         * @returns {string}
         */
        describeArc: function(x, y, radius, startAngle, endAngle){
            const start = this.polarToCartesian(x, y, radius, endAngle);
            const end = this.polarToCartesian(x, y, radius, startAngle);
            const largeArcFlag = endAngle - startAngle <= 180 ? "0" : "1";
            
            return [
                "M", start.x, start.y,
                "A", radius, radius, 0, largeArcFlag, 0, end.x, end.y
            ].join(" ");
        },
        
        processTransformString: function(transform) {
            let result = {x: 0, y: 0, k: 1, r:0};
            
            if (!transform || !Object.isString(transform)) {
                return result;
            }
            
            const foundTranslation = transform.match(/translate\((.*?)\)/i);
            
            if (foundTranslation) {
                const translate = foundTranslation[1].trim().split(/[, ]/);
                
                result.x = translate[0].trim();
                result.y = translate[1].trim();
            }
    
            const foundScale = transform.match(/scale\((.*?)\)/i);
    
            if (foundScale) {
                result.k = foundScale[1].trim();
            }
    
            const foundRotation = transform.match(/rotate\((.*?)\)/i);
    
            if (foundRotation) {
                result.r = foundRotation[1].trim();
            }
            
            return result;
        },
        
        processOrientation: function(d) {
            if (!d.hasOwnProperty('orientation') || d.orientation === null) {
                return 'M0,0';
            }
            
            const line = d3.line()
                .x(function (d) { return d[0]; })
                .y(function (d) { return d[1]; })
                .curve(d3.curveLinear);
    
            // We'll remove and add 16 to prevent the arrow from slipping under the object type title.
            switch (d.orientation) {
                default:
                case 'n':
                    return line([[-8, -(d.height / 2)], [8, -(d.height / 2)], [0, -(d.height / 2 + 8)]]);
        
                case 'e':
                    return line([[(d.width / 2), -8], [(d.width / 2), 8], [(d.width / 2 + 8), 0]]);
        
                case 's':
                    return line([[-8, (d.height / 2)], [8, (d.height / 2)], [0, (d.height / 2 + 8)]]);
        
                case 'w':
                    return line([[-(d.width / 2), -8], [-(d.width / 2), 8], [-(d.width / 2 + 8), 0]]);
            }
        }
    };
})();
