<?php

use idoit\Module\Floorplan\Model\Format;
use idoit\Module\Floorplan\Model\PolygonDao;

/**
 * i-doit
 *
 * Floorplan polygon editor popup.
 *
 * @package     Modules
 * @subpackage  Floorplan
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class isys_popup_floorplan_polygon_editor extends isys_component_popup
{
    /**
     * @var  isys_component_template
     */
    protected $template;

    /**
     * @var  isys_component_database
     */
    protected $database;

    /**
     * @var  isys_component_template_language_manager
     */
    protected $language;

    /**
     * Constructor method.
     *
     * @throws Exception
     */
    public function __construct()
    {
        parent::__construct();

        $this->database = isys_application::instance()->container->get('database');
        $this->template = isys_application::instance()->container->get('template');
        $this->language = isys_application::instance()->container->get('language');
    }

    /**
     * Handles SMARTY request for dialog plus lists and builds the list base on the specified table.
     *
     * @param   isys_component_template &$template
     * @param   array                   $parameters
     *
     * @return  string
     */
    public function handle_smarty_include(isys_component_template &$template, $parameters)
    {
        return '<button id="' . $parameters['name'] . '" type="button" onclick="' . $this->process_overlay('', 700, 500, $parameters) . '" class="btn mt5 mr5 hide" title="' . $this->language->get('LC__MODULE__FLOORPLAN__POPUP_POLYGON__BUTTON') . '" data-tooltip="1">' .
            '<img src="' . isys_application::instance()->www_path . 'images/axialis/cad/shape-points-edit.svg" alt="" />' .
            '</button>';
    }

    /**
     * Handle specific ajax requests.
     *
     * @param   isys_module_request $request
     *
     * @return  mixed
     * @author  Leonard Fischer <lfischer@i-doit.org>
     */
    public function &handle_ajax_request(isys_module_request $request)
    {
        header('Content-Type: application/json');

        $response = [
            'success' => true,
            'data'    => null,
            'message' => null
        ];

        try {
            switch ($_POST['action']) {
                case 'savePolygon':
                    $response['data'] = $this->save_polygon($_POST['name'], $_POST['data']);
                    break;

                case 'removePolygon':
                    $response['data'] = $this->remove_polygon($_POST['polygonId']);
                    break;
            }
        } catch (Exception $e) {
            $response['success'] = false;
            $response['message'] = $e->getMessage();
        }

        echo \isys_format_json::encode($response);
        die;
    }

    /**
     * Method for saving a given polygon to the database.
     *
     * @param   string $name
     * @param   string $data
     *
     * @return  array
     * @throws  isys_exception_dao
     * @throws  isys_exception_database
     */
    protected function save_polygon($name, $data)
    {
        $polygonDao = new PolygonDao($this->database);

        $polygons = [];
        $polygonId = $polygonDao->savePolygon(null, [
            'isys_floorplan_polygons__title'     => $name,
            'isys_floorplan_polygons__points'    => $data,
            'isys_floorplan_polygons__deletable' => 1
        ]);

        $polygonResult = $polygonDao->getData();

        while ($polygonRow = $polygonResult->get_row()) {
            try {
                $polygonRow['title'] = $this->language->get($polygonRow['title']);
                $polygonRow['points'] = \isys_format_json::decode($polygonRow['points']);
                $polygonRow['deletable'] = (bool)$polygonRow['deletable'];

                $polygons[$polygonRow['id']] = $polygonRow;
            } catch (Exception $e) {
                isys_notify::error('Your polygon "' . $polygonRow['title'] . '" is defect, possibly the point-data is corrupted.');
            }
        }

        return [
            'newId'    => $polygonId,
            'polygons' => $polygons
        ];
    }

    /**
     * Method for deleting a previously saved polygon. Only user-defined polygons can be removed!
     *
     * @param  integer $polygonId
     *
     * @return bool|string
     * @throws isys_exception_dao
     * @throws isys_exception_database
     */
    protected function remove_polygon($polygonId)
    {
        $polygon = new PolygonDao($this->database);

        if ($polygon->isDeletable($polygonId)) {
            return $polygon->deletePolygon($polygonId);
        }

        return $this->language->get('LC__MODULE__FLOORPLAN__POPUP_EDITOR__REMOVE_FAILURE');
    }

    /**
     * Method for handling the module request.
     *
     * @param   isys_module_request $request
     *
     * @return  null
     * @throws  isys_exception_database
     */
    public function &handle_module_request(isys_module_request $request)
    {
        $polygonDao = new PolygonDao($this->database);
        $objectId = (int)$_POST['selectedObjectId'];

        $polygons = [];
        $object = [];

        if ($objectId > 0) {
            $object = $polygonDao->getObjectData($objectId)->get_row();

            $object['objTypeTitle'] = $this->language->get($object['objTypeTitle']);
            $object['objTypeIcon'] = Format::objectTypeIconPath((int)$object['objTypeId']);
        }

        $polygonResult = $polygonDao->getData();

        while ($polygonRow = $polygonResult->get_row()) {
            try {
                $polygonRow['title'] = $this->language->get($polygonRow['title']);
                $polygonRow['points'] = \isys_format_json::decode($polygonRow['points']);
                $polygonRow['deletable'] = !!$polygonRow['deletable'];

                $polygons[$polygonRow['id']] = $polygonRow;
            } catch (Exception $e) {
                isys_notify::error('Your polygon "' . $polygonRow['title'] . '" is defect, possibly the point-data is corrupted.');
            }
        }

        $rules = [
            'floorplan-polygon-popup-selection'         => [
                'p_arData'          => [],
                'p_bSort'           => false,
                'p_bInfoIconSpacer' => 0,
                'p_bDbFieldNN'      => true,
                'p_strSelectedID'   => 'rectangle',
                'disableInputGroup' => true
            ],
            'floorplan-polygon-popup-selfdefined-title' => [
                'p_strPlaceholder'  => 'LC__MODULE__FLOORPLAN__POPUP_POLYGON__FORM_NAME',
                'disableInputGroup' => true
            ]
        ];

        $ajaxUrl = isys_helper_link::create_url([
            C__CMDB__GET__POPUP  => 'floorplan_polygon_editor',
            C__GET__AJAX_REQUEST => 'handle_ajax_request',
        ], true);

        $this->template->activate_editmode()
            ->assign('ajax_url', $ajaxUrl)
            ->assign('polygons', \isys_format_json::encode($polygons))
            ->assign('object', $object)
            ->smarty_tom_add_rules('tom.popup.floorplan', $rules)
            ->display(isys_module_floorplan::getPath() . 'templates/popup/polygon-editor.tpl');
        die;
    }
}
